"""
NEXUS-ATC – Pilot-Client (Schritt 3 + 5 + 6)
=================================================
Liest die eigene Position aus dem MSFS per SimConnect, sendet sie an den
Server UND empfängt die Positionen anderer Piloten.  Fremde Flugzeuge
werden als AI-Objekte in den MSFS injiziert.

Enthält ein kleines PyQt6-Fenster zur Eingabe von Callsign und Flugplan
(DEP, ARR, Flugzeugtyp).

Nutzung:
    python pilot_client.py
"""

from __future__ import annotations

import asyncio
import json
import pathlib
import sys
import math
import threading
import time
from typing import Dict, Optional

from PyQt6.QtCore import Qt, QTimer, pyqtSignal, QObject, QSettings
from PyQt6.QtGui import QFont, QPainter, QColor, QBrush, QPen
from PyQt6.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QLineEdit, QPushButton, QGroupBox, QTextEdit,
    QFormLayout, QMessageBox, QCheckBox, QScrollArea, QFrame,
    QDialog,
)

import websockets
from SimConnect import SimConnect, AircraftRequests, AircraftEvents
from model_matcher import (
    ModelMatcher, MissingModelLogger, PASSIVE_TITLES, AI_MODEL_CANDIDATES,
    AI_MODEL_DEFAULT, validate_model_path,
)
# Alias für Abwärtskompatibilität (intern wird PASSIVE_TITLES verwendet)
MODEL_MATCH = PASSIVE_TITLES


def _haversine_nm(lat1: float, lon1: float, lat2: float, lon2: float) -> float:
    """Haversine-Distanz in Nautischen Meilen."""
    R = 3440.065  # Erdradius in NM
    dlat = math.radians(lat2 - lat1)
    dlon = math.radians(lon2 - lon1)
    a = (math.sin(dlat / 2) ** 2 +
         math.cos(math.radians(lat1)) * math.cos(math.radians(lat2)) *
         math.sin(dlon / 2) ** 2)
    return R * 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))
from SimConnect.Enum import (
    SIMCONNECT_DATA_INITPOSITION,
    SIMCONNECT_DATATYPE,
)
from ctypes import c_float, c_ulong, pointer, sizeof
import ctypes.util

# ── Opus-DLL laden (MUSS vor pymumble-Import stehen) ──────────────────
def _ensure_opus_dll() -> bool:
    """Versucht die Opus-DLL zu laden, damit pymumble/opuslib funktioniert."""
    if ctypes.util.find_library("opus"):
        return True
    import os
    here = os.path.dirname(os.path.abspath(__file__))
    for sub in ("", "data", "lib"):
        p = os.path.join(here, sub, "opus.dll")
        if os.path.isfile(p):
            try:
                ctypes.cdll.LoadLibrary(p)
                return True
            except OSError:
                pass
    try:
        import pyogg
        dll = os.path.join(os.path.dirname(pyogg.__file__), "opus.dll")
        if os.path.isfile(dll):
            ctypes.cdll.LoadLibrary(dll)
            return True
    except (ImportError, OSError):
        pass
    return False

_OPUS_OK = _ensure_opus_dll()
if not _OPUS_OK:
    print("⚠  opus.dll nicht gefunden – Voice deaktiviert")
    print("   → Kopiere opus.dll nach:", os.path.dirname(os.path.abspath(__file__)))
    print("   → Download: https://github.com/nicedayzhu/opus_dll_for_python/releases")
# ─────────────────────────────────────────────────────────────────────────

# Optionale Voice-Abhängigkeiten
# --- Python 3.12+ Fix: ssl.wrap_socket wurde entfernt -----------------
import ssl
if not hasattr(ssl, 'wrap_socket'):
    def _wrap_socket_shim(sock, certfile=None, keyfile=None, ssl_version=None,
                          **kw):
        ctx = ssl.SSLContext(ssl.PROTOCOL_TLS_CLIENT)
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE
        if certfile:
            ctx.load_cert_chain(certfile, keyfile)
        return ctx.wrap_socket(sock, **kw)
    ssl.wrap_socket = _wrap_socket_shim
if not hasattr(ssl, 'PROTOCOL_TLSv1'):
    ssl.PROTOCOL_TLSv1 = ssl.PROTOCOL_TLS_CLIENT
if not hasattr(ssl, 'PROTOCOL_TLS'):
    ssl.PROTOCOL_TLS = ssl.PROTOCOL_TLS_CLIENT
# -----------------------------------------------------------------------
try:
    import pymumble_py3 as pymumble
    _HAS_PYMUMBLE = True
except Exception:
    pymumble = None
    _HAS_PYMUMBLE = False

try:
    from pynput import keyboard as pynput_keyboard
    _HAS_PYNPUT = True
except ImportError:
    pynput_keyboard = None
    _HAS_PYNPUT = False

try:
    import sounddevice as sd
    import numpy as np
    _HAS_AUDIO = True
except ImportError:
    sd = None
    np = None
    _HAS_AUDIO = False

# Joystick-Support (optional)
try:
    import pygame
    pygame.init()
    pygame.joystick.init()
    _HAS_PYGAME = True
except ImportError:
    pygame = None
    _HAS_PYGAME = False

# ── Voice-Config (Datei) ──────────────────────────────────────────────
import os as _os
_VOICE_CONFIG_PATH = _os.path.join(
    _os.path.dirname(_os.path.abspath(__file__)), "voice_config.json")

def _load_voice_config() -> dict:
    """Liest voice_config.json."""
    try:
        with open(_VOICE_CONFIG_PATH, "r", encoding="utf-8") as f:
            return json.load(f)
    except (FileNotFoundError, json.JSONDecodeError):
        return {}

def _save_voice_config(cfg: dict):
    """Schreibt voice_config.json."""
    try:
        existing = _load_voice_config()
        existing.update(cfg)
        with open(_VOICE_CONFIG_PATH, "w", encoding="utf-8") as f:
            json.dump(existing, f, indent=2, ensure_ascii=False)
    except Exception:
        pass


# ---------------------------------------------------------------------------
# Konfiguration
# ---------------------------------------------------------------------------
DEFAULT_SERVER = "ws://localhost:9000/ws/pilot"
DEFAULT_CALLSIGN = "PILOT1"
INTERVAL = 0.1  # Sekunden (10 Hz)

# Failover-Konfiguration
FAILOVER_MAX_RETRIES = 5          # Max. Reconnect-Versuche pro Server
FAILOVER_RETRY_DELAY_S = 3.0      # Wartezeit zwischen Versuchen
FAILOVER_HEARTBEAT_INTERVAL = 10  # Heartbeat-Prüfung alle N Sekunden


def probe_ai_model(sc: SimConnect) -> str:
    """Testet AI_MODEL_CANDIDATES und gibt den ersten funktionierenden zurück."""
    import os as _os
    for title in AI_MODEL_CANDIDATES:
        rq = sc.new_request_id()
        init = SIMCONNECT_DATA_INITPOSITION()
        init.Latitude  = 47.50
        init.Longitude = 8.55
        init.Altitude  = 5000.0
        init.Pitch     = 0.0
        init.Bank      = 0.0
        init.Heading   = 143.0
        init.OnGround  = 0
        init.Airspeed  = 200
        _os.environ.pop("SIMCONNECT_OBJECT_ID", None)
        sc.dll.AICreateNonATCAircraft(
            sc.hSimConnect,
            title.encode("utf-8"),
            b"MDLTEST",
            init,
            rq.value,
        )
        for _ in range(15):
            time.sleep(0.1)
            oid = _os.environ.get("SIMCONNECT_OBJECT_ID")
            if oid is not None:
                rq2 = sc.new_request_id()
                sc.dll.AIRemoveObject(sc.hSimConnect, int(oid), rq2.value)
                time.sleep(0.15)
                return title
    return "340 Passenger"


# ---------------------------------------------------------------------------
# AI-Flugzeug-Verwaltung mit Lerp-Interpolation (Schritt 5)
# ---------------------------------------------------------------------------
class AITarget:
    """Speichert Ist- und Ziel-Position für Interpolation."""

    def __init__(self, callsign: str, lat: float, lon: float, alt: float):
        self.callsign = callsign
        # Aktuelle (interpolierte) Position
        self.cur_lat = lat
        self.cur_lon = lon
        self.cur_alt = alt
        # Ziel-Position (letzte empfangene)
        self.tgt_lat = lat
        self.tgt_lon = lon
        self.tgt_alt = alt
        # Interpolations-Fortschritt (0.0 → 1.0)
        self.t = 1.0
        self.object_id: int = -1  # SimConnect-Objekt-ID
        self.heading: float = 0.0
        self.groundspeed: float = 0.0
        # Model-Matching-Info (für Panel-Anzeige)
        self.actype: str = ""
        self.airline: str = ""
        self.matched_model: str = ""
        self.is_fallback: bool = False

    def set_target(self, lat: float, lon: float, alt: float):
        """Neues Ziel setzen – Interpolation startet bei aktueller Position."""
        self.cur_lat, self.cur_lon, self.cur_alt = (
            self.lerp_lat(), self.lerp_lon(), self.lerp_alt()
        )
        self.tgt_lat = lat
        self.tgt_lon = lon
        self.tgt_alt = alt
        self.t = 0.0

    def advance(self, dt: float, duration: float = 0.5):
        """Fortschritt um dt Sekunden vorantreiben."""
        if duration > 0:
            self.t = min(1.0, self.t + dt / duration)

    def lerp_lat(self) -> float:
        return self.cur_lat + (self.tgt_lat - self.cur_lat) * self.t

    def lerp_lon(self) -> float:
        return self.cur_lon + (self.tgt_lon - self.cur_lon) * self.t

    def lerp_alt(self) -> float:
        return self.cur_alt + (self.tgt_alt - self.cur_alt) * self.t


SIMCONNECT_UNUSED = c_ulong(0xFFFFFFFF)


class AIManager:
    """Verwaltet AI-Objekte für andere Piloten im MSFS mit Interpolation.
    Verwendet MSFS 2024 Passive-Aircraft-Titel für die AI-Anzeige."""

    LERP_INTERVAL = 0.05   # 50 ms → 20 Hz Update
    LERP_DURATION = 0.5    # Dauer einer Interpolation (= Server-Sendeintervall)

    def __init__(self, sm: SimConnect, bridge: SignalBridge, default_model: str,
                 model_matcher: Optional["ModelMatcher"] = None):
        self.sm = sm
        self.bridge = bridge
        self.default_model = default_model
        self.model_matcher: Optional[ModelMatcher] = model_matcher
        self.missing_logger = MissingModelLogger()
        self._targets: Dict[str, AITarget] = {}
        self._running = True

        # SimConnect DataDefinition für INITPOSITION registrieren
        self._def_id = sm.new_def_id()
        sm.dll.AddToDataDefinition(
            sm.hSimConnect, self._def_id.value,
            b"Initial Position", b"",
            SIMCONNECT_DATATYPE.SIMCONNECT_DATATYPE_INITPOSITION,
            c_float(0), SIMCONNECT_UNUSED,
        )

        self._thread = threading.Thread(target=self._interpolation_loop, daemon=True)
        self._thread.start()

    def stop(self):
        """Alle AI-Objekte entfernen und Thread beenden."""
        self._running = False
        for cs, target in list(self._targets.items()):
            if target.object_id >= 0:
                try:
                    rq = self.sm.new_request_id()
                    self.sm.dll.AIRemoveObject(
                        self.sm.hSimConnect, target.object_id, rq.value)
                except Exception:
                    pass
        self._targets.clear()

    def update_or_create(self, callsign: str, lat: float, lon: float,
                         alt: float, hdg: float = 0.0, gs: float = 0.0,
                         model: str = "", actype: str = ""):
        """
        Setzt ein neues Ziel für einen AI-Piloten.
        Bei erstmaligem Empfang wird ein neues AI-Objekt in MSFS erstellt.
        Verwendet ModelMatcher für Typ+Airline-basierte Modellauswahl.
        """
        if callsign in self._targets:
            target = self._targets[callsign]
            target.set_target(lat, lon, alt)
            target.heading = hdg
            target.groundspeed = gs
        else:
            target = AITarget(callsign, lat, lon, alt)
            target.heading = hdg
            target.groundspeed = gs
            self._targets[callsign] = target

            # AI-Objekt in MSFS erstellen – Model Matching
            is_fallback = False
            airline = ""
            if self.model_matcher and actype:
                airline = ModelMatcher.extract_airline(callsign)
                use_model, is_fallback = self.model_matcher.match(actype, airline)
                # MSFS-Titel für SimConnect (Passive Aircraft)
                create_title, _ = self.model_matcher.resolve_title(
                    actype, airline, self.default_model)
            else:
                use_model = model or self.default_model
                create_title = PASSIVE_TITLES.get(
                    actype.upper(), self.default_model) if actype else self.default_model

            # Model-Matching-Info am Target speichern (für Panel-API)
            target.actype = actype
            target.airline = airline
            target.matched_model = use_model
            target.is_fallback = is_fallback

            # Detailliertes Logging der Modellwahl
            self.bridge.log_message.emit(
                f"  🛩  AI-Modell: {callsign} → Typ={actype} Airline={airline} "
                f"Modell={use_model} Titel={create_title} Fallback={is_fallback}"
            )

            # SimObjects-Pfad Validierung (prüfe ob Modell installiert ist)
            if use_model and use_model.startswith("SimObjects/"):
                import os as _validate_os
                _comm_candidates = [
                    pathlib.Path(_validate_os.environ.get("LOCALAPPDATA", "")) /
                        "Packages/Microsoft.Limitless_8wekyb3d8bbwe/"
                        "LocalCache/Packages/Community",
                    pathlib.Path(_validate_os.environ.get("APPDATA", "")) /
                        "Microsoft Flight Simulator 2024/Packages/Community",
                ]
                for _comm in _comm_candidates:
                    if _comm.exists():
                        if not validate_model_path(use_model, _comm):
                            self.bridge.log_message.emit(
                                f"  ⚠️ Modellpfad nicht gefunden: {_comm / use_model}"
                            )
                        break

            oid = self._create_ai_object(callsign, lat, lon, alt, hdg, create_title)
            if oid is not None:
                target.object_id = oid
                self.bridge.log_message.emit(
                    f"  ✅ AI erstellt: {callsign} OID={oid} → {create_title}"
                )
                if is_fallback:
                    # Missing-Model Logger
                    self.missing_logger.log(actype, airline, callsign)
                    warn = (f"⚠️ Keine passende Livery für "
                            f"{ModelMatcher.extract_airline(callsign)} "
                            f"{actype} gefunden – generisches Modell verwendet.")
                    self.bridge.log_message.emit(f"  {warn}")
                    self.bridge.model_warning.emit(warn)
            else:
                self.bridge.log_message.emit(
                    f"  ❌ AI-Erstellung fehlgeschlagen: {callsign} "
                    f"(Titel={create_title})"
                )

    def remove(self, callsign: str):
        """Entfernt ein AI-Flugzeug aus MSFS."""
        if callsign in self._targets:
            target = self._targets.pop(callsign)
            if target.object_id >= 0:
                try:
                    rq = self.sm.new_request_id()
                    self.sm.dll.AIRemoveObject(
                        self.sm.hSimConnect, target.object_id, rq.value)
                except Exception:
                    pass
            self.bridge.log_message.emit(f"  🛩  AI entfernt: {callsign}")

    def get_traffic_info(self) -> list[dict]:
        """Liefert Info über alle aktiven AI-Flugzeuge (für Panel-API)."""
        result = []
        for cs, t in self._targets.items():
            result.append({
                "callsign": cs,
                "actype": t.actype,
                "airline": t.airline,
                "model": t.matched_model,
                "fallback": t.is_fallback,
                "alt": round(t.lerp_alt()),
                "heading": round(t.heading),
                "groundspeed": round(t.groundspeed),
                "oid": t.object_id,
            })
        return result

    def _create_ai_object(self, callsign: str, lat: float, lon: float,
                          alt: float, hdg: float, model: str) -> Optional[int]:
        """Erstellt ein AI-Flugzeug in MSFS via SimConnect."""
        import os as _os
        rq = self.sm.new_request_id()
        init = SIMCONNECT_DATA_INITPOSITION()
        init.Latitude  = lat
        init.Longitude = lon
        init.Altitude  = alt
        init.Pitch     = -3.0
        init.Bank      = 0.0
        init.Heading   = hdg
        init.OnGround  = 0
        init.Airspeed  = 200
        _os.environ.pop("SIMCONNECT_OBJECT_ID", None)
        self.sm.dll.AICreateNonATCAircraft(
            self.sm.hSimConnect,
            model.encode("utf-8"),
            callsign.encode("utf-8"),
            init,
            rq.value,
        )
        for _ in range(30):
            time.sleep(0.1)
            oid = _os.environ.get("SIMCONNECT_OBJECT_ID")
            if oid is not None:
                return int(oid)
        return None

    def _move_ai_object(self, obj_id: int, lat: float, lon: float,
                        alt: float, hdg: float, spd: float):
        """Bewegt ein AI-Objekt an eine neue Position."""
        pos = SIMCONNECT_DATA_INITPOSITION()
        pos.Latitude  = lat
        pos.Longitude = lon
        pos.Altitude  = alt
        pos.Pitch     = -2.0 if alt > 500 else 0.0
        pos.Bank      = 0.0
        pos.Heading   = hdg
        pos.OnGround  = 1 if alt < 200 else 0
        pos.Airspeed  = spd
        self.sm.dll.SetDataOnSimObject(
            self.sm.hSimConnect,
            self._def_id.value,
            obj_id,
            0, 0,
            sizeof(SIMCONNECT_DATA_INITPOSITION),
            pointer(pos),
        )

    def _interpolation_loop(self):
        """
        Hintergrund-Thread: bewegt alle AI-Objekte in kleinen Schritten
        von der alten Position zur neuen (Lineare Interpolation / Lerp).
        """
        while self._running:
            dt = self.LERP_INTERVAL
            for target in list(self._targets.values()):
                if target.object_id < 0:
                    continue  # Noch nicht erstellt
                if target.t >= 1.0:
                    continue  # Ziel bereits erreicht

                target.advance(dt, self.LERP_DURATION)

                lat = target.lerp_lat()
                lon = target.lerp_lon()
                alt = target.lerp_alt()

                try:
                    self._move_ai_object(
                        target.object_id, lat, lon, alt,
                        target.heading, target.groundspeed,
                    )
                except Exception:
                    pass

            time.sleep(dt)


# ---------------------------------------------------------------------------
# Signal-Bridge (Thread → Qt)
# ---------------------------------------------------------------------------
class SignalBridge(QObject):
    """Brücke zwischen asyncio-Thread und Qt-UI."""
    log_message = pyqtSignal(str)
    connection_status = pyqtSignal(str)
    other_pilot = pyqtSignal(dict)   # Empfangene Daten anderer Piloten
    sim_status = pyqtSignal(bool)    # True = SimConnect verbunden
    server_status = pyqtSignal(bool) # True = Server verbunden
    ptt_status = pyqtSignal(bool)    # True = PTT aktiv
    atc_instruction = pyqtSignal(dict)   # ATC_UPDATE empfangen
    atc_list_update = pyqtSignal(list)   # Active ATC list
    atc_online = pyqtSignal(dict)        # Einzelner ATC online
    atc_offline = pyqtSignal(str)        # ATC-Station offline
    current_squawk = pyqtSignal(int)     # Aktueller Squawk aus MSFS
    current_com1 = pyqtSignal(float)      # Aktuelle COM1-Frequenz aus MSFS
    freq_message = pyqtSignal(dict)      # Frequenz-Nachricht empfangen
    voice_command = pyqtSignal(dict)     # Voice-Befehl vom Server (VOICE_MOVE etc.)
    cpdlc_message = pyqtSignal(dict)     # CPDLC-Nachricht vom ATC
    signal_quality = pyqtSignal(dict)    # Funkqualität vom Server
    radio_blocked = pyqtSignal(dict)     # Radio-Blocking (Heterodyne)
    stca_alert = pyqtSignal(dict)        # STCA-Konfliktwarnung
    slot_assigned = pyqtSignal(dict)     # Event Slot-Zuweisung / TSAT
    contact_request = pyqtSignal(dict)   # PILOT_CONTACT_REQUEST vom Server
    weather_sync = pyqtSignal(dict)       # Wetter-Sync Ergebnis (METAR gesetzt / Abweichung)
    simbrief_data = pyqtSignal(dict)      # SimBrief-Daten für UI-Update (Thread → Main)
    com_volume = pyqtSignal(int, int)     # COM1-Vol%, COM2-Vol% aus MSFS (0-100)
    model_warning = pyqtSignal(str)       # Fallback-Warnung wenn kein Modell/Livery passt


# ---------------------------------------------------------------------------
# Status-LED Widget
# ---------------------------------------------------------------------------
class StatusLED(QWidget):
    """Kleine farbige LED-Anzeige mit Label."""

    def __init__(self, label: str, parent=None):
        super().__init__(parent)
        self._color = QColor(80, 80, 80)
        self._label = label
        self.setFixedHeight(22)
        self.setMinimumWidth(110)

    def set_active(self, active: bool, color: QColor = None):
        self._color = (color or QColor(0, 180, 255)) if active else QColor(80, 80, 80)
        self.update()

    def paintEvent(self, event):
        p = QPainter(self)
        p.setRenderHint(QPainter.RenderHint.Antialiasing)
        # Glow-Effekt
        if self._color != QColor(80, 80, 80):
            glow = QColor(self._color)
            glow.setAlpha(40)
            p.setBrush(QBrush(glow))
            p.setPen(Qt.PenStyle.NoPen)
            p.drawEllipse(2, 3, 16, 16)
        p.setBrush(QBrush(self._color))
        p.setPen(QPen(self._color.darker(130), 1))
        p.drawEllipse(5, 5, 12, 12)
        p.setPen(QColor(200, 200, 210))
        p.setFont(QFont("Segoe UI", 9))
        p.drawText(22, 16, self._label)
        p.end()


# ---------------------------------------------------------------------------
# Headless Voice Manager (Mumble + PTT + VHF-Filter)
# ---------------------------------------------------------------------------

def _vhf_bandpass(audio_data: bytes, sample_rate: int = 48000,
                  low_hz: int = 300, high_hz: int = 3000,
                  static_mix: float = 0.0,
                  volume_factor: float = 1.0) -> bytes:
    """VHF radio bandpass filter (300 Hz – 3000 Hz) mit Signal-Degradation.

    Uses a basic FFT-based filter for authentic radio sound.
    static_mix: 0.0 (clean) bis 1.0 (nur Rauschen) – simuliert Funkstörungen
    volume_factor: Lautstärke-Multiplikator (0.0 – 1.0)
    """
    if np is None:
        return audio_data
    try:
        samples = np.frombuffer(audio_data, dtype=np.int16).astype(np.float32)
        n = len(samples)
        if n == 0:
            return audio_data

        # FFT Bandpass
        spectrum = np.fft.rfft(samples)
        freqs = np.fft.rfftfreq(n, 1.0 / sample_rate)

        # Zero out frequencies outside bandpass
        mask = (freqs >= low_hz) & (freqs <= high_hz)
        spectrum[~mask] = 0

        # Inverse FFT
        filtered = np.fft.irfft(spectrum, n=n)

        # VHF Radio-Effekt: Soft Clipping + Kompression
        max_val = np.max(np.abs(filtered)) if np.max(np.abs(filtered)) > 0 else 1.0
        filtered = filtered / max_val
        filtered = np.tanh(filtered * 1.5) * 0.8

        # ── Signal-Degradation: Weißes Rauschen beimischen ──
        if static_mix > 0.0:
            noise = np.random.normal(0, 0.15, n).astype(np.float32)
            # Rauschen ebenfalls Bandpass-filtern für realistischen Effekt
            noise_spec = np.fft.rfft(noise)
            noise_spec[~mask] = 0
            noise = np.fft.irfft(noise_spec, n=n)
            # Mischen: clean*(1-mix) + noise*mix
            filtered = filtered * (1.0 - static_mix) + noise * static_mix

        # Lautstärke anpassen
        filtered = filtered * volume_factor

        filtered = (filtered * 32767).astype(np.int16)
        return filtered.tobytes()
    except Exception:
        return audio_data


def _generate_heterodyne_squeal(duration_ms: int = 200,
                                 sample_rate: int = 48000) -> bytes:
    """Erzeugt einen Heterodyne-Squeal (Pfeifton bei Doppelsendung).

    Typisches Interferenz-Geräusch: ~1200 Hz + ~2400 Hz gemischt.
    """
    if np is None:
        return b""
    try:
        n = int(sample_rate * duration_ms / 1000)
        t = np.linspace(0, duration_ms / 1000, n, dtype=np.float32)
        # Zwei Sinustöne → Interferenz
        tone1 = np.sin(2 * np.pi * 1200 * t) * 0.3
        tone2 = np.sin(2 * np.pi * 2400 * t) * 0.15
        # Moduliertes Rauschen
        noise = np.random.normal(0, 0.05, n).astype(np.float32)
        signal = tone1 + tone2 + noise
        # Fade in/out
        fade = min(n // 10, 500)
        signal[:fade] *= np.linspace(0, 1, fade)
        signal[-fade:] *= np.linspace(1, 0, fade)
        return (signal * 32767).astype(np.int16).tobytes()
    except Exception:
        return b""


# ── Joystick-PTT Thread ──────────────────────────────────────────────
class JoystickPTTThread(threading.Thread):
    """Hintergrund-Thread der einen Joystick-Button für PTT abfragt."""

    def __init__(self, joy_id: int, button_index: int,
                 on_press, on_release):
        super().__init__(daemon=True)
        self.joy_id = joy_id
        self.button_index = button_index
        self._on_press = on_press
        self._on_release = on_release
        self._running = True
        self._was_pressed = False

    def run(self):
        if not _HAS_PYGAME:
            return
        try:
            joy = pygame.joystick.Joystick(self.joy_id)
            joy.init()
        except Exception:
            return

        while self._running:
            try:
                pygame.event.pump()
                pressed = joy.get_button(self.button_index)
                if pressed and not self._was_pressed:
                    self._was_pressed = True
                    self._on_press()
                elif not pressed and self._was_pressed:
                    self._was_pressed = False
                    self._on_release()
            except Exception:
                pass
            time.sleep(0.015)  # ~66 Hz Abfragerate

    def stop(self):
        self._running = False


def joystick_learn_button(timeout: float = 15.0) -> tuple[int, int] | None:
    """Wartet bis ein Joystick-Button gedrückt wird.

    Returns (joy_id, button_index) oder None bei Timeout.
    """
    if not _HAS_PYGAME:
        return None
    pygame.joystick.quit()
    pygame.joystick.init()
    n = pygame.joystick.get_count()
    if n == 0:
        return None
    joys = []
    for i in range(n):
        j = pygame.joystick.Joystick(i)
        j.init()
        joys.append(j)

    deadline = time.monotonic() + timeout
    while time.monotonic() < deadline:
        pygame.event.pump()
        for joy in joys:
            for b in range(joy.get_numbuttons()):
                if joy.get_button(b):
                    result = (joy.get_instance_id(), b)
                    return result
        time.sleep(0.02)
    return None


def list_joysticks() -> list[dict]:
    """Gibt alle angeschlossenen Joysticks/Gamepads zurück."""
    if not _HAS_PYGAME:
        return []
    pygame.joystick.quit()
    pygame.joystick.init()
    result = []
    for i in range(pygame.joystick.get_count()):
        j = pygame.joystick.Joystick(i)
        j.init()
        result.append({
            "id": j.get_instance_id(),
            "name": j.get_name(),
            "buttons": j.get_numbuttons(),
            "axes": j.get_numaxes(),
        })
    return result


class VoiceManager:
    """
    Verbindet sich im Hintergrund mit einem Mumble-Server.
    Push-to-Talk wird über pynput (Tastatur) ODER pygame (Joystick) erkannt.
    Audio-I/O über sounddevice (optional).
    VHF-Bandpass-Filter für authentischen Funk-Sound.
    Auto-Switch: Reagiert auf Server-Befehle für Kanalwechsel.
    """

    SAMPLE_RATE = 48000
    BLOCK_SIZE = 960   # 20 ms bei 48 kHz

    PTT_KEYS: dict[str, str] = {
        "rechte strg": "ctrl_r",
        "linke strg": "ctrl_l",
        "caps lock": "caps_lock",
        "rechte shift": "shift_r",
        "linke shift": "shift_l",
        "f1": "f1", "f2": "f2", "f3": "f3",
        "leertaste": "space",
    }

    def __init__(
        self, host: str, port: int, nick: str,
        password: str = "", ptt_key_name: str = "rechte strg",
        bridge: Optional[SignalBridge] = None,
        joystick_id: int = -1,
        joystick_button: int = -1,
    ):
        self.host = host
        self.port = port
        self.nick = nick
        self.password = password
        self.bridge = bridge
        self.mumble = None
        self.ptt_active = False
        self._running = False
        self._keyboard_listener = None
        self._joystick_thread: JoystickPTTThread | None = None
        self._mic_stream = None
        self._current_freq: str = ""
        self._vhf_filter_enabled = True  # VHF-Bandpass aktiv
        self._joystick_id = joystick_id
        self._joystick_button = joystick_button

        # Signal-Degradation (vom Server aktualisiert)
        self._signal_quality: float = 1.0
        self._static_mix: float = 0.0
        self._bandpass_lo: int = 300
        self._bandpass_hi: int = 3000
        self._volume_factor: float = 1.0
        self._via_relay: str = ""

        # PTT-Taste parsen
        self._ptt_key = None
        if _HAS_PYNPUT:
            attr = self.PTT_KEYS.get(ptt_key_name.lower(), "ctrl_r")
            try:
                self._ptt_key = getattr(pynput_keyboard.Key, attr)
            except AttributeError:
                self._ptt_key = pynput_keyboard.Key.ctrl_r

    def start(self) -> bool:
        """Mumble verbinden, PTT-Listener + Audio starten."""
        if not _HAS_PYMUMBLE:
            self._log("Voice: pymumble nicht installiert – übersprungen")
            return False

        try:
            self.mumble = pymumble.Mumble(
                self.host, self.nick,
                port=self.port, password=self.password,
                reconnect=True,
            )
            self.mumble.set_application_string("NEXUS-ATC Voice")
            self.mumble.start()
            self.mumble.is_ready()
            self.mumble.set_receive_sound(True)

            # Standard: stumm (PTT)
            self.mumble.users.myself.mute()

            # Empfangs-Callback
            from pymumble_py3.constants import PYMUMBLE_CLBK_SOUNDRECEIVED
            self.mumble.callbacks.set_callback(
                PYMUMBLE_CLBK_SOUNDRECEIVED, self._on_sound_received,
            )

            self._log("Voice: Mumble verbunden ✓")
        except Exception as e:
            self._log(f"Voice: Mumble-Fehler – {e}")
            return False

        # PTT-Tastatur-Listener
        if _HAS_PYNPUT and self._ptt_key:
            self._keyboard_listener = pynput_keyboard.Listener(
                on_press=self._on_key_press,
                on_release=self._on_key_release,
            )
            self._keyboard_listener.daemon = True
            self._keyboard_listener.start()
            self._log("Voice: PTT-Tastatur aktiv")

        # Joystick-PTT (wenn konfiguriert)
        if _HAS_PYGAME and self._joystick_id >= 0 and self._joystick_button >= 0:
            self._joystick_thread = JoystickPTTThread(
                self._joystick_id, self._joystick_button,
                on_press=self._on_joy_press,
                on_release=self._on_joy_release,
            )
            self._joystick_thread.start()
            self._log(f"Voice: Joystick-PTT aktiv (ID {self._joystick_id}, "
                      f"Button {self._joystick_button})")

        # Mikrofon-Stream
        if _HAS_AUDIO:
            try:
                self._mic_stream = sd.InputStream(
                    samplerate=self.SAMPLE_RATE,
                    channels=1, dtype="int16",
                    blocksize=self.BLOCK_SIZE,
                    callback=self._mic_callback,
                )
                self._mic_stream.start()
                self._log("Voice: Mikrofon aktiv")
            except Exception as e:
                self._log(f"Voice: Mikrofon-Fehler – {e}")

        self._running = True
        return True

    def stop(self):
        self._running = False
        if self._mic_stream:
            try:
                self._mic_stream.stop()
                self._mic_stream.close()
            except Exception:
                pass
        if self._keyboard_listener:
            self._keyboard_listener.stop()
        if self._joystick_thread:
            self._joystick_thread.stop()
        if self.mumble:
            try:
                self.mumble.stop()
            except Exception:
                pass
        self._log("Voice: Getrennt")

    def set_output_volume(self, volume_pct: int):
        """Setzt die Mumble-Empfangslautstärke (0-100%).

        Passt den internen _volume_factor an, der auf alle empfangenen
        Audio-Samples angewendet wird.  0 = stumm, 100 = volle Lautstärke.
        """
        vol = max(0, min(100, volume_pct))
        self._volume_factor = vol / 100.0

    def move_to_channel(self, channel_name: str) -> bool:
        """Wechselt den Mumble-Channel (erstellt ihn ggf.)."""
        if not self.mumble:
            return False
        try:
            target = None
            for ch_id, ch in self.mumble.channels.items():
                if ch["name"] == channel_name:
                    target = ch
                    break
            if target is None:
                self.mumble.channels.new_channel(0, channel_name, temporary=True)
                time.sleep(0.5)
                for ch_id, ch in self.mumble.channels.items():
                    if ch["name"] == channel_name:
                        target = ch
                        break
            if target:
                self.mumble.users.myself.move_in(target["channel_id"])
                self._current_freq = channel_name
                self._log(f"Voice: Channel → {channel_name}")
                return True
        except Exception as e:
            self._log(f"Voice: Channel-Fehler – {e}")
        return False

    def sync_frequency(self, com1_freq: float):
        """Auto-Switch: Wechselt Mumble-Channel auf COM1-Frequenz.

        Wird vom NetworkThread aufgerufen wenn sich die COM1-Frequenz ändert.
        Falls kein ATC auf der Frequenz online ist, wird in den UNICOM-Kanal
        gewechselt (122.800 oder 'UNICOM').
        """
        if com1_freq <= 0 or not self.mumble:
            return
        freq_str = f"{com1_freq:.3f}"
        if freq_str == self._current_freq:
            return  # Bereits auf dieser Frequenz
        self._log(f"Voice: Frequenzwechsel {self._current_freq} → {freq_str}")
        self.move_to_channel(freq_str)

    def handle_server_command(self, data: dict):
        """Handle server voice commands (VOICE_MOVE etc.)."""
        msg_type = data.get("type", "")
        if msg_type == "VOICE_MOVE":
            freq = data.get("frequency", "")
            if freq:
                self._log(f"Voice: Server-Befehl → Channel {freq}")
                self.move_to_channel(freq)

    # -- PTT Callbacks -------------------------------------------------------
    def _start_talking(self):
        """Gemeinsame PTT-Aktivierung (Tastatur + Joystick)."""
        if self.ptt_active:
            return
        self.ptt_active = True
        if self.mumble:
            try:
                self.mumble.users.myself.unmute()
            except Exception:
                pass
        if self.bridge:
            self.bridge.ptt_status.emit(True)

    def _stop_talking(self):
        """Gemeinsame PTT-Deaktivierung (Tastatur + Joystick)."""
        if not self.ptt_active:
            return
        self.ptt_active = False
        if self.mumble:
            try:
                self.mumble.users.myself.mute()
            except Exception:
                pass
        if self.bridge:
            self.bridge.ptt_status.emit(False)

    def _on_key_press(self, key):
        if key == self._ptt_key:
            self._start_talking()

    def _on_key_release(self, key):
        if key == self._ptt_key:
            self._stop_talking()

    def _on_joy_press(self):
        self._start_talking()

    def _on_joy_release(self):
        self._stop_talking()

    # -- Audio Callbacks -----------------------------------------------------
    def _mic_callback(self, indata, frames, time_info, status):
        """Mikrofon → VHF-Filter → Mumble (nur bei PTT)."""
        if self.ptt_active and self.mumble and self._running:
            try:
                raw = indata.tobytes()
                if self._vhf_filter_enabled:
                    raw = _vhf_bandpass(
                        raw, self.SAMPLE_RATE,
                        static_mix=self._static_mix,
                        volume_factor=self._volume_factor,
                    )
                self.mumble.sound_output.add_sound(raw)
            except Exception:
                pass

    def _on_sound_received(self, user, soundchunk):
        """Mumble → VHF-Filter → Lautsprecher."""
        if _HAS_AUDIO and self._running:
            try:
                pcm = soundchunk.pcm
                if self._vhf_filter_enabled:
                    pcm = _vhf_bandpass(
                        pcm, self.SAMPLE_RATE,
                        static_mix=self._static_mix,
                        volume_factor=self._volume_factor,
                    )
                audio = np.frombuffer(pcm, dtype=np.int16)
                sd.play(audio, samplerate=self.SAMPLE_RATE, blocking=False)
            except Exception:
                pass

    def update_signal_quality(self, data: dict):
        """Server SIGNAL_QUALITY → Audio-Parameter aktualisieren."""
        self._signal_quality = data.get("quality", 1.0)
        self._via_relay = data.get("via_relay", "")
        params = data.get("degradation", {})
        self._static_mix = params.get("static_mix", 0.0)
        self._bandpass_lo = params.get("bandpass_lo", 300)
        self._bandpass_hi = params.get("bandpass_hi", 3000)
        self._volume_factor = params.get("volume_factor", 1.0)
        q_pct = int(self._signal_quality * 100)
        relay_str = f" via {self._via_relay}" if self._via_relay else ""
        self._log(f"📡 Signal: {q_pct}%{relay_str}")

    def play_heterodyne(self, duration_s: float = 0.8):
        """Radio-Blocking: Heterodyn-Quietschton abspielen."""
        if _HAS_AUDIO:
            try:
                tone = _generate_heterodyne_squeal(self.SAMPLE_RATE, duration_s)
                audio = np.frombuffer(tone, dtype=np.int16)
                sd.play(audio, samplerate=self.SAMPLE_RATE, blocking=False)
            except Exception:
                pass

    def _log(self, msg: str):
        if self.bridge:
            self.bridge.log_message.emit(msg)


# ---------------------------------------------------------------------------
# Async Netzwerk-Loop (läuft in eigenem Thread)
# ---------------------------------------------------------------------------
class NetworkThread(threading.Thread):
    def __init__(
        self, server_url: str, callsign: str,
        flight_plan: dict,
        sm: SimConnect, ar: AircraftRequests,
        bridge: SignalBridge,
    ):
        super().__init__(daemon=True)
        self.server_url = server_url
        self.callsign = callsign
        self.flight_plan = flight_plan  # {"dep": ..., "arr": ..., "actype": ...}
        self.sm = sm
        self.ar = ar
        self.bridge = bridge
        self._running = True
        self._msg_queue: list[dict] = []  # Nachrichten-Queue (Thread-sicher über GIL)
        self._weather_metars: dict = {}  # empfangene METAR-Daten
        self._ae: AircraftEvents | None = None  # für Weather-Events
        self._ws = None   # aktiver WebSocket (für CPDLC-Responses)
        self._loop_ref = None  # asyncio event-loop Referenz
        # ── Weather Sync State ──
        self._weather_synced_icaos: set = set()  # bereits synchronisierte Airports
        self._live_weather_detected: bool = False  # True = MSFS Live-Wetter erkannt
        self._last_weather_sync: dict = {}  # Letztes Sync-Ergebnis pro ICAO
        # Failover: Liste der bekannten Server
        self._server_list: list[str] = [server_url]
        self._current_server_idx: int = 0

    def stop(self):
        self._running = False

    def run(self):
        asyncio.run(self._main_loop())

    def _fetch_backup_servers(self):
        """Holt Backup-Server-Liste vom primären Server (synchron, best-effort)."""
        try:
            import urllib.request
            # ws://host:port/ws/pilot → http://host:port/api/failover/servers
            base = self.server_url.replace("ws://", "http://").replace("wss://", "https://")
            base = base.split("/ws/")[0]
            url = f"{base}/api/failover/servers"
            req = urllib.request.Request(url, method="GET")
            with urllib.request.urlopen(req, timeout=5) as resp:
                data = json.loads(resp.read())
                servers = data.get("servers", [])
                # Server nach Priorität sortieren
                servers.sort(key=lambda s: s.get("priority", 99))
                new_list = []
                for s in servers:
                    surl = s.get("url", "")
                    if surl and surl not in new_list:
                        new_list.append(surl)
                if new_list:
                    self._server_list = new_list
                    self.bridge.log_message.emit(
                        f"🌐 Failover: {len(new_list)} Server bekannt")
        except Exception:
            pass  # Kein Failover-Endpoint verfügbar → nur Primär-Server nutzen

    async def _main_loop(self):
        self._loop_ref = asyncio.get_event_loop()
        # Backup-Server-Liste holen (einmalig)
        self._fetch_backup_servers()

        while self._running:
            server_url = self._server_list[self._current_server_idx]
            self.bridge.connection_status.emit(
                f"Verbinde mit Server ({self._current_server_idx + 1}/"
                f"{len(self._server_list)}) …")

            connected = False
            for attempt in range(FAILOVER_MAX_RETRIES):
                if not self._running:
                    return
                try:
                    async with websockets.connect(server_url) as ws:
                        self._ws = ws
                        connected = True
                        self.bridge.connection_status.emit(
                            f"Verbunden ✓ ({server_url.split('//')[1].split('/')[0]})")
                        self.bridge.server_status.emit(True)
                        self.bridge.log_message.emit(
                            f"Server verbunden: {server_url}")

                        send_task = asyncio.create_task(self._send_loop(ws))
                        recv_task = asyncio.create_task(self._recv_loop(ws))

                        done, pending = await asyncio.wait(
                            [send_task, recv_task],
                            return_when=asyncio.FIRST_COMPLETED,
                        )
                        for t in pending:
                            t.cancel()
                        # Verbindung verloren → Reconnect versuchen
                        self.bridge.connection_status.emit("Verbindung verloren – Reconnect …")
                        self.bridge.server_status.emit(False)
                        break  # Inner loop beenden, outer loop macht Failover

                except ConnectionRefusedError:
                    self.bridge.log_message.emit(
                        f"Server {server_url} nicht erreichbar "
                        f"(Versuch {attempt + 1}/{FAILOVER_MAX_RETRIES})")
                    await asyncio.sleep(FAILOVER_RETRY_DELAY_S)
                except Exception as e:
                    self.bridge.log_message.emit(
                        f"Verbindungsfehler: {e} "
                        f"(Versuch {attempt + 1}/{FAILOVER_MAX_RETRIES})")
                    await asyncio.sleep(FAILOVER_RETRY_DELAY_S)

            if not self._running:
                return

            if not connected:
                # Alle Versuche für diesen Server fehlgeschlagen → nächsten probieren
                self._current_server_idx = (
                    (self._current_server_idx + 1) % len(self._server_list)
                )
                self.bridge.log_message.emit(
                    f"⚠ Failover → Server {self._current_server_idx + 1}/"
                    f"{len(self._server_list)}")
            else:
                # Verbindung war da, wurde getrennt → gleichen Server erneut versuchen
                await asyncio.sleep(FAILOVER_RETRY_DELAY_S)

        self.bridge.connection_status.emit("Getrennt")
        self.bridge.server_status.emit(False)

    async def _send_loop(self, ws):
        """Liest MSFS-Daten und sendet an Server."""
        while self._running:
            try:
                lat = self.ar.get("PLANE_LATITUDE")
                lon = self.ar.get("PLANE_LONGITUDE")
                alt = self.ar.get("PLANE_ALTITUDE")
                hdg = self.ar.get("PLANE_HEADING_DEGREES_TRUE")
                gs  = self.ar.get("GROUND_VELOCITY")
                com1_raw = self.ar.get("COM_ACTIVE_FREQUENCY:1")
                xpdr_raw = self.ar.get("TRANSPONDER_CODE:1")
                ident_raw = self.ar.get("TRANSPONDER_IDENT:1")
                kohlsman_raw = self.ar.get("KOHLSMAN_SETTING_MB")
                # Transponder-Modus: 0=OFF, 1=STBY, 2=TEST, 3=ON, 4=ALT, 5=GND
                xpdr_state_raw = self.ar.get("TRANSPONDER_STATE:1")

                # ── COM Volume (0.0 – 1.0 → 0 – 100%) ──
                com1_vol_raw = self.ar.get("COM_RECEIVE_ALL:1")
                com2_vol_raw = self.ar.get("COM_RECEIVE_ALL:2")
                com1_vol = max(0, min(100, int(float(com1_vol_raw) * 100))) if com1_vol_raw is not None else 100
                com2_vol = max(0, min(100, int(float(com2_vol_raw) * 100))) if com2_vol_raw is not None else 100
                self.bridge.com_volume.emit(com1_vol, com2_vol)

                # ── Lautstärke an Mumble weitergeben ──
                self._sync_audio_volume(com1_vol)

                # ── Erweiterte Telemetrie ──
                ias_raw = self.ar.get("AIRSPEED_INDICATED")
                vs_raw = self.ar.get("VERTICAL_SPEED")
                sel_alt_raw = self.ar.get("AUTOPILOT_ALTITUDE_LOCK_VAR")

                # COM1-Frequenz aufbereiten
                com1 = 0.0
                if com1_raw is not None:
                    com1 = float(com1_raw)
                    # Normalisierung: Falls Hz statt MHz
                    if com1 > 10000:
                        com1 = com1 / 1_000_000
                    elif com1 > 1000:
                        com1 = com1 / 1000
                self.bridge.current_com1.emit(round(com1, 3))

                # Transponder-Code aufbereiten  (SimConnect → BCD-Decode)
                squawk = 2000
                if xpdr_raw is not None:
                    raw_int = int(xpdr_raw)
                    # SimConnect liefert BCD-codiert: 0x1200 = Squawk 1200
                    if raw_int > 7777:
                        # BCD → Dezimal
                        d0 = (raw_int >> 12) & 0xF
                        d1 = (raw_int >> 8) & 0xF
                        d2 = (raw_int >> 4) & 0xF
                        d3 = raw_int & 0xF
                        squawk = d0 * 1000 + d1 * 100 + d2 * 10 + d3
                    else:
                        squawk = raw_int

                # IDENT-Status (Bool)
                squawk_ident = bool(ident_raw) if ident_raw else False

                # Kohlsman / QNH
                kohlsman_mb = 0.0
                if kohlsman_raw is not None:
                    kohlsman_mb = float(kohlsman_raw) * 16.0  # SimConnect → mbar Korrektur
                    # SimConnect liefert Wert in mbar / 16 (interne Einheit)
                    if kohlsman_mb > 2000:
                        kohlsman_mb = kohlsman_mb / 16.0
                    if kohlsman_mb < 800 or kohlsman_mb > 1100:
                        kohlsman_mb = 0.0  # ungültiger Wert

                # Transponder-Modus auswerten
                # MSFS TRANSPONDER_STATE: 0=OFF, 1=STBY, 2=TEST, 3=ON, 4=ALT/TA, 5=GND
                transponder_mode = ""
                if xpdr_state_raw is not None:
                    xpdr_state = int(xpdr_state_raw)
                    xpdr_mode_map = {0: "OFF", 1: "STBY", 2: "TEST",
                                     3: "ON", 4: "TA/RA", 5: "GND"}
                    transponder_mode = xpdr_mode_map.get(xpdr_state, "")

                # Aktuellen Squawk an UI melden (für Matching)
                self.bridge.current_squawk.emit(squawk)

                payload = {
                    "callsign": self.callsign,
                    "lat": float(lat) if lat is not None else 0.0,
                    "lon": float(lon) if lon is not None else 0.0,
                    "alt": float(alt) if alt is not None else 0.0,
                    "heading": float(hdg) if hdg is not None else 0.0,
                    "groundspeed": float(gs) if gs is not None else 0.0,
                    "com1_freq": round(com1, 3),
                    "squawk": squawk,
                    "squawk_ident": squawk_ident,
                    "kohlsman_mb": round(kohlsman_mb, 1),
                    "transponder_mode": transponder_mode,
                    # Erweiterte Telemetrie
                    "ias": round(float(ias_raw), 1) if ias_raw is not None else 0.0,
                    "vs": round(float(vs_raw), 0) if vs_raw is not None else 0.0,
                    "sel_alt": round(float(sel_alt_raw), 0) if sel_alt_raw is not None else 0.0,
                }
                # Flugplan-Daten mitsenden
                payload.update(self.flight_plan)

                await ws.send(json.dumps(payload))
                self.bridge.log_message.emit(
                    f"→ {self.callsign}  "
                    f"lat={payload['lat']:.5f}  "
                    f"lon={payload['lon']:.5f}  "
                    f"alt={payload['alt']:.0f}  "
                    f"gs={payload['groundspeed']:.0f}kn  "
                    f"sq={squawk:04d}  "
                    f"f={com1:.3f}"
                )

                # Queued messages (Pilot → Server) absenden
                while self._msg_queue:
                    queued = self._msg_queue.pop(0)
                    await ws.send(json.dumps(queued))
                    self.bridge.log_message.emit(
                        f"✉ Nachricht gesendet: {queued.get('message', '')[:50]}")

            except Exception as e:
                self.bridge.log_message.emit(f"Send-Fehler: {e}")
                break

            await asyncio.sleep(INTERVAL)

    async def _recv_loop(self, ws):
        """Empfängt Positionen anderer Piloten und ATC-Nachrichten vom Server."""
        while self._running:
            try:
                raw = await ws.recv()
                data = json.loads(raw)

                # Typed Messages (ATC_UPDATE, ATC_ONLINE, ATC_OFFLINE)
                if isinstance(data, dict) and "type" in data:
                    self._handle_typed_message(data)
                    continue

                if isinstance(data, list):
                    for item in data:
                        self._handle_other_pilot(item)
                else:
                    self._handle_other_pilot(data)
            except websockets.ConnectionClosed:
                break
            except Exception as e:
                self.bridge.log_message.emit(f"Recv-Fehler: {e}")
                break

    def _handle_typed_message(self, data: dict):
        """Verarbeitet typisierte Server-Nachrichten."""
        msg_type = data.get("type", "")

        if msg_type == "ATC_UPDATE":
            self.bridge.atc_instruction.emit(data)
            self.bridge.log_message.emit(
                f"📡 ATC-Anweisung von {data.get('from', '?')}: "
                f"SQ={data.get('assigned_squawk', '')} "
                f"ALT={data.get('assigned_alt', '')} "
                f"CLR={data.get('cleared', '')}")

        elif msg_type == "ATC_ONLINE":
            self.bridge.atc_online.emit(data)
            self.bridge.log_message.emit(
                f"🗼 ATC online: {data.get('station', '?')} "
                f"({data.get('frequency', 0):.3f} MHz)")

        elif msg_type == "ATC_OFFLINE":
            station = data.get("station", "")
            self.bridge.atc_offline.emit(station)
            self.bridge.log_message.emit(f"🗼 ATC offline: {station}")

        elif msg_type == "FREQ_MSG":
            self.bridge.freq_message.emit(data)
            sender = data.get("from", "?")
            message = data.get("message", "")
            self.bridge.log_message.emit(f"💬 [{sender}]: {message}")

        elif msg_type == "VOICE_MOVE":
            self.bridge.voice_command.emit(data)
            freq = data.get("frequency", "")
            self.bridge.log_message.emit(f"🎙 Voice: Server → Channel {freq}")

        elif msg_type == "WEATHER_SYNC":
            self._weather_metars = data.get("metars", {})
            n = len(self._weather_metars)
            self.bridge.log_message.emit(f"🌦️ Wetter empfangen: {n} METARs")
            self._apply_weather()

        elif msg_type == "CPDLC_MSG":
            self.bridge.cpdlc_message.emit(data)
            sender = data.get("from", "ATC")
            text = data.get("text", "")
            self.bridge.log_message.emit(
                f"📡 CPDLC von {sender}: {text}")

        elif msg_type == "SIGNAL_QUALITY":
            self.bridge.signal_quality.emit(data)

        elif msg_type == "RADIO_BLOCKED":
            self.bridge.radio_blocked.emit(data)
            blocker = data.get("blocked_by", "?")
            self.bridge.log_message.emit(
                f"⚠️ Radio-Blocking! Gleichzeitige TX von {blocker}")

        elif msg_type == "STCA_ALERT":
            alerts = data.get("alerts", [])
            cs = self.callsign
            for a in alerts:
                if cs in (a.get("callsign_a"), a.get("callsign_b")):
                    self.bridge.stca_alert.emit(a)
                    other = a.get("callsign_b") if a.get("callsign_a") == cs else a.get("callsign_a")
                    self.bridge.log_message.emit(
                        f"🔴 STCA! Konflikt mit {other} in "
                        f"{a.get('time_to_conflict_s', 0):.0f}s")
                    break

        elif msg_type == "PILOT_CONTACT_REQUEST":
            self.bridge.contact_request.emit(data)
            station = data.get("station", "?")
            freq = data.get("frequency", 0)
            self.bridge.log_message.emit(
                f"📡 CONTACT {station} ON {freq:.3f} MHz")

        elif msg_type == "SLOT_ASSIGNED":
            self.bridge.slot_assigned.emit(data)
            slot_t = data.get("slot_time", "")
            tsat = data.get("tsat", "")
            evt_name = data.get("event_name", "")
            apt = data.get("airport", "")
            direction = data.get("direction", "")
            self.bridge.log_message.emit(
                f"🎫 Slot zugewiesen: {evt_name} – {apt} {direction} "
                f"Slot: {slot_t[11:16] if len(slot_t) > 16 else slot_t}Z "
                f"TSAT: {tsat[11:16] if len(tsat) > 16 else tsat}Z")

        elif msg_type == "PILOT_METAR":
            icao = data.get("icao", "")
            raw = data.get("metar_raw", "")
            wind_dir = data.get("wind_dir", 0)
            wind_spd = data.get("wind_speed_kt", 0)
            gust = data.get("wind_gust_kt", 0)
            qnh = data.get("qnh", 0)
            rwy = data.get("recommended_runway", "")
            vis = data.get("visibility_m", 9999)
            wx_str = data.get("wx", "")

            gust_txt = f"G{gust}" if gust else ""
            rwy_txt = f" → empfohlene RWY {rwy}" if rwy else ""
            vis_txt = f" VIS {vis}m" if vis < 9999 else ""
            wx_txt = f" {wx_str}" if wx_str else ""
            self.bridge.log_message.emit(
                f"🌦️ METAR {icao}: {wind_dir:03d}°/{wind_spd}{gust_txt}kt "
                f"QNH {qnh}{vis_txt}{wx_txt}{rwy_txt}")
            if raw:
                self.bridge.log_message.emit(
                    f"   RAW: {raw[:90]}")

    # ── Weather ───────────────────────────────────────────────────
    _weather_dll = None          # MSFS 2024 SDK SimConnect DLL (native)
    _weather_hsim = None         # eigener SimConnect handle für Weather
    _weather_warned = False      # Warnung schon gezeigt?

    def _init_weather_dll(self):
        """Lädt die MSFS 2024 SDK SimConnect.dll für Weather-APIs."""
        import ctypes, os
        from ctypes import c_void_p, byref
        sdk_paths = [
            r"C:\MSFS 2024 SDK\SimConnect SDK\lib\SimConnect.dll",
            r"C:\MSFS SDK\SimConnect SDK\lib\SimConnect.dll",
            os.path.join(os.path.dirname(__file__), "SimConnect_SDK.dll"),
        ]
        for p in sdk_paths:
            if os.path.isfile(p):
                try:
                    dll = ctypes.WinDLL(p)
                    h = c_void_p()
                    hr = dll.SimConnect_Open(
                        byref(h), b"NexusATCWeather", None, 0, None, 0)
                    if hr == 0 and h.value:
                        self._weather_dll = dll
                        self._weather_hsim = h
                        self.bridge.log_message.emit(
                            f"🌦️ MSFS 2024 SDK DLL geladen")
                        return True
                except Exception:
                    pass
        return False

    def _apply_weather(self):
        """Wendet empfangene METARs auf den Simulator an."""
        if not self._weather_metars:
            return
        try:
            if self._ae is None:
                self._ae = AircraftEvents(self.sm)

            # ── 1) Versuche Live-Wetter via MSFS 2024 SDK ────────────
            if self._weather_dll is None:
                self._init_weather_dll()

            if self._weather_dll and self._weather_hsim:
                try:
                    # CUSTOM-Modus setzen (nicht Global = Live-Wetter!)
                    self._weather_dll.SimConnect_WeatherSetModeCustom(
                        self._weather_hsim)
                except Exception:
                    pass

            # ── 2) QNH / Kohlsman setzen (funktioniert!) ─────────────
            nearest_icao = ""
            nearest_qnh = 0
            for icao, metar in self._weather_metars.items():
                if not isinstance(metar, dict):
                    continue
                qnh_hpa = metar.get("qnh_hpa", 0)
                qnh_inhg = metar.get("qnh_inhg", 0)
                if qnh_hpa and 900 < qnh_hpa < 1100:
                    nearest_icao = icao
                    nearest_qnh = qnh_hpa
                    try:
                        evt = self._ae.find("KOHLSMAN_SET")
                        val = int(round(qnh_hpa * 16))  # mbar × 16
                        evt(val)
                        self.bridge.log_message.emit(
                            f"🌦️ QNH gesetzt: {icao} "
                            f"({qnh_hpa} hPa / {qnh_inhg:.2f} inHg)")
                    except Exception as e:
                        self.bridge.log_message.emit(
                            f"🌦️ QNH-Fehler: {e}")
                    break

            # ── 3) Wetter-Abgleich: MSFS Ambient vs METAR ────────────
            self._check_weather_match(nearest_icao)

        except Exception as e:
            self.bridge.log_message.emit(f"🌦️ Wetter-Fehler: {e}")

    def _check_weather_match(self, icao: str):
        """Vergleicht MSFS-Wetter mit METAR – warnt bei Abweichung."""
        if not icao or icao not in self._weather_metars:
            return
        metar = self._weather_metars[icao]
        if not isinstance(metar, dict):
            return
        try:
            sim_temp = self.ar.get("AMBIENT_TEMPERATURE")
            sim_wind_dir = self.ar.get("AMBIENT_WIND_DIRECTION")
            sim_wind_spd = self.ar.get("AMBIENT_WIND_VELOCITY")

            if sim_temp is None:
                return

            metar_temp = metar.get("temperature_c", 15)
            metar_wind = metar.get("wind_speed", 0)
            metar_dir = metar.get("wind_dir", 0)

            # Toleranz: Temp ±5°C, Wind ±10kt
            temp_ok = abs(float(sim_temp) - metar_temp) < 8
            wind_ok = abs(float(sim_wind_spd) - metar_wind) < 15

            if temp_ok and wind_ok:
                self.bridge.log_message.emit(
                    f"✅ Wetter passt: {icao} "
                    f"(Sim: {sim_temp:.0f}°C/{sim_wind_spd:.0f}kt, "
                    f"METAR: {metar_temp}°C/{metar_wind}kt)")
                self._weather_warned = False
            else:
                if not self._weather_warned:
                    self._weather_warned = True
                    self.bridge.log_message.emit(
                        f"⚠️ WETTER-ABWEICHUNG erkannt!")
                    self.bridge.log_message.emit(
                        f"   Sim: {sim_temp:.0f}°C, Wind {sim_wind_dir:.0f}°/{sim_wind_spd:.0f}kt")
                    self.bridge.log_message.emit(
                        f"   METAR {icao}: {metar_temp}°C, Wind {metar_dir}°/{metar_wind}kt")
                    self.bridge.log_message.emit(
                        f"   → Bitte in MSFS: Flugbedingungen → Wetter → "
                        f"'Live-Wetter' aktivieren!")
        except Exception:
            pass

    # ── Automatische Wetter-Synchronisation ────────────────────────────
    def _fetch_metar_http(self, icao: str) -> Optional[dict]:
        """Holt METAR von Server via HTTP (synchron, für Thread-Kontext)."""
        if not icao or len(icao) != 4:
            return None
        import urllib.request
        try:
            server_base = self.server_url.replace("ws://", "http://").replace(
                "wss://", "https://").replace("/ws/pilot", "")
            url = f"{server_base}/metar/{icao.upper()}"
            req = urllib.request.Request(url, headers={"Accept": "application/json"})
            with urllib.request.urlopen(req, timeout=8) as resp:
                data = json.loads(resp.read().decode())
            if data and isinstance(data, dict):
                return data
        except Exception as e:
            self.bridge.log_message.emit(f"🌦️ METAR-Abruf {icao}: {e}")
        return None

    def _get_sim_weather(self) -> Optional[dict]:
        """Liest aktuelles Sim-Wetter (Ambient) via SimConnect."""
        try:
            sim_temp = self.ar.get("AMBIENT_TEMPERATURE")
            sim_wind_dir = self.ar.get("AMBIENT_WIND_DIRECTION")
            sim_wind_spd = self.ar.get("AMBIENT_WIND_VELOCITY")
            sim_pressure = self.ar.get("AMBIENT_PRESSURE")
            sim_visibility = self.ar.get("AMBIENT_VISIBILITY")
            if sim_temp is None:
                return None
            return {
                "temperature_c": float(sim_temp) if sim_temp is not None else 15.0,
                "wind_dir": float(sim_wind_dir) if sim_wind_dir is not None else 0.0,
                "wind_speed_kt": float(sim_wind_spd) if sim_wind_spd is not None else 0.0,
                "pressure_hpa": float(sim_pressure) * 33.8639 if sim_pressure is not None else 1013.0,
                "visibility_m": float(sim_visibility) if sim_visibility is not None else 9999.0,
            }
        except Exception:
            return None

    def _detect_weather_mode(self, metar_data: dict) -> str:
        """
        Erkennt ob MSFS Live-Wetter aktiv ist.
        Vergleicht METAR mit Sim-Ambient — bei guter Übereinstimmung = LIVE.
        Returns: 'LIVE', 'CUSTOM', oder 'UNKNOWN'
        """
        sim_wx = self._get_sim_weather()
        if sim_wx is None:
            return "UNKNOWN"

        metar_temp = metar_data.get("temperature_c", metar_data.get("temperature", 15))
        metar_wind = metar_data.get("wind_speed_kt", metar_data.get("wind_speed", 0))
        metar_qnh = metar_data.get("qnh", metar_data.get("qnh_hpa", 1013))

        # QNH-Vergleich (AMBIENT_PRESSURE ist in inHg)
        sim_qnh = sim_wx["pressure_hpa"]

        temp_diff = abs(sim_wx["temperature_c"] - float(metar_temp))
        wind_diff = abs(sim_wx["wind_speed_kt"] - float(metar_wind))
        qnh_diff = abs(sim_qnh - float(metar_qnh))

        # Heuristik: Wenn Temp ±5°C UND Wind ±8kt UND QNH ±3hPa → Live
        if temp_diff < 5 and wind_diff < 8 and qnh_diff < 3:
            self._live_weather_detected = True
            return "LIVE"
        else:
            self._live_weather_detected = False
            return "CUSTOM"

    def _build_metar_observation(self, metar_data: dict) -> str:
        """
        Baut einen METAR-Observation-String für WEATHER_SET_OBSERVATION.
        Nutzt den raw METAR wenn verfügbar.
        """
        raw = metar_data.get("raw", metar_data.get("metar_raw", ""))
        if raw:
            return raw.strip()
        # Fallback: Synthetisches METAR generieren
        icao = metar_data.get("icao", metar_data.get("station", "XXXX"))
        wind_dir = metar_data.get("wind_dir", 0)
        wind_spd = metar_data.get("wind_speed_kt", metar_data.get("wind_speed", 0))
        gust = metar_data.get("wind_gust_kt", metar_data.get("wind_gust", 0))
        vis = metar_data.get("visibility_m", 9999)
        temp = metar_data.get("temperature_c", metar_data.get("temperature", 15))
        dew = metar_data.get("dewpoint_c", metar_data.get("dewpoint", 10))
        qnh = metar_data.get("qnh", metar_data.get("qnh_hpa", 1013))
        clouds = metar_data.get("clouds", "")

        wind_str = f"{wind_dir:03d}{wind_spd:02d}"
        if gust:
            wind_str += f"G{gust:02d}"
        wind_str += "KT"

        vis_str = f"{vis:04d}" if vis < 9999 else "9999"

        t_str = f"M{abs(temp):02d}" if temp < 0 else f"{temp:02d}"
        d_str = f"M{abs(dew):02d}" if dew < 0 else f"{dew:02d}"

        cloud_str = clouds if clouds else "SKC"

        return f"{icao} AUTO {wind_str} {vis_str} {cloud_str} {t_str}/{d_str} Q{qnh:04d}"

    def _set_weather_from_metar(self, icao: str, metar_data: dict):
        """
        Setzt das Simulator-Wetter aus METAR-Daten.
        MSFS 2024 blockiert Wetteränderungen wenn Live-Wetter aktiv ist.
        Lösung: CLEAR_SKIES → Pause → CUSTOM → METAR setzen.
        """
        try:
            if self._ae is None:
                self._ae = AircraftEvents(self.sm)

            metar_str = self._build_metar_observation(metar_data)
            set_methods = []

            # ── Schritt 0: SDK DLL laden (für WeatherSetModeCustom) ──
            if self._weather_dll is None:
                self._init_weather_dll()

            # ── Schritt 1: CLEAR_SKIES erzwingen (Live-Wetter-Block brechen) ──
            clear_metar = f"{icao} AUTO 00000KT 9999 SKC 15/10 Q1013"
            try:
                evt_obs = self._ae.find("WEATHER_SET_OBSERVATION")
                if evt_obs:
                    evt_obs(clear_metar)
                    set_methods.append("CLEAR")
                    self.bridge.log_message.emit(
                        f"🌦️ Schritt 1: CLEAR_SKIES für {icao} gesetzt")
            except Exception:
                pass

            # SDK DLL: WeatherSetModeCustom setzen (statt Global!)
            if self._weather_dll and self._weather_hsim:
                try:
                    # Versuche WeatherSetModeCustom (MSFS 2024 SDK)
                    self._weather_dll.SimConnect_WeatherSetModeCustom(
                        self._weather_hsim)
                    set_methods.append("CUSTOM-MODE")
                    self.bridge.log_message.emit(
                        f"🌦️ Schritt 1b: SDK CUSTOM-Modus aktiviert")
                except Exception:
                    pass

            # ── Schritt 2: Warten (MSFS braucht Zeit um Modus zu wechseln) ──
            time.sleep(1.5)

            # ── Schritt 3: Eigentliches METAR setzen ──
            try:
                evt = self._ae.find("WEATHER_SET_OBSERVATION")
                if evt:
                    evt(metar_str)
                    set_methods.append("OBSERVATION")
            except Exception:
                pass

            # ── Schritt 4: QNH / Kohlsman setzen ──
            qnh_hpa = metar_data.get("qnh", metar_data.get("qnh_hpa", 0))
            if qnh_hpa and 900 < qnh_hpa < 1100:
                try:
                    evt_k = self._ae.find("KOHLSMAN_SET")
                    val = int(round(qnh_hpa * 16))  # mbar × 16
                    evt_k(val)
                    set_methods.append("QNH")
                except Exception:
                    pass

            # ── Schritt 5: Nochmals METAR bestätigen (doppelt hält besser) ──
            time.sleep(0.5)
            try:
                evt2 = self._ae.find("WEATHER_SET_OBSERVATION")
                if evt2:
                    evt2(metar_str)
                    set_methods.append("CONFIRM")
            except Exception:
                pass

            methods_str = "+".join(set_methods) if set_methods else "KEINE"

            wind_dir = metar_data.get("wind_dir", 0)
            wind_spd = metar_data.get("wind_speed_kt", metar_data.get("wind_speed", 0))
            temp = metar_data.get("temperature_c", metar_data.get("temperature", 15))
            vis = metar_data.get("visibility_m", 9999)
            clouds = metar_data.get("clouds", "")

            self.bridge.log_message.emit(
                f"🌦️ Wetter gesetzt für {icao} [{methods_str}]: "
                f"{wind_dir:03d}°/{wind_spd}kt "
                f"QNH {qnh_hpa} "
                f"T{temp:+d}°C "
                f"VIS {vis}m "
                f"{clouds}")

            return True

        except Exception as e:
            self.bridge.log_message.emit(f"🌦️ Wetter-Setzung fehlgeschlagen: {e}")
            return False

    def _log_weather_deviation(self, icao: str, metar_data: dict):
        """Loggt Abweichung zwischen METAR und Sim-Wetter (bei Live-Wetter)."""
        sim_wx = self._get_sim_weather()
        if sim_wx is None:
            return

        metar_temp = float(metar_data.get("temperature_c", metar_data.get("temperature", 15)))
        metar_wind = float(metar_data.get("wind_speed_kt", metar_data.get("wind_speed", 0)))
        metar_qnh = float(metar_data.get("qnh", metar_data.get("qnh_hpa", 1013)))
        metar_wind_dir = metar_data.get("wind_dir", 0)
        metar_vis = metar_data.get("visibility_m", 9999)

        temp_diff = abs(sim_wx["temperature_c"] - metar_temp)
        wind_diff = abs(sim_wx["wind_speed_kt"] - metar_wind)
        qnh_diff = abs(sim_wx["pressure_hpa"] - metar_qnh)

        sync_result = {
            "icao": icao,
            "mode": "LIVE",
            "metar_temp": metar_temp,
            "metar_wind": metar_wind,
            "metar_qnh": metar_qnh,
            "sim_temp": sim_wx["temperature_c"],
            "sim_wind": sim_wx["wind_speed_kt"],
            "sim_qnh": sim_wx["pressure_hpa"],
            "temp_diff": temp_diff,
            "wind_diff": wind_diff,
            "qnh_diff": qnh_diff,
        }
        self._last_weather_sync[icao] = sync_result
        self.bridge.weather_sync.emit(sync_result)

        if temp_diff > 3 or qnh_diff > 2:
            self.bridge.log_message.emit(
                f"⚠️ Wetterabweichung bei {icao}: "
                f"Sim={sim_wx['temperature_c']:.0f}°C/{sim_wx['wind_speed_kt']:.0f}kt/"
                f"{sim_wx['pressure_hpa']:.0f}hPa  "
                f"METAR={metar_temp:.0f}°C/{metar_wind:.0f}kt/{metar_qnh:.0f}hPa")
        else:
            self.bridge.log_message.emit(
                f"✅ Live-Wetter für {icao} stimmt überein "
                f"(ΔT={temp_diff:.1f}°C, ΔWind={wind_diff:.0f}kt, ΔQNH={qnh_diff:.0f}hPa)")

    def _sync_audio_volume(self, com1_vol_pct: int):
        """Synchronisiert die MSFS COM1-Empfangslautstärke mit dem Mumble-Client.

        Wird bei jedem Sim-Loop aufgerufen.  Setzt die Mumble-Ausgabelautstärke
        nur wenn sich der Wert signifikant geändert hat (±2%), um
        unnötige Updates zu vermeiden.
        """
        if not hasattr(self, '_last_synced_vol'):
            self._last_synced_vol = -1
        if abs(com1_vol_pct - self._last_synced_vol) < 2:
            return  # Kein nennenswerter Unterschied
        self._last_synced_vol = com1_vol_pct

        # Mumble-Lautstärke anpassen (VoiceManager)
        if hasattr(self, 'voice_mgr') and self.voice_mgr:
            self.voice_mgr.set_output_volume(com1_vol_pct)
        elif hasattr(self, '_parent_window') and self._parent_window:
            w = self._parent_window
            if hasattr(w, 'voice_mgr') and w.voice_mgr:
                w.voice_mgr.set_output_volume(com1_vol_pct)

    def _get_current_position(self) -> tuple:
        """Gibt (lat, lon) der aktuellen Sim-Position zurück."""
        try:
            lat = self.ar.get("PLANE_LATITUDE")
            lon = self.ar.get("PLANE_LONGITUDE")
            if lat is not None and lon is not None:
                return float(lat), float(lon)
        except Exception:
            pass
        return 0.0, 0.0

    def _get_nearest_icao_from_server(self, lat: float, lon: float) -> str:
        """Fragt den Server nach dem nächsten ICAO (synchron, für Thread)."""
        try:
            import urllib.request
            server_base = self.server_url.replace("ws://", "http://").replace(
                "wss://", "https://").replace("/ws/pilot", "")
            url = f"{server_base}/nearest_airport?lat={lat}&lon={lon}"
            req = urllib.request.Request(url, headers={"Accept": "application/json"})
            with urllib.request.urlopen(req, timeout=5) as resp:
                data = json.loads(resp.read().decode())
            if data and isinstance(data, list) and len(data) > 0:
                return (data[0].get("icao", "") or "").upper()
        except Exception:
            pass
        return ""

    def sync_weather_for_airport(self, icao: str, force: bool = False):
        """
        Automatische Wetter-Synchronisation.
        Setzt Wetter NUR wenn der Pilot sich in der Nähe befindet (<10 NM).
        1. Aktuelle Position aus SimConnect holen
        2. Nächsten Flughafen bestimmen
        3. METAR abrufen
        4. Entfernung prüfen (<10 NM)
        5. Wetter setzen/loggen
        """
        icao = icao.strip().upper()
        if not icao or len(icao) != 4:
            return

        # Bereits synchronisiert? (außer force=True)
        if not force and icao in self._weather_synced_icaos:
            return

        # ── Aktuelle Position holen ──
        pilot_lat, pilot_lon = self._get_current_position()

        self.bridge.log_message.emit(
            f"🌦️ Weather-Sync gestartet für {icao} …")

        # 1) METAR abrufen
        metar_data = self._fetch_metar_http(icao)
        if not metar_data:
            self.bridge.log_message.emit(
                f"🌦️ Kein METAR verfügbar für {icao}")
            return

        # ── Entfernung zum Zielflughafen prüfen ──
        apt_lat = metar_data.get("station_lat", metar_data.get("lat", 0))
        apt_lon = metar_data.get("station_lon", metar_data.get("lon", 0))
        if pilot_lat != 0 and pilot_lon != 0 and apt_lat and apt_lon:
            dist_nm = _haversine_nm(pilot_lat, pilot_lon, float(apt_lat), float(apt_lon))
            if dist_nm > 10.0:
                self.bridge.log_message.emit(
                    f"⚠️ METAR {icao} ignoriert – {dist_nm:.0f} NM entfernt "
                    f"(nur <10 NM wird gesetzt). Wird gecacht.")
                # Trotzdem cachen für späteren Anflug
                self._weather_metars[icao] = metar_data
                return
        elif pilot_lat != 0 and pilot_lon != 0:
            # Keine Koordinaten vom METAR → nächsten Airport per Server prüfen
            nearest_icao = self._get_nearest_icao_from_server(pilot_lat, pilot_lon)
            if nearest_icao and nearest_icao != icao:
                self.bridge.log_message.emit(
                    f"⚠️ Nächster Airport ist {nearest_icao}, nicht {icao} – "
                    f"Wetter für {icao} wird nur gecacht.")
                self._weather_metars[icao] = metar_data
                return

        # In Cache speichern
        self._weather_metars[icao] = metar_data
        self._weather_synced_icaos.add(icao)

        raw = metar_data.get("raw", metar_data.get("metar", ""))
        self.bridge.log_message.emit(
            f"🌦️ METAR {icao}: {raw[:100]}")

        # 2) Wettermodus erkennen
        # Kurze Wartezeit damit SimConnect Ambient-Werte laden kann
        time.sleep(1.0)
        mode = self._detect_weather_mode(metar_data)
        self.bridge.log_message.emit(
            f"🌦️ Wettermodus: {mode}")

        # 3) Wetter setzen oder Abweichung loggen
        if mode == "LIVE":
            self.bridge.log_message.emit(
                f"🌦️ Live-Wetter aktiv – keine Wettersetzung für {icao}")
            self._log_weather_deviation(icao, metar_data)

            # Trotzdem QNH setzen (Kohlsman)
            qnh_hpa = metar_data.get("qnh", metar_data.get("qnh_hpa", 0))
            if qnh_hpa and 900 < qnh_hpa < 1100:
                try:
                    if self._ae is None:
                        self._ae = AircraftEvents(self.sm)
                    evt_k = self._ae.find("KOHLSMAN_SET")
                    val = int(round(qnh_hpa * 16))
                    evt_k(val)
                    self.bridge.log_message.emit(
                        f"🌦️ QNH gesetzt: {icao} ({qnh_hpa} hPa)")
                except Exception:
                    pass

            sync_result = {
                "icao": icao, "mode": "LIVE",
                "action": "QNH_ONLY",
                "qnh": qnh_hpa,
            }
        else:
            # CUSTOM oder UNKNOWN → Wetter aus METAR setzen
            success = self._set_weather_from_metar(icao, metar_data)
            sync_result = {
                "icao": icao, "mode": mode,
                "action": "FULL_SET" if success else "FAILED",
                "qnh": metar_data.get("qnh", metar_data.get("qnh_hpa", 1013)),
                "wind_dir": metar_data.get("wind_dir", 0),
                "wind_speed": metar_data.get("wind_speed_kt", metar_data.get("wind_speed", 0)),
                "temperature": metar_data.get("temperature_c", metar_data.get("temperature", 15)),
                "visibility": metar_data.get("visibility_m", 9999),
                "clouds": metar_data.get("clouds", ""),
            }
            # Nach dem Setzen: Verifizierung
            time.sleep(2.0)
            self._check_weather_match(icao)

        self._last_weather_sync[icao] = sync_result
        self.bridge.weather_sync.emit(sync_result)

    def _handle_other_pilot(self, data: dict):
        cs = data.get("callsign", "")
        if cs == self.callsign:
            return  # eigene Daten ignorieren

        if data.get("event") == "disconnect":
            self.bridge.log_message.emit(f"← {cs} getrennt")
            self.bridge.other_pilot.emit(data)
            return

        self.bridge.log_message.emit(
            f"← {cs}  lat={data.get('lat', 0):.5f}  "
            f"lon={data.get('lon', 0):.5f}  "
            f"alt={data.get('alt', 0):.0f}"
        )
        self.bridge.other_pilot.emit(data)


# ---------------------------------------------------------------------------
# PyQt6 GUI (Schritt 6 – Flugplan-Eingabe)
# ---------------------------------------------------------------------------
class PilotWindow(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("NEXUS-ATC – Pilot-Client")
        self.setMinimumSize(540, 740)
        self.resize(600, 880)
        self.setStyleSheet("""
            * { margin: 0; padding: 0; }
            QWidget#central {
                background: #0d1117;
            }
            QWidget {
                background: #0d1117;
                color: #c9d1d9;
                font-family: 'Segoe UI', sans-serif;
                font-size: 13px;
            }
            QLineEdit {
                background: #161b22;
                border: 1px solid #30363d;
                color: #e6edf3;
                padding: 5px 8px;
                border-radius: 6px;
                font-family: Consolas, 'Courier New';
                font-size: 13px;
                selection-background-color: #1f6feb;
                min-height: 18px;
            }
            QLineEdit:focus {
                border-color: #58a6ff;
                background: #1c2333;
            }
            QLineEdit::placeholder {
                color: #484f58;
            }
            QPushButton {
                background: #21262d;
                color: #58a6ff;
                padding: 7px 14px;
                border: 1px solid #30363d;
                border-radius: 6px;
                font-weight: bold;
                font-size: 12px;
            }
            QPushButton:hover {
                background: #30363d;
                border-color: #58a6ff;
            }
            QPushButton:pressed {
                background: #1f6feb;
                color: #fff;
            }
            QPushButton:disabled {
                color: #484f58;
                border-color: #21262d;
                background: #161b22;
            }
            QGroupBox {
                background: transparent;
                border: none;
                margin: 0px;
                padding: 0px;
                padding-top: 28px;
            }
            QGroupBox::title {
                subcontrol-origin: padding;
                subcontrol-position: top left;
                left: 0px;
                top: 4px;
                color: #58a6ff;
                font-size: 11px;
                font-weight: bold;
                letter-spacing: 2px;
                padding: 0px;
                background: transparent;
            }
            QTextEdit {
                background: #161b22;
                color: #c8d6e5;
                font-family: Consolas;
                font-size: 11px;
                border: 1px solid #30363d;
                border-radius: 6px;
                padding: 6px;
            }
            QCheckBox {
                color: #c9d1d9;
                spacing: 6px;
                padding: 2px 0;
            }
            QCheckBox::indicator {
                width: 16px; height: 16px;
                border: 1px solid #30363d;
                border-radius: 3px;
                background: #161b22;
            }
            QCheckBox::indicator:checked {
                background: #1f6feb;
                border-color: #58a6ff;
            }
            QLabel {
                font-size: 13px;
                background: transparent;
                padding: 0px;
            }
            QScrollArea {
                border: none;
                background: transparent;
            }
            QScrollBar:vertical {
                background: #161b22;
                width: 8px;
                border-radius: 4px;
            }
            QScrollBar::handle:vertical {
                background: #30363d;
                border-radius: 4px;
                min-height: 20px;
            }
            QScrollBar::handle:vertical:hover {
                background: #484f58;
            }
            QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {
                height: 0px;
            }
        """)

        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.setSpacing(0)

        # Scrollable content area
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        scroll_content = QWidget()
        scroll_content.setObjectName("central")
        layout = QVBoxLayout(scroll_content)
        layout.setContentsMargins(20, 12, 20, 12)
        layout.setSpacing(4)
        scroll.setWidget(scroll_content)
        main_layout.addWidget(scroll)

        # --- Verbindungs-Gruppe ---
        conn_group = QGroupBox("VERBINDUNG")
        conn_form = QFormLayout(conn_group)
        conn_form.setContentsMargins(0, 0, 0, 8)
        conn_form.setSpacing(8)
        conn_form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        self.txt_server = QLineEdit(DEFAULT_SERVER)
        conn_form.addRow("Server:", self.txt_server)

        self.txt_pilot_name = QLineEdit()
        self.txt_pilot_name.setPlaceholderText("Dein Name")
        conn_form.addRow("Name:", self.txt_pilot_name)

        self.txt_callsign = QLineEdit(DEFAULT_CALLSIGN)
        conn_form.addRow("Callsign:", self.txt_callsign)

        layout.addWidget(conn_group)

        # --- Status-LEDs ---
        led_layout = QHBoxLayout()
        self.led_sim = StatusLED("SimConnect")
        self.led_server = StatusLED("Server")
        self.led_ptt = StatusLED("PTT")
        led_layout.addWidget(self.led_sim)
        led_layout.addWidget(self.led_server)
        led_layout.addWidget(self.led_ptt)
        led_layout.addStretch()
        layout.addLayout(led_layout)

        # --- Flugplan-Gruppe ---
        fp_group = QGroupBox("FLUGPLAN")
        fp_form = QFormLayout(fp_group)
        fp_form.setContentsMargins(0, 0, 0, 8)
        fp_form.setSpacing(8)
        fp_form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        self.txt_dep = QLineEdit()
        self.txt_dep.setPlaceholderText("z.B. EDDF")
        fp_form.addRow("DEP:", self.txt_dep)

        self.txt_arr = QLineEdit()
        self.txt_arr.setPlaceholderText("z.B. EDDM")
        fp_form.addRow("ARR:", self.txt_arr)

        self.txt_actype = QLineEdit()
        self.txt_actype.setPlaceholderText("z.B. A320")
        fp_form.addRow("Flugzeugtyp:", self.txt_actype)

        self.txt_cruise = QLineEdit()
        self.txt_cruise.setPlaceholderText("z.B. FL340")
        fp_form.addRow("Reiseflughöhe:", self.txt_cruise)

        self.txt_route = QLineEdit()
        self.txt_route.setPlaceholderText("z.B. MARUN Y163 ASLAD")
        fp_form.addRow("Route:", self.txt_route)

        # SimBrief Import Button
        btn_simbrief = QPushButton("✈ SimBrief importieren")
        btn_simbrief.clicked.connect(self._on_import_simbrief)
        fp_form.addRow(btn_simbrief)

        self.txt_simbrief_id = QLineEdit()
        self.txt_simbrief_id.setPlaceholderText("SimBrief Pilot ID oder Username")
        fp_form.addRow("SimBrief ID:", self.txt_simbrief_id)

        layout.addWidget(fp_group)

        # --- Voice (Mumble) ---
        voice_group = QGroupBox("VOICE · MUMBLE")
        voice_form = QFormLayout(voice_group)
        voice_form.setContentsMargins(0, 0, 0, 8)
        voice_form.setSpacing(8)
        voice_form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        self.txt_mumble_host = QLineEdit("localhost")
        self.txt_mumble_host.setPlaceholderText("z.B. mumble.example.com")
        voice_form.addRow("Mumble-Server:", self.txt_mumble_host)

        self.txt_mumble_port = QLineEdit("64738")
        voice_form.addRow("Port:", self.txt_mumble_port)

        # PTT-Tastatur-Auswahl (Combobox)
        from PyQt6.QtWidgets import QComboBox
        self.cmb_ptt_key = QComboBox()
        self.cmb_ptt_key.addItems([
            "Rechte Strg", "Linke Strg",
            "Rechte Shift", "Linke Shift",
            "Leertaste", "Caps Lock",
            "F1", "F2", "F3",
        ])
        # Aus Config laden
        saved_key = _load_voice_config().get("pilot_ptt_key", "Rechte Strg")
        idx = self.cmb_ptt_key.findText(saved_key)
        if idx >= 0:
            self.cmb_ptt_key.setCurrentIndex(idx)
        voice_form.addRow("⌨ PTT-Taste:", self.cmb_ptt_key)

        # Joystick-PTT
        joy_row = QHBoxLayout()
        self.lbl_joy_info = QLabel("Kein Joystick")
        self.lbl_joy_info.setStyleSheet("color: #888; font-size: 10px;")
        joy_row.addWidget(self.lbl_joy_info)
        self.btn_joy_learn = QPushButton("🎮 Joystick-PTT zuweisen")
        self.btn_joy_learn.setToolTip(
            "Drücke den gewünschten Joystick-Button innerhalb von 15 Sekunden")
        self.btn_joy_learn.clicked.connect(self._on_joy_learn)
        joy_row.addWidget(self.btn_joy_learn)
        voice_form.addRow("Joystick:", joy_row)

        # Joystick-Info beim Start aktualisieren
        self._joy_id = -1
        self._joy_button = -1
        self._load_joystick_config()

        self.chk_voice = QCheckBox("Voice aktivieren")
        voice_form.addRow(self.chk_voice)

        layout.addWidget(voice_group)

        # --- COM1 Frequenz-Anzeige ---
        com1_group = QGroupBox("COM1 FREQUENZ")
        com1_layout = QHBoxLayout(com1_group)
        com1_layout.setContentsMargins(4, 4, 4, 4)
        self.lbl_com1_freq = QLabel("---")
        self.lbl_com1_freq.setStyleSheet(
            "color: #c8d6e5; font-weight: bold; font-size: 18px; font-family: Consolas;")
        self.lbl_com1_freq.setAlignment(Qt.AlignmentFlag.AlignCenter)
        com1_layout.addWidget(self.lbl_com1_freq)
        self.lbl_com1_status = QLabel("")
        self.lbl_com1_status.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.lbl_com1_status.setVisible(False)
        com1_layout.addWidget(self.lbl_com1_status)
        layout.addWidget(com1_group)

        # --- ATC-Anweisungen (empfangen) ---
        atc_group = QGroupBox("ATC-ANWEISUNGEN")
        atc_layout = QFormLayout(atc_group)
        atc_layout.setContentsMargins(0, 0, 0, 8)
        atc_layout.setSpacing(5)
        atc_layout.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        self.lbl_assigned_sq = QLabel("---")
        self.lbl_assigned_sq.setStyleSheet("color: #f0883e; font-weight: bold; font-size: 14px; font-family: Consolas;")
        atc_layout.addRow("Assigned SQ:", self.lbl_assigned_sq)

        self.lbl_assigned_alt = QLabel("---")
        self.lbl_assigned_alt.setStyleSheet("color: #f0883e; font-weight: bold; font-size: 14px; font-family: Consolas;")
        atc_layout.addRow("Assigned ALT:", self.lbl_assigned_alt)

        self.lbl_cleared = QLabel("---")
        self.lbl_cleared.setStyleSheet("color: #f0883e; font-weight: bold; font-size: 14px; font-family: Consolas;")
        atc_layout.addRow("Status:", self.lbl_cleared)

        self.lbl_atc_from = QLabel("---")
        self.lbl_atc_from.setStyleSheet("color: #8b949e; font-size: 11px;")
        atc_layout.addRow("Von:", self.lbl_atc_from)

        # Kontakt-Frequenz (groß + rot blinkend)
        self.lbl_contact_freq = QLabel("")
        self.lbl_contact_freq.setVisible(False)
        self.lbl_contact_freq.setAlignment(Qt.AlignmentFlag.AlignCenter)
        atc_layout.addRow(self.lbl_contact_freq)

        # ATC-Nachricht
        self.lbl_atc_message = QLabel("")
        self.lbl_atc_message.setVisible(False)
        self.lbl_atc_message.setWordWrap(True)
        atc_layout.addRow(self.lbl_atc_message)

        # --- Auto-Tune Button (CONTACT REQUEST) ---
        self.btn_tune_contact = QPushButton("")
        self.btn_tune_contact.setVisible(False)
        self.btn_tune_contact.setMinimumHeight(36)
        self.btn_tune_contact.setCursor(Qt.CursorShape.PointingHandCursor)
        self.btn_tune_contact.clicked.connect(self._on_tune_contact_clicked)
        atc_layout.addRow(self.btn_tune_contact)

        layout.addWidget(atc_group)

        # --- CPDLC / DCDU (Airbus-Style) ---
        cpdlc_group = QGroupBox("CPDLC / DCDU")
        cpdlc_group.setStyleSheet("""
            QGroupBox { color: #58a6ff; border: 2px solid #0a2a60;
                        border-radius: 4px; padding: 10px; margin-top: 8px;
                        background: #000000; font-family: Consolas; font-weight: bold; }
            QGroupBox::title { subcontrol-origin: margin; left: 10px;
                               color: #58a6ff; }
        """)
        cpdlc_layout = QVBoxLayout(cpdlc_group)
        cpdlc_layout.setContentsMargins(4, 4, 4, 4)
        cpdlc_layout.setSpacing(4)

        # DCDU-Nachrichtenanzeige (Airbus: blau/grün auf schwarz)
        self.txt_cpdlc_dcdu = QTextEdit()
        self.txt_cpdlc_dcdu.setReadOnly(True)
        self.txt_cpdlc_dcdu.setMaximumHeight(100)
        self.txt_cpdlc_dcdu.setStyleSheet(
            "QTextEdit { background: #000000; color: #58a6ff; border: 1px solid #0a2a60;"
            " font-family: Consolas; font-size: 12px; font-weight: bold;"
            " border-radius: 2px; padding: 4px; }")
        self.txt_cpdlc_dcdu.setPlaceholderText("— NO ACTIVE CPDLC —")
        cpdlc_layout.addWidget(self.txt_cpdlc_dcdu)

        # WILCO / UNABLE / STANDBY Buttons
        cpdlc_btn_row = QHBoxLayout()
        cpdlc_btn_row.setSpacing(4)

        self.btn_cpdlc_wilco = QPushButton("WILCO")
        self.btn_cpdlc_wilco.setEnabled(False)
        self.btn_cpdlc_wilco.setStyleSheet(
            "QPushButton { background: #0a2a60; color: #58a6ff; padding: 8px 16px;"
            " border: 2px solid #3a7bd5; border-radius: 4px; font-weight: bold;"
            " font-family: Consolas; font-size: 13px; }"
            " QPushButton:hover { background: #0d3a80; }"
            " QPushButton:disabled { background: #111; color: #333; border-color: #222; }")
        self.btn_cpdlc_wilco.clicked.connect(lambda: self._cpdlc_respond("WILCO"))
        cpdlc_btn_row.addWidget(self.btn_cpdlc_wilco)

        self.btn_cpdlc_unable = QPushButton("UNABLE")
        self.btn_cpdlc_unable.setEnabled(False)
        self.btn_cpdlc_unable.setStyleSheet(
            "QPushButton { background: #381000; color: #ff6040; padding: 8px 16px;"
            " border: 2px solid #cc4400; border-radius: 4px; font-weight: bold;"
            " font-family: Consolas; font-size: 13px; }"
            " QPushButton:hover { background: #552200; }"
            " QPushButton:disabled { background: #111; color: #333; border-color: #222; }")
        self.btn_cpdlc_unable.clicked.connect(lambda: self._cpdlc_respond("UNABLE"))
        cpdlc_btn_row.addWidget(self.btn_cpdlc_unable)

        self.btn_cpdlc_standby = QPushButton("STANDBY")
        self.btn_cpdlc_standby.setEnabled(False)
        self.btn_cpdlc_standby.setStyleSheet(
            "QPushButton { background: #002038; color: #00b4ff; padding: 8px 16px;"
            " border: 2px solid #0080cc; border-radius: 4px; font-weight: bold;"
            " font-family: Consolas; font-size: 13px; }"
            " QPushButton:hover { background: #003355; }"
            " QPushButton:disabled { background: #111; color: #333; border-color: #222; }")
        self.btn_cpdlc_standby.clicked.connect(lambda: self._cpdlc_respond("STANDBY"))
        cpdlc_btn_row.addWidget(self.btn_cpdlc_standby)

        cpdlc_layout.addLayout(cpdlc_btn_row)

        # Virtual Printer / Paper Strip
        self.txt_cpdlc_printer = QTextEdit()
        self.txt_cpdlc_printer.setReadOnly(True)
        self.txt_cpdlc_printer.setMaximumHeight(80)
        self.txt_cpdlc_printer.setStyleSheet(
            "QTextEdit { background: #f5f0e0; color: #1a1a1a; border: 1px solid #c0b090;"
            " font-family: 'Courier New', Consolas; font-size: 10px;"
            " border-radius: 2px; padding: 3px; }")
        self.txt_cpdlc_printer.setPlaceholderText("--- ACARS PRINTER ---")
        cpdlc_layout.addWidget(self.txt_cpdlc_printer)

        layout.addWidget(cpdlc_group)

        # --- Frequenz-Chat (Nachrichten senden/empfangen) ---
        chat_group = QGroupBox("FUNK-NACHRICHTEN")
        chat_layout = QVBoxLayout(chat_group)
        chat_layout.setContentsMargins(4, 4, 4, 4)
        chat_layout.setSpacing(4)

        self.txt_chat_log = QTextEdit()
        self.txt_chat_log.setReadOnly(True)
        self.txt_chat_log.setMaximumHeight(100)
        self.txt_chat_log.setStyleSheet(
            "QTextEdit { background: #0d1117; color: #c8d6e5; border: 1px solid #30363d;"
            " font-family: Consolas; font-size: 11px; border-radius: 4px; }")
        chat_layout.addWidget(self.txt_chat_log)

        msg_row = QHBoxLayout()
        msg_row.setSpacing(4)
        self.txt_chat_input = QLineEdit()
        self.txt_chat_input.setPlaceholderText("Nachricht auf Frequenz senden...")
        self.txt_chat_input.setStyleSheet(
            "QLineEdit { background: #161b22; color: #c8d6e5; border: 1px solid #30363d;"
            " padding: 6px; font-size: 12px; border-radius: 4px; }"
            " QLineEdit:focus { border-color: #58a6ff; }")
        self.txt_chat_input.returnPressed.connect(self._send_freq_message)
        msg_row.addWidget(self.txt_chat_input, 1)

        self.btn_chat_send = QPushButton("Senden")
        self.btn_chat_send.setStyleSheet(
            "QPushButton { background: #21262d; color: #58a6ff; padding: 6px 12px;"
            " border: 1px solid #30363d; border-radius: 4px; font-weight: bold; }"
            " QPushButton:hover { background: #30363d; }")
        self.btn_chat_send.clicked.connect(self._send_freq_message)
        msg_row.addWidget(self.btn_chat_send)

        chat_layout.addLayout(msg_row)
        layout.addWidget(chat_group)

        # --- Active ATC Liste (klickbar – setzt COM1-Frequenz) ---
        atc_list_group = QGroupBox("ACTIVE ATC  ·  klicken → Frequenz")
        atc_main_layout = QVBoxLayout(atc_list_group)
        atc_main_layout.setContentsMargins(0, 0, 0, 8)
        atc_main_layout.setSpacing(4)

        # Scroll-Bereich für ATC-Buttons
        self._atc_scroll = QScrollArea()
        self._atc_scroll.setWidgetResizable(True)
        self._atc_scroll.setMaximumHeight(130)
        self._atc_scroll.setStyleSheet(
            "QScrollArea { border: none; background: transparent; }")
        self._atc_btn_container = QWidget()
        self._atc_btn_layout = QVBoxLayout(self._atc_btn_container)
        self._atc_btn_layout.setContentsMargins(2, 2, 2, 2)
        self._atc_btn_layout.setSpacing(2)
        self._atc_btn_layout.addStretch()
        self._atc_scroll.setWidget(self._atc_btn_container)
        atc_main_layout.addWidget(self._atc_scroll)

        # Fallback-Textfeld (für Rückwärtskompatibilität)
        self.txt_atc_list = QTextEdit()
        self.txt_atc_list.setReadOnly(True)
        self.txt_atc_list.setMaximumHeight(0)
        self.txt_atc_list.setVisible(False)

        layout.addWidget(atc_list_group)

        # --- Airport Info / QNH ---
        info_group = QGroupBox("AIRPORT INFO · QNH")
        info_form = QFormLayout(info_group)
        info_form.setContentsMargins(0, 0, 0, 8)
        info_form.setSpacing(8)
        info_form.setLabelAlignment(Qt.AlignmentFlag.AlignRight)

        self.lbl_nearest_apt = QLabel("---")
        self.lbl_nearest_apt.setStyleSheet("color: #79c0ff; font-weight: bold; font-size: 13px; font-family: Consolas;")
        info_form.addRow("Nächster:", self.lbl_nearest_apt)

        self.lbl_metar = QLabel("---")
        self.lbl_metar.setWordWrap(True)
        self.lbl_metar.setStyleSheet("color: #8b949e; font-family: Consolas; font-size: 11px;")
        info_form.addRow("METAR:", self.lbl_metar)

        self.lbl_qnh_warning = QLabel("")
        self.lbl_qnh_warning.setVisible(False)
        self.lbl_qnh_warning.setAlignment(Qt.AlignmentFlag.AlignCenter)
        info_form.addRow(self.lbl_qnh_warning)

        self.lbl_apt_freqs = QLabel("---")
        self.lbl_apt_freqs.setWordWrap(True)
        self.lbl_apt_freqs.setStyleSheet("color: #58a6ff; font-family: Consolas; font-size: 11px;")
        info_form.addRow("Frequenzen:", self.lbl_apt_freqs)

        layout.addWidget(info_group)

        # --- TCAS / Nearby Traffic ---
        tcas_group = QGroupBox("TCAS · NEARBY TRAFFIC")
        tcas_group.setStyleSheet(
            "QGroupBox { color: #ff6060; font-weight: bold; font-family: Consolas;"
            " border: 1px solid #30363d; border-radius: 4px; padding-top: 14px; }"
            " QGroupBox::title { subcontrol-origin: margin; left: 8px; padding: 0 3px;}")
        tcas_layout = QVBoxLayout(tcas_group)
        tcas_layout.setContentsMargins(4, 4, 4, 4)
        tcas_layout.setSpacing(2)

        self.txt_tcas = QTextEdit()
        self.txt_tcas.setReadOnly(True)
        self.txt_tcas.setMaximumHeight(90)
        self.txt_tcas.setStyleSheet(
            "QTextEdit { background: #0d1117; color: #c8d6e5; border: 1px solid #30363d;"
            " font-family: Consolas; font-size: 10px; border-radius: 4px; }")
        self.txt_tcas.setPlaceholderText("No nearby traffic")
        tcas_layout.addWidget(self.txt_tcas)

        layout.addWidget(tcas_group)

        # Traffic-Daten Speicher
        self._nearby_traffic: dict[str, dict] = {}  # callsign → {lat, lon, alt, hdg, gs}
        self._model_warnings: list[str] = []           # Fallback-Warnungen für InGame Panel

        # --- Buttons ---
        btn_layout = QHBoxLayout()
        btn_layout.setSpacing(10)
        btn_layout.setContentsMargins(0, 12, 0, 4)
        self.btn_connect = QPushButton("✈  Verbinden & Senden")
        self.btn_connect.setMinimumHeight(42)
        self.btn_connect.setStyleSheet(
            "QPushButton { background: #1a5bb5; color: #fff; border: none;"
            " padding: 10px 20px; font-size: 14px; font-weight: bold; border-radius: 6px; }"
            " QPushButton:hover { background: #2070d0; }"
            " QPushButton:disabled { background: #161b22; color: #484f58; }")
        self.btn_connect.clicked.connect(self._on_connect)
        btn_layout.addWidget(self.btn_connect, 3)

        self.btn_disconnect = QPushButton("■  Stopp")
        self.btn_disconnect.setMinimumHeight(42)
        self.btn_disconnect.setStyleSheet(
            "QPushButton { background: #da3633; color: #fff; border: none;"
            " padding: 10px 20px; font-size: 14px; font-weight: bold; border-radius: 6px; }"
            " QPushButton:hover { background: #f85149; }"
            " QPushButton:disabled { background: #161b22; color: #484f58; }")
        self.btn_disconnect.setEnabled(False)
        self.btn_disconnect.clicked.connect(self._on_disconnect)
        btn_layout.addWidget(self.btn_disconnect, 1)

        layout.addLayout(btn_layout)

        # --- Status ---
        self.lbl_status = QLabel("Bereit")
        self.lbl_status.setStyleSheet("color: #8b949e; padding: 6px 0px; font-size: 11px;")
        layout.addWidget(self.lbl_status)

        # --- Signal-Qualität ---
        self.lbl_signal = QLabel("📡 --")
        self.lbl_signal.setStyleSheet("color: #8b949e; padding: 2px 0px; font-size: 11px;")
        layout.addWidget(self.lbl_signal)

        # --- TSAT / Event-Slot ---
        self.lbl_tsat = QLabel("")
        self.lbl_tsat.setStyleSheet(
            "color: #d29922; padding: 4px 8px; font-size: 12px; font-weight: bold; "
            "font-family: Consolas; background: rgba(210,153,34,0.1); "
            "border: 1px solid rgba(210,153,34,0.3); border-radius: 6px;")
        self.lbl_tsat.setVisible(False)
        layout.addWidget(self.lbl_tsat)

        # --- Log ---
        self.log = QTextEdit()
        self.log.setReadOnly(True)
        self.log.setMinimumHeight(80)
        self.log.setMaximumHeight(120)
        layout.addWidget(self.log)

        layout.addStretch()

        # --- Internes ---
        self.bridge = SignalBridge()
        self.bridge.log_message.connect(self._append_log)
        self.bridge.connection_status.connect(self._set_status)
        self.bridge.other_pilot.connect(self._on_other_pilot)
        self.bridge.sim_status.connect(
            lambda on: self.led_sim.set_active(on, QColor(0, 180, 255)))
        self.bridge.server_status.connect(
            lambda on: self.led_server.set_active(on, QColor(0, 150, 255)))
        self.bridge.ptt_status.connect(
            lambda on: self.led_ptt.set_active(on, QColor(255, 140, 0)))
        self.bridge.atc_instruction.connect(self._on_atc_instruction)
        self.bridge.atc_online.connect(self._on_atc_online)
        self.bridge.atc_offline.connect(self._on_atc_offline)
        self.bridge.current_squawk.connect(self._on_current_squawk)
        self.bridge.current_com1.connect(self._on_current_com1)
        self.bridge.freq_message.connect(self._on_freq_message)
        self.bridge.voice_command.connect(self._on_voice_command)
        self.bridge.cpdlc_message.connect(self._on_cpdlc_message)
        self.bridge.signal_quality.connect(self._on_signal_quality)
        self.bridge.radio_blocked.connect(self._on_radio_blocked)
        self.bridge.stca_alert.connect(self._on_stca_alert)
        self.bridge.slot_assigned.connect(self._on_slot_assigned)
        self.bridge.contact_request.connect(self._on_contact_request)
        self.bridge.weather_sync.connect(self._on_weather_sync)
        self.bridge.simbrief_data.connect(self._on_simbrief_data)
        self.bridge.com_volume.connect(self._on_com_volume)
        self.bridge.model_warning.connect(self._on_model_warning)

        self.net_thread: Optional[NetworkThread] = None
        self.sm: Optional[SimConnect] = None
        self.ai_mgr: Optional[AIManager] = None
        self.voice_mgr: Optional[VoiceManager] = None

        # ATC instruction state
        self._assigned_squawk: int = 0
        self._assigned_alt: str = ""
        self._cleared: bool = False
        self._atc_from: str = ""
        self._contact_freq: float = 0.0
        self._contact_station: str = ""       # Station-Name für CONTACT REQUEST
        self._contact_blink_on: bool = True     # Blink-State für TUNE-Button
        self._atc_blink_on: bool = True
        self._active_atc: dict[str, dict] = {}
        self._current_squawk: int = 2000  # Aktueller Squawk aus MSFS
        self._current_com1_freq: float = 0.0  # Aktuelle COM1-Frequenz aus MSFS
        self._com1_volume: int = 100  # COM1-Empfangslautstärke (0-100)
        self._com2_volume: int = 100  # COM2-Empfangslautstärke (0-100)

        # CPDLC state
        self._cpdlc_current_msg: Optional[dict] = None
        self._cpdlc_chime_path: str = ""
        self._create_cpdlc_chime()

        # Blink-Timer für ATC-Anweisungen (500ms)
        self._atc_blink_timer = QTimer(self)
        self._atc_blink_timer.timeout.connect(self._atc_blink_tick)
        self._atc_blink_timer.start(500)

        # Airport-Info State
        self._last_airport_info: Optional[dict] = None
        self._nearest_airport_data: Optional[dict] = None
        self._last_nearest_check: float = 0.0
        self._last_pilot_lat: float = 0.0
        self._last_pilot_lon: float = 0.0
        self._last_com1_freq: float = 0.0

        # Nachrichten-Puffer für InGame-Panel API
        self._api_freq_messages: list[dict] = []
        self._api_weather_cache: dict = {}
        self._api_wake_msg: Optional[dict] = None

        # Info-Update Timer (alle 2s Airport-Info/QNH prüfen)
        self._info_timer = QTimer(self)
        self._info_timer.timeout.connect(self._update_airport_panel)
        self._info_timer.start(2000)

        # Lokale HTTP-API für das MSFS InGame-Panel starten (Port 9001)
        self._api_server = InGamePanelAPI(self)
        self._api_server.start()

    # -- Callbacks ----------------------------------------------------------
    def _on_connect(self):
        # Name Pflichtfeld
        pilot_name = self.txt_pilot_name.text().strip()
        if not pilot_name:
            QMessageBox.warning(
                self, "Name fehlt",
                "Bitte gib deinen Namen ein bevor du dich verbindest."
            )
            return

        # SimConnect verbinden
        try:
            self.sm = SimConnect()
        except Exception as e:
            QMessageBox.critical(
                self, "MSFS nicht gefunden",
                f"SimConnect-Verbindung fehlgeschlagen:\n{e}\n\n"
                "Stelle sicher, dass der MSFS läuft."
            )
            return

        ar = AircraftRequests(self.sm, _time=0)

        # AI-Modell für Multiplayer proben (eingebaute MSFS 2024 Titel)
        import model_matcher as _mm
        self.bridge.log_message.emit("  AI-Modell wird gesucht …")
        _mm.AI_MODEL_DEFAULT = probe_ai_model(self.sm)
        self.bridge.log_message.emit(f"  AI-Modell: {_mm.AI_MODEL_DEFAULT!r}")

        # Model-Matching-System laden
        self._model_matcher = ModelMatcher("models.json")
        self.bridge.log_message.emit(
            f"  Model-DB geladen: {self._model_matcher.stats()}"
        )

        self.ai_mgr = AIManager(
            self.sm, self.bridge, AI_MODEL_DEFAULT,
            model_matcher=self._model_matcher,
        )
        self.bridge.sim_status.emit(True)

        # Voice starten (optional)
        if self.chk_voice.isChecked() and _HAS_PYMUMBLE:
            mumble_host = self.txt_mumble_host.text().strip() or "localhost"
            mumble_port = int(self.txt_mumble_port.text().strip() or "64738")
            ptt_key = self.cmb_ptt_key.currentText() or "Rechte Strg"
            # PTT-Taste in Config speichern
            _save_voice_config({"pilot_ptt_key": ptt_key})
            cs_voice = self.txt_callsign.text().strip() or DEFAULT_CALLSIGN
            self.voice_mgr = VoiceManager(
                mumble_host, mumble_port, cs_voice,
                ptt_key_name=ptt_key, bridge=self.bridge,
                joystick_id=self._joy_id,
                joystick_button=self._joy_button,
            )
            self.voice_mgr.start()

        callsign = self.txt_callsign.text().strip() or DEFAULT_CALLSIGN
        server = self.txt_server.text().strip() or DEFAULT_SERVER
        flight_plan = {
            "dep": self.txt_dep.text().strip().upper(),
            "arr": self.txt_arr.text().strip().upper(),
            "actype": self.txt_actype.text().strip(),
            "cruise_alt": self.txt_cruise.text().strip().upper(),
            "route": self.txt_route.text().strip().upper(),
            "pilot_name": pilot_name,
        }

        self.net_thread = NetworkThread(
            server, callsign, flight_plan, self.sm, ar, self.bridge
        )
        self.net_thread.voice_mgr = self.voice_mgr  # für ACP Volume Sync
        self.net_thread.start()

        self.btn_connect.setEnabled(False)
        self.btn_disconnect.setEnabled(True)
        self._set_inputs_enabled(False)

        # ── Automatische Wetter-Synchronisation beim Login ────────────
        # Wetter nur für aktuelle Position setzen (nicht für DEP/ARR)
        def _weather_sync_login():
            # Warte bis NetworkThread verbunden ist
            time.sleep(3)
            try:
                pilot_lat, pilot_lon = self.net_thread._get_current_position()
                if pilot_lat != 0 and pilot_lon != 0:
                    nearest = self.net_thread._get_nearest_icao_from_server(
                        pilot_lat, pilot_lon)
                    if nearest:
                        self.bridge.log_message.emit(
                            f"🌦️ Login: Wetter-Sync für aktuelle Position: {nearest}")
                        self.net_thread.sync_weather_for_airport(nearest, force=True)
                    else:
                        dep_icao = flight_plan.get("dep", "").strip().upper()
                        if dep_icao:
                            self.net_thread.sync_weather_for_airport(dep_icao)
                else:
                    dep_icao = flight_plan.get("dep", "").strip().upper()
                    if dep_icao:
                        self.net_thread.sync_weather_for_airport(dep_icao)
            except Exception as e:
                self.bridge.log_message.emit(f"🌦️ Wetter-Sync Fehler: {e}")
        t_wx = threading.Thread(target=_weather_sync_login, daemon=True)
        t_wx.start()

    # -- Joystick Config & Lern-Modus ----------------------------------------
    def _load_joystick_config(self):
        """Lädt Joystick-PTT aus voice_config.json."""
        cfg = _load_voice_config()
        self._joy_id = cfg.get("joystick_id", -1)
        self._joy_button = cfg.get("joystick_button_index", -1)
        self._update_joy_label()

    def _update_joy_label(self):
        """Aktualisiert das Joystick-Info-Label."""
        if self._joy_id >= 0 and self._joy_button >= 0:
            # Joystick-Name ermitteln
            name = f"Joystick {self._joy_id}"
            if _HAS_PYGAME:
                try:
                    for info in list_joysticks():
                        if info["id"] == self._joy_id:
                            name = info["name"]
                            break
                except Exception:
                    pass
            self.lbl_joy_info.setText(f"✅ {name} · Button {self._joy_button}")
            self.lbl_joy_info.setStyleSheet("color: #58a6ff; font-size: 10px;")
        else:
            n = pygame.joystick.get_count() if _HAS_PYGAME else 0
            if n > 0:
                self.lbl_joy_info.setText(f"{n} Gerät(e) erkannt")
                self.lbl_joy_info.setStyleSheet("color: #8ba4b8; font-size: 10px;")
            else:
                self.lbl_joy_info.setText("Kein Joystick")
                self.lbl_joy_info.setStyleSheet("color: #888; font-size: 10px;")

    def _on_joy_learn(self):
        """Startet den Joystick-Lern-Modus: Button drücken zum Zuweisen."""
        if not _HAS_PYGAME:
            QMessageBox.warning(self, "Fehler",
                                "pygame ist nicht installiert.\n"
                                "Installiere mit: pip install pygame")
            return
        joys = list_joysticks()
        if not joys:
            QMessageBox.warning(self, "Fehler",
                                "Kein Joystick/Gamepad erkannt.\n"
                                "Bitte Gerät anschließen und neu starten.")
            return

        # Info-Dialog
        self.lbl_joy_info.setText("⏳ Drücke jetzt den PTT-Button …")
        self.lbl_joy_info.setStyleSheet("color: #f0c040; font-size: 10px; font-weight: bold;")
        self.btn_joy_learn.setEnabled(False)
        QApplication.processEvents()

        # Lern-Thread starten (blockiert nicht die GUI)
        def _learn_worker():
            result = joystick_learn_button(timeout=15.0)
            # Zurück im Qt-Thread
            QTimer.singleShot(0, lambda: self._on_joy_learned(result))

        t = threading.Thread(target=_learn_worker, daemon=True)
        t.start()

    def _on_joy_learned(self, result):
        """Callback nach dem Lern-Modus."""
        self.btn_joy_learn.setEnabled(True)
        if result is None:
            self.lbl_joy_info.setText("❌ Timeout – kein Button erkannt")
            self.lbl_joy_info.setStyleSheet("color: #f85149; font-size: 10px;")
            return
        joy_id, btn = result
        self._joy_id = joy_id
        self._joy_button = btn
        # In Config speichern
        _save_voice_config({
            "joystick_id": joy_id,
            "joystick_button_index": btn,
        })
        self._update_joy_label()
        self.bridge.log_message.emit(
            f"🎮 Joystick-PTT zugewiesen: ID {joy_id}, Button {btn}")

    def _on_disconnect(self):
        if self.net_thread:
            self.net_thread.stop()
            self.net_thread = None
        if self.ai_mgr:
            self.ai_mgr.stop()
            self.ai_mgr = None
        if self.voice_mgr:
            self.voice_mgr.stop()
            self.voice_mgr = None
        if self.sm:
            try:
                self.sm.exit()
            except Exception:
                pass
            self.sm = None
        self.ai_mgr = None

        self.led_sim.set_active(False)
        self.led_server.set_active(False)
        self.led_ptt.set_active(False)

        self.btn_connect.setEnabled(True)
        self.btn_disconnect.setEnabled(False)
        self._set_inputs_enabled(True)
        self._set_status("Getrennt")

    def _set_inputs_enabled(self, enabled: bool):
        self.txt_server.setEnabled(enabled)
        self.txt_callsign.setEnabled(enabled)
        self.txt_dep.setEnabled(enabled)
        self.txt_arr.setEnabled(enabled)
        self.txt_actype.setEnabled(enabled)
        self.txt_cruise.setEnabled(enabled)
        self.txt_route.setEnabled(enabled)
        self.txt_mumble_host.setEnabled(enabled)
        self.txt_mumble_port.setEnabled(enabled)
        self.cmb_ptt_key.setEnabled(enabled)
        self.chk_voice.setEnabled(enabled)

    def _append_log(self, msg: str):
        self.log.append(msg)
        # Auto-scroll
        scrollbar = self.log.verticalScrollBar()
        scrollbar.setValue(scrollbar.maximum())

    def _set_status(self, text: str):
        self.lbl_status.setText(text)

    def _on_other_pilot(self, data: dict):
        """AI-Injection: Fremdes Flugzeug im MSFS erstellen/aktualisieren + TCAS."""
        cs = data.get("callsign", "")

        if data.get("event") == "disconnect":
            self._nearby_traffic.pop(cs, None)
            if self.ai_mgr:
                self.ai_mgr.remove(cs)
            self._update_tcas_display()
            return

        if not self.ai_mgr:
            return

        lat = data.get("lat", 0)
        lon = data.get("lon", 0)
        alt = data.get("alt", 0)
        hdg = data.get("heading", 0)
        gs  = data.get("groundspeed", 0)
        actype = data.get("actype", "")
        model = MODEL_MATCH.get(actype, AI_MODEL_DEFAULT)
        self.ai_mgr.update_or_create(cs, lat, lon, alt, hdg, gs, model, actype)

        # Traffic-Daten für TCAS speichern
        self._nearby_traffic[cs] = {
            "lat": lat, "lon": lon, "alt": alt,
            "heading": hdg, "groundspeed": gs,
        }
        self._update_tcas_display()

    def _update_tcas_display(self):
        """Aktualisiert die TCAS Nearby-Traffic-Anzeige."""
        if not hasattr(self, "txt_tcas"):
            return
        if not self._nearby_traffic:
            self.txt_tcas.setPlainText("No nearby traffic")
            return

        # Eigene Position (letzte bekannte)
        own_lat = getattr(self, "_own_lat", 0)
        own_lon = getattr(self, "_own_lon", 0)
        own_alt = getattr(self, "_own_alt", 0)

        lines = []
        for cs, t in sorted(self._nearby_traffic.items()):
            alt_fl = int(t["alt"] / 100)
            gs = int(t["groundspeed"])
            # Ungefähre Entfernung
            dlat = t["lat"] - own_lat
            dlon = t["lon"] - own_lon
            import math as _m
            cos_lat = _m.cos(_m.radians(own_lat)) if own_lat else 1
            nm = _m.sqrt((dlat * 60) ** 2 + (dlon * 60 * cos_lat) ** 2)
            alt_diff = int(t["alt"] - own_alt)
            arrow = "↑" if alt_diff > 200 else ("↓" if alt_diff < -200 else "=")
            warn = ""
            if nm < 5 and abs(alt_diff) < 1000:
                warn = " ⚠ STCA"
            lines.append(
                f"{cs:<10} FL{alt_fl:03d} {arrow}{abs(alt_diff):+5d}ft  "
                f"{nm:5.1f}nm  GS{gs}{warn}"
            )
        self.txt_tcas.setPlainText("\n".join(lines))

    # -- ATC-Anweisungen -----------------------------------------------------
    def _on_current_squawk(self, sq: int):
        """Aktualisiert den aktuellen Squawk aus dem MSFS."""
        old = self._current_squawk
        self._current_squawk = sq
        # Wenn sich der Squawk geändert hat, Labels aktualisieren
        if old != sq:
            self._update_atc_labels()

    def _on_current_com1(self, freq: float):
        """Aktualisiert die aktuelle COM1-Frequenz aus MSFS."""
        self._current_com1_freq = freq
        # COM1 Label aktualisieren
        if freq > 0:
            self.lbl_com1_freq.setText(f"{freq:.3f} MHz")
        else:
            self.lbl_com1_freq.setText("---")

        # Matching mit zugewiesener Contact-Frequenz
        if self._contact_freq > 0 and freq > 0:
            if abs(freq - self._contact_freq) < 0.01:
                self.lbl_com1_status.setText("TUNED")
                self.lbl_com1_status.setStyleSheet(
                    "color: #58a6ff; font-weight: bold; font-size: 14px;"
                    "background: #0a2040; border: 1px solid #58a6ff;"
                    "border-radius: 4px; padding: 3px 8px;")
                self.lbl_com1_status.setVisible(True)
                # Contact-Freq Label + TUNE-Button verstecken wenn getuned
                self.lbl_contact_freq.setVisible(False)
                self.btn_tune_contact.setVisible(False)
            else:
                self.lbl_com1_status.setText("TUNE TO " + f"{self._contact_freq:.3f}")
                self.lbl_com1_status.setStyleSheet(
                    "color: #ff4444; font-weight: bold; font-size: 12px;"
                    "background: #330000; border: 1px solid #ff4444;"
                    "border-radius: 4px; padding: 3px 8px;")
                self.lbl_com1_status.setVisible(True)
        else:
            self.lbl_com1_status.setVisible(False)

        # Voice: Auto-Switch Mumble-Channel auf COM1-Frequenz
        if self.voice_mgr and freq > 0:
            self.voice_mgr.sync_frequency(freq)

    def _on_com_volume(self, com1_vol: int, com2_vol: int):
        """Aktualisiert die COM-Empfangslautstärke aus MSFS."""
        self._com1_volume = com1_vol
        self._com2_volume = com2_vol

    def _on_model_warning(self, warning: str):
        """Speichert Fallback-Warnung für InGame Panel Anzeige."""
        self._model_warnings.append(warning)
        # Max 20 Warnungen behalten
        if len(self._model_warnings) > 20:
            self._model_warnings = self._model_warnings[-20:]

    def _show_sim_notification(self, title: str, text: str):
        """Zeigt Text direkt im MSFS-Simulator an (oben im Bild)."""
        msg = f"[{title}] {text}"
        # SimConnect sendText – BUG-FIX: Python-SimConnect berechnet cbData falsch
        # (sizeof(c_double)*len statt len), daher direkt via ctypes aufrufen
        if self.sm is not None:
            try:
                import ctypes
                from ctypes import c_void_p, cast
                from SimConnect.Enum import SIMCONNECT_TEXT_TYPE
                text_type = SIMCONNECT_TEXT_TYPE.SIMCONNECT_TEXT_TYPE_SCROLL_WHITE
                raw = msg.encode('utf-8')
                buf = (ctypes.c_char * len(raw))(*raw)
                ptr = cast(buf, c_void_p)
                self.sm.dll.Text(
                    self.sm.hSimConnect,
                    text_type,
                    10,  # Sekunden sichtbar
                    0,   # EventID
                    len(raw),  # cbData – KORREKTE Größe!
                    ptr
                )
                self.bridge.log_message.emit(f"📺 Sim-Text: {msg[:60]}")
            except Exception as e:
                self.bridge.log_message.emit(f"⚠ Sim-Text Fehler: {e}")
        # Versuch 2: Windows Toast als Fallback
        try:
            import subprocess
            t_safe = title.replace("'", "''").replace('"', '')
            x_safe = text.replace("'", "''").replace('"', '')
            ps_cmd = (
                f"[Windows.UI.Notifications.ToastNotificationManager, Windows.UI.Notifications, "
                f"ContentType = WindowsRuntime] > $null; "
                f"$xml = [Windows.UI.Notifications.ToastNotificationManager]::GetTemplateContent(1); "
                f"$texts = $xml.GetElementsByTagName('text'); "
                f"$texts[0].AppendChild($xml.CreateTextNode('{t_safe}')) > $null; "
                f"$texts[1].AppendChild($xml.CreateTextNode('{x_safe}')) > $null; "
                f"$notifier = [Windows.UI.Notifications.ToastNotificationManager]::CreateToastNotifier('NEXUS-ATC'); "
                f"$notifier.Show([Windows.UI.Notifications.ToastNotification]::new($xml))"
            )
            subprocess.Popen(
                ['powershell', '-WindowStyle', 'Hidden', '-Command', ps_cmd],
                creationflags=0x08000000
            )
        except Exception:
            pass

    def _on_atc_instruction(self, data: dict):
        """Empfängt ATC_UPDATE vom Server und zeigt sie an."""
        if "assigned_squawk" in data:
            self._assigned_squawk = data["assigned_squawk"]
        if "assigned_alt" in data:
            self._assigned_alt = data["assigned_alt"]
        if "cleared" in data:
            self._cleared = data["cleared"]
        self._atc_from = data.get("from", "")

        # Contact-Frequenz prüfen
        contact_freq = data.get("contact_freq", 0)
        if contact_freq and contact_freq > 0:
            self._contact_freq = contact_freq
            self.lbl_contact_freq.setText(f"📻  {contact_freq:.3f} MHz")
            self.lbl_contact_freq.setStyleSheet(
                "color: #ff4444; font-weight: bold; font-size: 15px;"
                "background: #330000; border: 2px solid #ff4444;"
                "border-radius: 4px; padding: 6px;")
            self.lbl_contact_freq.setVisible(True)

        # Textnachricht prüfen
        message = data.get("message", "")
        if message:
            self.lbl_atc_message.setText(f"✉  {message}")
            self.lbl_atc_message.setStyleSheet(
                "color: #ffaa00; font-weight: bold; font-size: 12px;"
                "background: #332200; border: 1px solid #ffaa00;"
                "border-radius: 4px; padding: 4px;")
            self.lbl_atc_message.setVisible(True)

        self._update_atc_labels()

        # -- Windows Notification (sichtbar über MSFS) --
        notif_parts = []
        if "assigned_squawk" in data:
            notif_parts.append(f"Squawk: {data['assigned_squawk']:04d}")
        if "assigned_alt" in data:
            notif_parts.append(f"Alt: {data['assigned_alt']}")
        if "cleared" in data:
            notif_parts.append("CLEARED" if data['cleared'] else "NOT CLEARED")
        if contact_freq and contact_freq > 0:
            notif_parts.append(f"Contact: {contact_freq:.3f} MHz")
        if message:
            notif_parts.append(message)
        if notif_parts:
            self._show_sim_notification(
                f"ATC: {self._atc_from}",
                " | ".join(notif_parts)
            )

        # Ping-Sound abspielen
        try:
            import winsound
            winsound.Beep(880, 200)
        except Exception:
            pass

    def _on_contact_request(self, data: dict):
        """PILOT_CONTACT_REQUEST vom Server: Pilot soll ATC kontaktieren."""
        station = data.get("station", "")
        station_name = data.get("station_name", station)
        full_name = data.get("full_name", station_name)
        freq = data.get("frequency", 0.0)
        role = data.get("role", "")

        if freq <= 0:
            return

        self._contact_freq = freq
        self._contact_station = f"{full_name} [{role}]"

        # Contact-Freq Label aktualisieren
        self.lbl_contact_freq.setText(f"📻  CONTACT {full_name} ON {freq:.3f}")
        self.lbl_contact_freq.setStyleSheet(
            "color: #ff4444; font-weight: bold; font-size: 15px;"
            "background: #330000; border: 2px solid #ff4444;"
            "border-radius: 4px; padding: 6px;")
        self.lbl_contact_freq.setVisible(True)

        # TUNE-Button aktivieren (blinkend)
        self.btn_tune_contact.setText(f"✈  TUNE {freq:.3f} MHz  ({station})")
        self.btn_tune_contact.setVisible(True)
        self._contact_blink_on = True

        # In-Sim Notification via SimConnect
        contact_text = data.get("message", f"CONTACT {station_name} ON {freq:.3f}")
        self._show_sim_notification("ATC CONTACT", contact_text)

        # Audio-Alarm (3x Piepton)
        try:
            import winsound
            for _ in range(3):
                winsound.Beep(1200, 120)
                import time as _t
                _t.sleep(0.08)
        except Exception:
            pass

        self.bridge.log_message.emit(
            f"📡 CONTACT REQUEST: {station_name} auf {freq:.3f} MHz [{role}]")

    def _on_weather_sync(self, data: dict):
        """Handler für Weather-Sync Ergebnisse – aktualisiert UI."""
        icao = data.get("icao", "")
        mode = data.get("mode", "UNKNOWN")
        action = data.get("action", "")
        qnh = data.get("qnh", 0)

        # Wetter-Status in Airport-Panel anzeigen
        if mode == "LIVE":
            wx_status = f"🌦️ {icao}: Live-Wetter aktiv"
            if qnh:
                wx_status += f" – QNH {qnh} hPa gesetzt"
            style = "color: #58a6ff; font-size: 11px;"
        elif action == "FULL_SET":
            temp = data.get("temperature", "")
            wind = data.get("wind_speed", "")
            wx_status = f"🌦️ {icao}: Wetter aus METAR gesetzt"
            if qnh:
                wx_status += f" – QNH {qnh}"
            if temp:
                wx_status += f" T{temp:+.0f}°C" if isinstance(temp, (int, float)) else f" T{temp}°C"
            if wind:
                wx_status += f" W{wind}kt"
            style = "color: #f0883e; font-size: 11px;"
        else:
            wx_status = f"🌦️ {icao}: Wetter-Sync ({mode})"
            style = "color: #6b7989; font-size: 11px;"

        # Update QNH-Warning Label
        try:
            self.lbl_qnh_warning.setText(wx_status)
            self.lbl_qnh_warning.setStyleSheet(style)
        except Exception:
            pass

    def _on_tune_contact_clicked(self):
        """3-in-1 TUNE: COM1 setzen + Mumble-Channel wechseln + ATC benachrichtigen.

        Führt drei Aktionen gleichzeitig aus:
        1. Frequenz im Simulator (Standby → Active) setzen via SimConnect
        2. Mumble-Channel auf die neue Frequenz wechseln
        3. Dem Lotsen im Radar signalisieren: 'Pilot ist jetzt auf deiner Frequenz'
        """
        freq = self._contact_freq
        if freq <= 0:
            return

        station_name = self._contact_station or "ATC"
        freq_str = f"{freq:.3f}"

        # ── 1) COM1 setzen via SimConnect ──
        sim_set_ok = False
        if hasattr(self, 'sm') and self.sm is not None:
            try:
                ae = AircraftEvents(self.sm)
                freq_hz = int(round(freq * 1000) * 1000)
                evt = ae.find("COM_RADIO_SET_HZ")
                if evt is not None:
                    evt(freq_hz)
                    sim_set_ok = True
                else:
                    # BCD16 Fallback
                    freq_khz = round(freq * 1000)
                    digits = f"{freq_khz - 100000:04d}"
                    bcd = int(digits[:4], 16)
                    evt2 = ae.find("COM_RADIO_SET")
                    if evt2 is not None:
                        evt2(bcd)
                        sim_set_ok = True
            except Exception as e:
                self.bridge.log_message.emit(f"⚠️ Auto-Tune Fehler: {e}")

        # ── 2) Mumble-Channel wechseln ──
        if self.voice_mgr:
            self.voice_mgr.sync_frequency(freq)
            self.bridge.log_message.emit(
                f"🎙️ Voice-Channel → {freq_str}")

        # ── 3) Server benachrichtigen: Pilot hat getuned ──
        if hasattr(self, 'net_thread') and self.net_thread:
            self.net_thread._msg_queue.append({
                "type": "PILOT_TUNED",
                "frequency": freq,
                "station": station_name,
            })

        # UI-Feedback
        status_parts = []
        if sim_set_ok:
            status_parts.append("SIM ✓")
        status_parts.append("VOICE ✓" if self.voice_mgr else "VOICE –")
        status_parts.append("ATC ✓")
        self.bridge.log_message.emit(
            f"✅ TUNE {freq_str} MHz → {station_name}  [{' | '.join(status_parts)}]")

        # Button verstecken nach Klick
        self.btn_tune_contact.setVisible(False)
        self.lbl_contact_freq.setVisible(False)
        self._show_sim_notification(
            "TUNED", f"COM1 → {freq_str} MHz | {station_name}")

    def _atc_blink_tick(self):
        """Blink-Timer für unbestätigte ATC-Anweisungen."""
        self._atc_blink_on = not self._atc_blink_on
        self._update_atc_labels()

        # TUNE-Button blinken lassen (rot ↔ dunkelrot)
        self._contact_blink_on = not self._contact_blink_on
        if self.btn_tune_contact.isVisible() and self._contact_freq > 0:
            if self._contact_blink_on:
                self.btn_tune_contact.setStyleSheet(
                    "QPushButton { background: #cc0000; color: #ffffff;"
                    " font-family: Consolas; font-size: 14px; font-weight: bold;"
                    " border: 2px solid #ff4444; border-radius: 6px;"
                    " padding: 6px 12px; }"
                    " QPushButton:hover { background: #ff2222; }")
            else:
                self.btn_tune_contact.setStyleSheet(
                    "QPushButton { background: #330000; color: #ff8888;"
                    " font-family: Consolas; font-size: 14px; font-weight: bold;"
                    " border: 2px solid #661111; border-radius: 6px;"
                    " padding: 6px 12px; }"
                    " QPushButton:hover { background: #550000; }")

    def _send_freq_message(self):
        """Sendet eine Text-Nachricht auf der aktuellen Frequenz."""
        text = self.txt_chat_input.text().strip()
        if not text:
            return
        if not self.net_thread or not self.net_thread._running:
            self._append_chat("System", "Nicht verbunden!", "#ff4444")
            return

        callsign = self.txt_callsign.text().strip() or "PILOT"
        freq = self._current_com1_freq if self._current_com1_freq > 0 else 0.0
        msg = {
            "type": "FREQ_MSG",
            "callsign": callsign,
            "frequency": round(freq, 3),
            "message": text,
        }
        self.net_thread._msg_queue.append(msg)
        self.txt_chat_input.clear()
        self._append_chat(callsign, text, "#58a6ff")

    def _on_freq_message(self, data: dict):
        """Empfängt eine Frequenz-Nachricht von ATC oder anderem Piloten."""
        sender = data.get("from", data.get("callsign", "?"))
        message = data.get("message", "")
        freq = data.get("frequency", 0)
        color = "#ffaa00" if "ATC" in sender or "_" in sender else "#79c0ff"
        self._append_chat(sender, message, color)

        # Für InGame-Panel API speichern (max 50)
        self._api_freq_messages.append({
            "callsign": sender, "frequency": freq,
            "message": message, "time": time.time()
        })
        if len(self._api_freq_messages) > 50:
            self._api_freq_messages = self._api_freq_messages[-50:]

        # Auch im Simulator anzeigen
        if message:
            self._show_sim_notification(sender, message)

    def _on_voice_command(self, data: dict):
        """Verarbeitet Voice-Befehle vom Server (VOICE_MOVE etc.)."""
        if self.voice_mgr:
            self.voice_mgr.handle_server_command(data)

        # UI-Feedback: Station-Name anzeigen wenn verfügbar
        msg_type = data.get("type", "")
        if msg_type == "VOICE_MOVE":
            freq = data.get("frequency", "")
            station = data.get("station", "")
            station_name = data.get("station_name", station)
            channel = data.get("channel", freq)

            if station:
                self.bridge.log_message.emit(
                    f"🎙️ Voice-Channel: {freq} ({station_name})")
                if hasattr(self, "lbl_com1_status"):
                    self.lbl_com1_status.setText(
                        f"ON FREQ — {station_name}")
                    self.lbl_com1_status.setStyleSheet(
                        "color: #58a6ff; font-weight: bold; font-size: 11px;"
                        "background: #0a1a40; border: 1px solid #58a6ff;"
                        "border-radius: 4px; padding: 3px 8px;")
                    self.lbl_com1_status.setVisible(True)
            elif channel == "UNICOM":
                self.bridge.log_message.emit(
                    f"🎙️ Voice-Channel: UNICOM ({freq})")
                if hasattr(self, "lbl_com1_status"):
                    self.lbl_com1_status.setText("UNICOM")
                    self.lbl_com1_status.setStyleSheet(
                        "color: #d29922; font-weight: bold; font-size: 11px;"
                        "background: #2a2200; border: 1px solid #d29922;"
                        "border-radius: 4px; padding: 3px 8px;")
                    self.lbl_com1_status.setVisible(True)

    # ── Signal Quality & Radio Blocking ────────────────────────────
    def _on_signal_quality(self, data: dict):
        """Server meldet aktuelle Funkqualität → VoiceManager."""
        if self.voice_mgr:
            self.voice_mgr.update_signal_quality(data)
        # Update Signal-Anzeige im UI
        q = data.get("quality", 1.0)
        q_pct = int(q * 100)
        relay = data.get("via_relay", "")
        lbl = f"📡 {q_pct}%"
        if relay:
            lbl += f" via {relay}"
        if hasattr(self, "lbl_signal"):
            self.lbl_signal.setText(lbl)
            if q < 0.3:
                self.lbl_signal.setStyleSheet("color: red; font-weight: bold;")
            elif q < 0.6:
                self.lbl_signal.setStyleSheet("color: orange;")
            else:
                self.lbl_signal.setStyleSheet("color: #58a6ff;")

    def _on_radio_blocked(self, data: dict):
        """Server meldet Radio-Blocking → Heterodyn abspielen."""
        if self.voice_mgr:
            self.voice_mgr.play_heterodyne(0.8)

    def _on_stca_alert(self, data: dict):
        """STCA-Warnung → TRAFFIC TRAFFIC Audio ausgeben."""
        self._play_stca_audio()

    def _on_slot_assigned(self, data: dict):
        """Event-Slot oder TSAT zugewiesen → Label aktualisieren."""
        slot_time = data.get("slot_time", "")
        tsat = data.get("tsat", "")
        evt_name = data.get("event_name", "")
        apt = data.get("airport", "")
        direction = data.get("direction", "")
        dir_de = "Abflug" if direction == "DEP" else "Anflug"

        # Slot-Zeit kürzen
        try:
            slot_short = slot_time[11:16] + "Z" if len(slot_time) > 16 else slot_time
            tsat_short = tsat[11:16] + "Z" if len(tsat) > 16 else tsat
        except Exception:
            slot_short = slot_time
            tsat_short = tsat

        if tsat and tsat != slot_time:
            # TSAT weicht vom Slot ab → Verzögerung anzeigen
            text = f"🎫 {evt_name} | {apt} {dir_de} | Slot: {slot_short} | ⏱ TSAT: {tsat_short}"
            self.lbl_tsat.setStyleSheet(
                "color: #d29922; padding: 4px 8px; font-size: 12px; font-weight: bold; "
                "font-family: Consolas; background: rgba(210,153,34,0.15); "
                "border: 1px solid rgba(210,153,34,0.4); border-radius: 6px;")
        else:
            text = f"🎫 {evt_name} | {apt} {dir_de} | Slot: {slot_short}"
            self.lbl_tsat.setStyleSheet(
                "color: #58a6ff; padding: 4px 8px; font-size: 12px; font-weight: bold; "
                "font-family: Consolas; background: rgba(88,166,255,0.1); "
                "border: 1px solid rgba(88,166,255,0.3); border-radius: 6px;")

        self.lbl_tsat.setText(text)
        self.lbl_tsat.setVisible(True)

    def _play_stca_audio(self):
        """Spielt TRAFFIC-TRAFFIC Warnung ab."""
        if not _HAS_AUDIO:
            return
        try:
            import math
            sr = 22050
            duration = 1.5
            n = int(sr * duration)
            samples = []
            for i in range(n):
                t = i / sr
                # Warnton: 700Hz Puls (0.3s an, 0.15s aus)
                cycle = t % 0.45
                env = 0.7 if cycle < 0.3 else 0.0
                val = env * math.sin(2 * math.pi * 700 * t)
                samples.append(int(val * 20000))
            audio = np.array(samples, dtype=np.int16)
            sd.play(audio, samplerate=sr, blocking=False)
        except Exception:
            pass

    # ── CPDLC ─────────────────────────────────────────────────────
    def _create_cpdlc_chime(self):
        """Erzeugt einen kurzen CPDLC-Chime als WAV-Datei (doppel-Ding)."""
        import struct, tempfile, math, os
        try:
            sample_rate = 22050
            duration = 0.6
            n_samples = int(sample_rate * duration)
            samples = []
            for i in range(n_samples):
                t = i / sample_rate
                # Doppelton: 880Hz + 1100Hz mit Hüllkurve
                env = max(0, 1 - t * 3) if t < 0.25 else max(0, 1 - (t - 0.3) * 3)
                val = env * (0.5 * math.sin(2 * math.pi * 880 * t)
                             + 0.3 * math.sin(2 * math.pi * 1100 * t))
                samples.append(int(val * 16000))
            wav_path = os.path.join(tempfile.gettempdir(), "nexus_cpdlc_chime.wav")
            with open(wav_path, "wb") as f:
                n = len(samples)
                f.write(b"RIFF")
                f.write(struct.pack("<I", 36 + n * 2))
                f.write(b"WAVEfmt ")
                f.write(struct.pack("<IHHIIHH", 16, 1, 1, sample_rate,
                                    sample_rate * 2, 2, 16))
                f.write(b"data")
                f.write(struct.pack("<I", n * 2))
                for s in samples:
                    f.write(struct.pack("<h", max(-32768, min(32767, s))))
            self._cpdlc_chime_path = wav_path
        except Exception:
            self._cpdlc_chime_path = ""

    def _play_cpdlc_chime(self):
        """Spielt den CPDLC-Empfangs-Chime ab."""
        if not self._cpdlc_chime_path:
            return
        try:
            import winsound
            winsound.PlaySound(self._cpdlc_chime_path,
                               winsound.SND_FILENAME | winsound.SND_ASYNC)
        except Exception:
            pass

    def _on_cpdlc_message(self, data: dict):
        """Verarbeitet eine eingehende CPDLC-Nachricht vom ATC."""
        msg_id = data.get("msg_id", "")
        sender = data.get("from", "ATC")
        cpdlc_type = data.get("cpdlc_type", "FREE")
        text = data.get("text", "")
        import time as _t
        ts = _t.strftime("%H:%M:%S")

        # DCDU-Anzeige aktualisieren (Airbus-Style)
        self.txt_cpdlc_dcdu.clear()
        self.txt_cpdlc_dcdu.append(
            f'<span style="color:#0088ff;font-size:10px">{ts} FROM {sender}</span>')
        self.txt_cpdlc_dcdu.append(
            f'<span style="color:#58a6ff;font-size:14px;font-weight:bold">'
            f'{text}</span>')
        self.txt_cpdlc_dcdu.append(
            f'<span style="color:#555;font-size:9px">[{cpdlc_type}] ID:{msg_id}</span>')

        # Printer-Strip (ACARS-Papierstreifen)
        self.txt_cpdlc_printer.append(
            f"═══════════════════════════════════\n"
            f"  ACARS MSG  {ts}  [{cpdlc_type}]\n"
            f"  FROM: {sender}\n"
            f"  {text}\n"
            f"  MSG-ID: {msg_id}\n"
            f"═══════════════════════════════════"
        )
        sb = self.txt_cpdlc_printer.verticalScrollBar()
        sb.setValue(sb.maximum())

        # State speichern + Buttons aktivieren
        self._cpdlc_current_msg = data
        self.btn_cpdlc_wilco.setEnabled(True)
        self.btn_cpdlc_unable.setEnabled(True)
        self.btn_cpdlc_standby.setEnabled(True)

        # Chime abspielen
        self._play_cpdlc_chime()

        # Für InGame-Panel API speichern
        if not hasattr(self, '_api_cpdlc_messages'):
            self._api_cpdlc_messages = []
        self._api_cpdlc_messages.append({
            "msg_id": msg_id, "from": sender, "type": cpdlc_type,
            "text": text, "time": _t.time(), "status": "RECEIVED",
        })
        if len(self._api_cpdlc_messages) > 50:
            self._api_cpdlc_messages = self._api_cpdlc_messages[-50:]

    def _cpdlc_respond(self, response: str):
        """Sendet eine CPDLC-Antwort (WILCO/UNABLE/STANDBY) an den Server."""
        if not self._cpdlc_current_msg:
            return
        msg_id = self._cpdlc_current_msg.get("msg_id", "")
        if not msg_id:
            return

        # Antwort via WebSocket senden
        if self.net_thread and self.net_thread._ws:
            import asyncio
            resp_msg = json.dumps({
                "type": "CPDLC_RESPONSE",
                "msg_id": msg_id,
                "response": response,
                "text": "",
            })
            loop = self.net_thread._loop_ref
            ws = self.net_thread._ws
            if loop and ws:
                asyncio.run_coroutine_threadsafe(ws.send(resp_msg), loop)

        # UI aktualisieren
        import time as _t
        ts = _t.strftime("%H:%M:%S")
        resp_colors = {"WILCO": "#58a6ff", "UNABLE": "#ff6040", "STANDBY": "#00b4ff"}
        c = resp_colors.get(response, "#ccc")
        self.txt_cpdlc_dcdu.append(
            f'<span style="color:{c};font-size:13px;font-weight:bold">'
            f'>>> {response} <<<  {ts}</span>')

        # Printer
        self.txt_cpdlc_printer.append(
            f"  >>> RESPONSE: {response}  {ts}\n"
            f"───────────────────────────────────"
        )

        # Buttons deaktivieren (wenn nicht STANDBY)
        if response != "STANDBY":
            self.btn_cpdlc_wilco.setEnabled(False)
            self.btn_cpdlc_unable.setEnabled(False)
            self.btn_cpdlc_standby.setEnabled(False)
            self._cpdlc_current_msg = None

        self._set_status(f"📡 CPDLC: {response} gesendet")

    def _append_chat(self, sender: str, text: str, color: str = "#c8d6e5"):
        """Fügt eine Zeile im Chat-Log hinzu."""
        import time as _t
        ts = _t.strftime("%H:%M:%S")
        self.txt_chat_log.append(
            f'<span style="color:#8b949e">[{ts}]</span> '
            f'<span style="color:{color};font-weight:bold">{sender}:</span> '
            f'<span style="color:#c8d6e5">{text}</span>'
        )
        # Auto-scroll
        sb = self.txt_chat_log.verticalScrollBar()
        sb.setValue(sb.maximum())

    def _update_atc_labels(self):
        """ATC-Anweisungs-Labels aktualisieren (mit Blink-Effekt)."""
        blink_color = "#ff4444" if self._atc_blink_on else "#661111"
        matched_color = "#58a6ff"

        # Squawk – mit Matching-Anzeige
        if self._assigned_squawk > 0:
            sq_matched = (self._current_squawk == self._assigned_squawk)
            if sq_matched:
                sq_text = f"{self._assigned_squawk:04d} ✓ MATCHED"
                sq_color = matched_color
            else:
                sq_text = f"{self._assigned_squawk:04d}  (aktuell: {self._current_squawk:04d})"
                sq_color = blink_color
            self.lbl_assigned_sq.setText(sq_text)
            self.lbl_assigned_sq.setStyleSheet(
                f"color: {sq_color}; font-weight: bold; font-size: 13px; font-family: Consolas;")
        else:
            self.lbl_assigned_sq.setText("---")
            self.lbl_assigned_sq.setStyleSheet("color: #888; font-weight: bold; font-size: 13px;")

        # Altitude
        if self._assigned_alt:
            self.lbl_assigned_alt.setText(self._assigned_alt)
            self.lbl_assigned_alt.setStyleSheet(
                f"color: {blink_color}; font-weight: bold; font-size: 13px;")
        else:
            self.lbl_assigned_alt.setText("---")
            self.lbl_assigned_alt.setStyleSheet("color: #888; font-weight: bold; font-size: 13px;")

        # Cleared
        if self._cleared:
            self.lbl_cleared.setText("CLEARED ✓")
            self.lbl_cleared.setStyleSheet(
                f"color: {matched_color}; font-weight: bold; font-size: 13px;")
        else:
            self.lbl_cleared.setText("NOT CLEARED")
            self.lbl_cleared.setStyleSheet("color: #888; font-weight: bold; font-size: 13px;")

        # Absender
        self.lbl_atc_from.setText(self._atc_from or "---")

    # -- Active ATC Liste ----------------------------------------------------
    def _on_atc_online(self, data: dict):
        station = data.get("station", "")
        if station:
            self._active_atc[station] = data
        self._refresh_atc_list()

    def _on_atc_offline(self, station: str):
        self._active_atc.pop(station, None)
        self._refresh_atc_list()

    def _refresh_atc_list(self):
        """ATC-Liste als klickbare Buttons anzeigen."""
        # Alte Buttons entfernen
        while self._atc_btn_layout.count() > 1:  # letztes Item ist Stretch
            item = self._atc_btn_layout.takeAt(0)
            w = item.widget()
            if w:
                w.deleteLater()

        if not self._active_atc:
            lbl = QLabel("(kein ATC online)")
            lbl.setStyleSheet("color: #666; font-family: Consolas; font-size: 10px; padding: 4px;")
            self._atc_btn_layout.insertWidget(0, lbl)
            return

        for i, (st, info) in enumerate(self._active_atc.items()):
            freq = info.get("frequency", 0)
            role = info.get("role", "?")
            full_name = info.get("full_name", st)

            # Farbe je nach Rolle
            role_colors = {
                "GND": "#44aa44", "TWR": "#ff8844", "APP": "#4488ff",
                "DEP": "#aa44ff", "CTR": "#ff4444", "DEL": "#aaaaaa",
            }
            color = role_colors.get(role, "#00ccff")

            btn = QPushButton(f"📻 {full_name}   {freq:.3f} MHz   [{role}]")
            btn.setStyleSheet(
                f"QPushButton {{ background: #0d1b2a; color: {color};"
                f" font-family: Consolas; font-size: 10px; font-weight: bold;"
                f" border: 1px solid {color}; border-radius: 3px;"
                f" padding: 3px 6px; text-align: left; }}"
                f" QPushButton:hover {{ background: #1a3a5a; }}")
            btn.setToolTip(f"Klicken → COM1 auf {freq:.3f} setzen")
            btn.clicked.connect(lambda _, f=freq, s=st: self._on_atc_clicked(f, s))
            self._atc_btn_layout.insertWidget(i, btn)

    def _on_atc_clicked(self, freq: float, station: str):
        """Wenn der Pilot auf einen ATC-Eintrag klickt → COM1 Auto-Tune."""
        self.lbl_status.setText(f"📻 {station} – tune COM1 to {freq:.3f} MHz")
        self.lbl_status.setStyleSheet(
            "color: #58a6ff; font-weight: bold; font-size: 11px;"
            " background: #0a2a0a; padding: 4px; border-radius: 3px;")

        # SimConnect COM1 setzen
        if hasattr(self, 'sm') and self.sm is not None:
            try:
                ae = AircraftEvents(self.sm)
                # Frequenz in BCD16 umwandeln für COM_RADIO_SET
                # freq ist in MHz (z.B. 118.300)
                # BCD16: z.B. 118.30 → 0x1830 (die Ziffern nach 1xx.xxx)
                freq_khz = round(freq * 1000)  # 118300
                # SimConnect COM_RADIO_SET erwartet BCD16 der letzten 4 Stellen
                # Format: xxyy → die .xx.yy nach der 1
                # Alternativ: COM_RADIO_SET_HZ erwartet Hz direkt
                freq_hz = int(freq_khz * 1000)  # z.B. 118300000 Hz
                evt = ae.find("COM_RADIO_SET_HZ")
                if evt is not None:
                    evt(freq_hz)
                    self.bridge.log_message.emit(
                        f"✅ COM1 auf {freq:.3f} MHz gesetzt ({station})")
                else:
                    # Fallback: BCD16
                    # Entferne die führende 1 → z.B. 118.30 → 1830
                    digits = f"{freq_khz - 100000:04d}"  # "18300" → nimm 4 Ziffern
                    bcd = int(digits[:4], 16)
                    evt2 = ae.find("COM_RADIO_SET")
                    if evt2 is not None:
                        evt2(bcd)
                        self.bridge.log_message.emit(
                            f"✅ COM1 auf {freq:.3f} MHz gesetzt (BCD) ({station})")
                    else:
                        self.bridge.log_message.emit(
                            "⚠️ COM_RADIO_SET Event nicht verfügbar")
            except Exception as e:
                self.bridge.log_message.emit(f"⚠️ COM1-Tune Fehler: {e}")

    # -- SimBrief Import ---------------------------------------------------
    def _on_import_simbrief(self):
        """SimBrief OFP importieren und Flugplan-Felder ausfüllen."""
        pilot_id = self.txt_simbrief_id.text().strip()
        if not pilot_id:
            QMessageBox.warning(self, "SimBrief", "Bitte SimBrief Pilot ID oder Username eingeben.")
            return

        def _fetch():
            try:
                import urllib.request
                import urllib.parse
                import urllib.error
                import ssl as _ssl
                # SSL-Kontext ohne Zertifikatsprüfung (SimBrief API)
                ctx = _ssl.create_default_context()
                ctx.check_hostname = False
                ctx.verify_mode = _ssl.CERT_NONE

                # SimBrief: userid= für Zahlen, username= für Text
                pid = pilot_id.strip()
                if pid.isdigit():
                    param = f"userid={urllib.parse.quote(pid)}"
                else:
                    param = f"username={urllib.parse.quote(pid)}"
                url = f"https://www.simbrief.com/api/xml.fetcher.php?{param}&json=1"

                req = urllib.request.Request(url, headers={
                    "User-Agent": "Mozilla/5.0 (NEXUS-ATC Lotsenradar)",
                    "Accept": "application/json, text/html, */*",
                })

                # HTTP-Fehler (400/404) separat abfangen und Body lesen
                try:
                    with urllib.request.urlopen(req, timeout=15, context=ctx) as resp:
                        raw = resp.read().decode("utf-8", errors="replace")
                except urllib.error.HTTPError as http_err:
                    # SimBrief gibt bei Fehlern HTTP 400 mit JSON-Body zurück
                    body = ""
                    try:
                        body = http_err.read().decode("utf-8", errors="replace")
                    except Exception:
                        pass
                    err_msg = f"HTTP {http_err.code}"
                    if body:
                        try:
                            err_data = json.loads(body)
                            fetch_info = err_data.get("fetch") or {}
                            sb_status = fetch_info.get("status", "")
                            if sb_status:
                                err_msg = sb_status
                        except (json.JSONDecodeError, ValueError):
                            err_msg = body[:200]
                    self.bridge.log_message.emit(f"SimBrief-Fehler: {err_msg}")
                    return

                data = json.loads(raw)

                # SimBrief-Fehler abfangen (z.B. "No flight plan found")
                if "fetch" in data and "status" in data.get("fetch", {}):
                    status = data["fetch"]["status"]
                    if status != "Success":
                        self.bridge.log_message.emit(
                            f"SimBrief: {data['fetch'].get('status', 'Fehler')}")
                        return

                # Sichere Dict-Zugriffe (or {} fängt auch null-Werte ab)
                origin = data.get("origin") or {}
                dest = data.get("destination") or {}
                general = data.get("general") or {}
                aircraft = data.get("aircraft") or {}
                route_str = general.get("route", "") or ""
                actype = aircraft.get("icaocode", "") or ""
                cruise = general.get("initial_altitude", "") or ""

                dep_icao = (origin.get("icao_code", "") or "").upper()
                arr_icao = (dest.get("icao_code", "") or "").upper()

                # ── Callsign aus SimBrief ──
                icao_airline = (general.get("icao_airline", "") or "").strip().upper()
                flight_number = (general.get("flight_number", "") or "").strip()
                sb_callsign = ""
                if icao_airline and flight_number:
                    sb_callsign = f"{icao_airline}{flight_number}"
                elif icao_airline:
                    sb_callsign = icao_airline
                elif flight_number:
                    sb_callsign = flight_number

                # ── Vollständige Flugzeug-Info ──
                ac_name = (aircraft.get("name", "") or "").strip()
                ac_reg = (aircraft.get("reg", "") or "").strip()

                # Cruise FL sicher parsen
                cruise_fl = ""
                if cruise:
                    try:
                        cruise_int = int(float(str(cruise).strip()))
                        if cruise_int > 1000:
                            cruise_fl = f"FL{cruise_int // 100:03d}"
                        elif cruise_int > 0:
                            cruise_fl = f"FL{cruise_int:03d}"
                    except (ValueError, TypeError):
                        cruise_fl = str(cruise)

                self.bridge.log_message.emit(
                    f"SimBrief: {sb_callsign or '?'}  "
                    f"{dep_icao or '?'} \u2192 {arr_icao or '?'}  "
                    f"{actype} ({ac_name})  {ac_reg}  {cruise_fl}")
                self.bridge.log_message.emit(
                    f"  Route: {route_str[:80]}")

                # UI in Main-Thread aktualisieren (Thread-Safety!)
                # QTimer.singleShot funktioniert NICHT aus Background-Thread
                # → Signal verwenden, das im Main-Thread empfangen wird
                self.bridge.simbrief_data.emit({
                    "dep": dep_icao, "arr": arr_icao,
                    "actype": actype, "cruise": cruise_fl,
                    "route": route_str, "callsign": sb_callsign,
                    "ac_name": ac_name, "ac_reg": ac_reg,
                })
            except Exception as e:
                self.bridge.log_message.emit(f"SimBrief-Fehler: {e}")

        t = threading.Thread(target=_fetch, daemon=True)
        t.start()

    def _on_simbrief_data(self, data: dict):
        """Slot für SimBrief-Signal – leitet an _apply_simbrief weiter."""
        self._apply_simbrief(
            data.get("dep", ""), data.get("arr", ""),
            data.get("actype", ""), data.get("cruise", ""),
            data.get("route", ""), data.get("callsign", ""),
            data.get("ac_name", ""), data.get("ac_reg", ""))

    def _apply_simbrief(self, dep: str, arr: str, actype: str, cruise: str,
                         route: str, callsign: str = "",
                         ac_name: str = "", ac_reg: str = ""):
        """Wendet SimBrief-Daten im Main-Thread an."""
        self.txt_dep.setText(dep)
        self.txt_arr.setText(arr)
        # Flugzeugtyp: ICAO-Code + voller Name wenn verfügbar
        if ac_name and actype:
            self.txt_actype.setText(f"{actype} ({ac_name})")
        else:
            self.txt_actype.setText(actype)
        self.txt_cruise.setText(cruise)
        self.txt_route.setText(route)

        # ── Callsign aus SimBrief übernehmen ──
        if callsign:
            self.txt_callsign.setText(callsign)
            self.bridge.log_message.emit(
                f"✈ Callsign gesetzt: {callsign}")

        log_parts = ["✅ SimBrief Flugplan importiert!"]
        if callsign:
            log_parts.append(f"Callsign: {callsign}")
        if ac_reg:
            log_parts.append(f"Reg: {ac_reg}")
        self.bridge.log_message.emit("  ".join(log_parts))

        # ── Automatische Wetter-Synchronisation (nur aktuelle Position) ────
        if self.net_thread:
            def _weather_sync_fp():
                time.sleep(1)
                # Wetter nur für aktuelle Position setzen
                try:
                    pilot_lat, pilot_lon = self.net_thread._get_current_position()
                    if pilot_lat != 0 and pilot_lon != 0:
                        # Nächsten Airport für aktuelle Position bestimmen
                        nearest = self.net_thread._get_nearest_icao_from_server(
                            pilot_lat, pilot_lon)
                        if nearest:
                            self.bridge.log_message.emit(
                                f"🌦️ Wetter-Sync für aktuelle Position: {nearest}")
                            self.net_thread.sync_weather_for_airport(
                                nearest, force=True)
                        elif dep:
                            # Fallback: DEP verwenden
                            self.net_thread.sync_weather_for_airport(
                                dep, force=True)
                    elif dep:
                        # Kein SimConnect → DEP verwenden
                        self.net_thread.sync_weather_for_airport(
                            dep, force=True)
                except Exception as e:
                    self.bridge.log_message.emit(
                        f"🌦️ Wetter-Sync Fehler: {e}")
            t_wx = threading.Thread(target=_weather_sync_fp, daemon=True)
            t_wx.start()

    # -- Airport-Info + QNH Fetch -------------------------------------------
    def _fetch_airport_info(self, icao: str):
        """Holt Flughafeninfo + METAR vom Server (Hintergrund)."""
        def _fetch():
            try:
                import urllib.request
                server_base = self.txt_server.text().strip().replace("ws://", "http://").replace("/ws/pilot", "")
                # METAR abrufen
                url_metar = f"{server_base}/metar/{icao}"
                req = urllib.request.Request(url_metar)
                with urllib.request.urlopen(req, timeout=5) as resp:
                    metar_data = json.loads(resp.read().decode())

                # Airport-Info abrufen
                url_apt = f"{server_base}/airport/{icao}"
                req2 = urllib.request.Request(url_apt)
                try:
                    with urllib.request.urlopen(req2, timeout=5) as resp2:
                        apt_data = json.loads(resp2.read().decode())
                except Exception:
                    apt_data = {}

                self._last_airport_info = {
                    "icao": icao,
                    "metar": metar_data,
                    "airport": apt_data,
                }
            except Exception as e:
                self.bridge.log_message.emit(f"Airport-Info Fehler: {e}")

        t = threading.Thread(target=_fetch, daemon=True)
        t.start()

    def _check_nearest_airport(self, lat: float, lon: float):
        """Prüft nächsten Flughafen und holt Info."""
        def _fetch():
            try:
                import urllib.request
                server_base = self.txt_server.text().strip().replace("ws://", "http://").replace("/ws/pilot", "")
                url = f"{server_base}/nearest_airport?lat={lat}&lon={lon}"
                req = urllib.request.Request(url)
                with urllib.request.urlopen(req, timeout=5) as resp:
                    data = json.loads(resp.read().decode())
                if data and isinstance(data, list) and len(data) > 0:
                    nearest = data[0]
                    icao = nearest.get("icao", "???")
                    name = nearest.get("name", "")
                    dist = nearest.get("distance_nm", 0)
                    self._nearest_airport_data = {
                        "icao": icao, "name": name, "dist_nm": dist
                    }
                    # METAR holen für QNH-Vergleich
                    self._fetch_airport_info(icao)
            except Exception as e:
                pass

        t = threading.Thread(target=_fetch, daemon=True)
        t.start()

    def closeEvent(self, event):
        self._on_disconnect()
        event.accept()

    # -- Airport Panel Update ------------------------------------------------
    def _update_airport_panel(self):
        """Periodisch: Airport-Info + QNH-Warnung aktualisieren."""
        # Nearest Airport anzeigen
        nad = self._nearest_airport_data
        if nad:
            self.lbl_nearest_apt.setText(
                f"{nad['icao']} – {nad['name']} ({nad['dist_nm']:.1f} nm)")

        # METAR + QNH-Warnung
        info = self._last_airport_info
        if info:
            metar = info.get("metar", {})
            raw = metar.get("raw", metar.get("metar", ""))
            if raw:
                self.lbl_metar.setText(raw[:150])

            # QNH-Warnung (simuliert, echte Prüfung via kohlsman_mb)
            qnh = metar.get("qnh")
            if qnh and isinstance(qnh, (int, float)) and qnh > 900:
                # TODO: Vergleich mit tatsächlichem Kohlsman wenn SimConnect aktiv
                self.lbl_qnh_warning.setVisible(False)

            # Frequenzen
            apt = info.get("airport", {})
            freqs = apt.get("frequencies", [])
            if freqs:
                freq_lines = [f"{f.get('kind','?')}: {f.get('mhz','?')} MHz"
                              for f in freqs[:6]]
                self.lbl_apt_freqs.setText("\n".join(freq_lines))

        # Nächsten Airport periodisch prüfen (alle 30s)
        now = time.time()
        if self.sm and now - self._last_nearest_check > 30:
            self._last_nearest_check = now
            try:
                ar = AircraftRequests(self.sm, _time=0)
                lat = ar.get("PLANE_LATITUDE")
                lon = ar.get("PLANE_LONGITUDE")
                if lat and lon:
                    self._last_pilot_lat = float(lat)
                    self._last_pilot_lon = float(lon)
                    # Eigene Position für TCAS
                    self._own_lat = float(lat)
                    self._own_lon = float(lon)
                    try:
                        alt_raw = ar.get("PLANE_ALTITUDE")
                        if alt_raw:
                            self._own_alt = float(alt_raw)
                    except Exception:
                        pass
                    self._check_nearest_airport(float(lat), float(lon))
            except Exception:
                pass

        # Voice: Frequenzbasierter Channel-Wechsel
        if self.voice_mgr and self.sm:
            try:
                ar = AircraftRequests(self.sm, _time=0)
                com1 = ar.get("COM_ACTIVE_FREQUENCY:1")
                if com1 is not None:
                    freq = float(com1)
                    if abs(freq - self._last_com1_freq) > 0.001:
                        self._last_com1_freq = freq
                        ch_name = f"{freq:.3f}"
                        self.voice_mgr.move_to_channel(ch_name)
                        self.bridge.log_message.emit(
                            f"Voice: Frequenz-Wechsel → {ch_name} MHz")
            except Exception:
                pass


# ---------------------------------------------------------------------------
# Lokale HTTP-API (Port 9001) – Für das MSFS In-Game Panel
# ---------------------------------------------------------------------------
class InGamePanelAPI(threading.Thread):
    """HTTP-Server auf Port 9001 für das MSFS In-Game Panel.

    Endpoints:
      GET  /status  → {callsign, frequency, voice_connected, ptt_active, sim_connected}
      GET  /atc     → [{station, frequency, role, controller_name}, ...]
      GET  /freq    → [{callsign, frequency, message, time}, ...]
      GET  /wake    → {from_station, frequency, message} | {}
      GET  /weather → {ICAO: {raw, wind_dir, ...}, ...}
      POST /tune    → {frequency: 123.450}
    """
    def __init__(self, window: PilotWindow):
        super().__init__(daemon=True)
        self.window = window

    def run(self):
        from http.server import HTTPServer, BaseHTTPRequestHandler
        window = self.window

        class Handler(BaseHTTPRequestHandler):
            def log_message(self, *a):
                pass  # kein Logging

            def _cors(self):
                self.send_header("Access-Control-Allow-Origin", "*")
                self.send_header("Access-Control-Allow-Methods", "GET, POST, OPTIONS")
                self.send_header("Access-Control-Allow-Headers", "Content-Type, Accept")

            def _json_response(self, data, code=200):
                body = json.dumps(data).encode("utf-8")
                self.send_response(code)
                self.send_header("Content-Type", "application/json")
                self._cors()
                self.end_headers()
                self.wfile.write(body)

            def do_OPTIONS(self):
                self.send_response(204)
                self._cors()
                self.end_headers()

            def do_GET(self):
                path = self.path.split("?")[0]

                if path == "/status":
                    connected = (
                        window.net_thread is not None
                        and window.net_thread._running
                    )
                    voice_on = (
                        window.voice_mgr is not None
                        and hasattr(window.voice_mgr, '_mumble')
                        and window.voice_mgr._mumble is not None
                    )
                    ptt = (
                        window.voice_mgr is not None
                        and hasattr(window.voice_mgr, '_talking')
                        and window.voice_mgr._talking
                    )
                    freq = window._current_com1_freq
                    cs = ""
                    try:
                        cs = window.txt_callsign.text().strip()
                    except Exception:
                        pass
                    self._json_response({
                        "callsign": cs,
                        "frequency": f"{freq:.3f}" if freq > 0 else "0.000",
                        "voice_connected": voice_on,
                        "ptt_active": ptt,
                        "sim_connected": window.sm is not None,
                        "server_connected": connected,
                        "squawk": window._current_squawk,
                        "assigned_squawk": window._assigned_squawk,
                        "assigned_alt": window._assigned_alt,
                        "cleared": window._cleared,
                        "com1_volume": window._com1_volume,
                        "com2_volume": window._com2_volume,
                        "model_warnings": window._model_warnings[-5:],
                    })

                elif path == "/atc":
                    stations = []
                    for st, info in window._active_atc.items():
                        stations.append({
                            "station": st,
                            "frequency": info.get("frequency", 0),
                            "role": info.get("role", ""),
                            "controller_name": info.get("controller_name", ""),
                        })
                    self._json_response(stations)

                elif path == "/freq":
                    self._json_response(window._api_freq_messages[-20:])

                elif path == "/wake":
                    wake = window._api_wake_msg
                    if wake:
                        window._api_wake_msg = None
                        self._json_response(wake)
                    else:
                        self._json_response({})

                elif path == "/weather":
                    # Wetter aus NetworkThread holen
                    metars = {}
                    weather_sync = {}
                    live_weather = False
                    if window.net_thread and hasattr(window.net_thread, '_weather_metars'):
                        metars = window.net_thread._weather_metars or {}
                        weather_sync = getattr(window.net_thread, '_last_weather_sync', {})
                        live_weather = getattr(window.net_thread, '_live_weather_detected', False)
                    # Auch vom Airport-Info
                    if not metars and window._last_airport_info:
                        m = window._last_airport_info.get("metar", {})
                        if m and m.get("raw"):
                            icao = m.get("station", "XXXX")
                            metars = {icao: m}
                    self._json_response({
                        "metars": metars,
                        "weather_sync": weather_sync,
                        "live_weather": live_weather,
                    })

                elif path == "/cpdlc":
                    # Alle CPDLC-Nachrichten + aktueller DCDU-Text
                    cpdlc_text = ""
                    try:
                        cpdlc_text = window.txt_cpdlc_dcdu.toPlainText()
                    except Exception:
                        pass
                    msgs = getattr(window, '_api_cpdlc_messages', [])
                    has_pending = (
                        hasattr(window, '_cpdlc_current_msg')
                        and window._cpdlc_current_msg is not None
                    )
                    self._json_response({
                        "message": cpdlc_text,
                        "has_message": bool(cpdlc_text and "NO ACTIVE" not in cpdlc_text),
                        "messages": msgs[-20:],
                        "pending": has_pending,
                        "current_msg_id": (
                            window._cpdlc_current_msg.get("msg_id", "")
                            if has_pending else ""
                        ),
                    })

                elif path == "/handoff":
                    # Handoff-Info (Contact-Frequenz)
                    contact_freq = getattr(window, '_contact_freq', 0)
                    contact_station = getattr(window, '_contact_station', "")
                    self._json_response({
                        "frequency": contact_freq,
                        "station": contact_station,
                        "pending": contact_freq > 0,
                    })

                elif path == "/ai_traffic":
                    # AI-Flugzeuge mit Modell-Info (für InGame Panel)
                    traffic = []
                    if window.ai_mgr:
                        traffic = window.ai_mgr.get_traffic_info()
                    self._json_response(traffic)

                else:
                    self._json_response({"error": "not found"}, 404)

            def do_POST(self):
                path = self.path.split("?")[0]
                length = int(self.headers.get("Content-Length", 0))
                body = self.rfile.read(length) if length > 0 else b"{}"
                try:
                    data = json.loads(body)
                except Exception:
                    data = {}

                if path == "/tune":
                    freq = data.get("frequency", 0)
                    if freq and freq > 100 and window.sm:
                        try:
                            ae = AircraftEvents(window.sm)
                            freq_hz = int(round(freq * 1_000_000))
                            evt = ae.find("COM_RADIO_SET_HZ")
                            if evt:
                                evt(freq_hz)
                                self._json_response({"ok": True, "frequency": freq})
                            else:
                                self._json_response({"ok": False, "error": "event not found"}, 500)
                        except Exception as e:
                            self._json_response({"ok": False, "error": str(e)}, 500)
                    else:
                        self._json_response({"ok": False, "error": "invalid"}, 400)

                elif path == "/cpdlc/respond":
                    # CPDLC-Antwort senden (WILCO / UNABLE / STANDBY)
                    response = data.get("response", "").upper()
                    if response in ("WILCO", "UNABLE", "STANDBY"):
                        try:
                            from PyQt6.QtCore import QMetaObject, Qt, Q_ARG
                            QMetaObject.invokeMethod(
                                window, "_cpdlc_respond",
                                Qt.ConnectionType.QueuedConnection,
                                Q_ARG(str, response),
                            )
                            self._json_response({"ok": True, "response": response})
                        except Exception as e:
                            # Fallback: direkt aufrufen (thread-unsafe aber funktioniert)
                            try:
                                window._cpdlc_respond(response)
                                self._json_response({"ok": True, "response": response})
                            except Exception as e2:
                                self._json_response({"ok": False, "error": str(e2)}, 500)
                    else:
                        self._json_response(
                            {"ok": False, "error": "response must be WILCO, UNABLE or STANDBY"}, 400)

                elif path == "/freq_msg":
                    # Funk-Nachricht auf aktueller Frequenz senden
                    message = data.get("message", "").strip()
                    if not message:
                        self._json_response({"ok": False, "error": "empty message"}, 400)
                    elif not window.net_thread or not window.net_thread._running:
                        self._json_response({"ok": False, "error": "not connected"}, 503)
                    else:
                        try:
                            callsign = ""
                            try:
                                callsign = window.txt_callsign.text().strip()
                            except Exception:
                                pass
                            callsign = callsign or "PILOT"
                            freq = window._current_com1_freq if window._current_com1_freq > 0 else 0.0
                            msg = {
                                "type": "FREQ_MSG",
                                "callsign": callsign,
                                "frequency": round(freq, 3),
                                "message": message,
                            }
                            window.net_thread._msg_queue.append(msg)
                            # Auch local anzeigen
                            window._api_freq_messages.append({
                                "callsign": callsign, "frequency": freq,
                                "message": message,
                            })
                            if len(window._api_freq_messages) > 50:
                                window._api_freq_messages = window._api_freq_messages[-50:]
                            self._json_response({"ok": True, "callsign": callsign})
                        except Exception as e:
                            self._json_response({"ok": False, "error": str(e)}, 500)

                elif path == "/voice/toggle":
                    # Voice Verbindung ein/ausschalten
                    action = data.get("action", "toggle")
                    if window.voice_mgr:
                        connected = (
                            hasattr(window.voice_mgr, '_mumble')
                            and window.voice_mgr._mumble is not None
                        )
                        self._json_response({
                            "ok": True,
                            "voice_connected": connected,
                            "info": "Voice wird über den Pilot Client gesteuert",
                        })
                    else:
                        self._json_response({
                            "ok": False,
                            "error": "Voice nicht verfügbar",
                        }, 503)

                else:
                    self._json_response({"error": "not found"}, 404)

        try:
            server = HTTPServer(("127.0.0.1", 9001), Handler)
            print("[InGame-API] Port 9001 bereit")
            server.serve_forever()
        except OSError as e:
            print(f"[InGame-API] Port 9001 belegt: {e}")


# ---------------------------------------------------------------------------
# Main
# ---------------------------------------------------------------------------
# ═══════════════════════════════════════════════════════════════════════════
#  LOGIN DIALOG
# ═══════════════════════════════════════════════════════════════════════════
class PilotLoginDialog(QDialog):
    """VID-based login dialog für NEXUS-ATC."""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.server_http = DEFAULT_SERVER.replace("ws://", "http://").replace("wss://", "https://")
        if "/ws/" in self.server_http:
            self.server_http = self.server_http[:self.server_http.index("/ws/")]
        self.user_data = None
        self._settings = QSettings("NEXUS-ATC", "PilotClient")
        self.setWindowTitle("NEXUS-ATC — Login")
        self.setFixedSize(380, 400)
        self.setStyleSheet("""
            QDialog { background: #0d1117; }
            QLabel  { color: #c9d1d9; font-size: 13px; }
            QLineEdit { background: #161b22; border: 1px solid #30363d; color: #e6edf3;
                        border-radius: 6px; padding: 8px 12px; font-size: 13px; }
            QLineEdit:focus { border-color: #00ccff; }
            QCheckBox { color: #8b949e; font-size: 12px; }
            QCheckBox::indicator { width: 14px; height: 14px; }
            QPushButton#loginBtn { background: #00ccff; color: #0d1117; border: none;
                                   border-radius: 6px; padding: 10px; font-size: 14px;
                                   font-weight: bold; }
            QPushButton#loginBtn:hover { background: #33d6ff; }
            QPushButton#loginBtn:disabled { background: #21262d; color: #484f58; }
        """)

        layout = QVBoxLayout(self)
        layout.setSpacing(12)
        layout.setContentsMargins(32, 28, 32, 28)

        title = QLabel("NEXUS-ATC")
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title.setStyleSheet("font-size:22px; font-weight:700; color:#00ccff; letter-spacing:2px;")
        layout.addWidget(title)

        sub = QLabel("Login mit deiner 7-stelligen VID")
        sub.setAlignment(Qt.AlignmentFlag.AlignCenter)
        sub.setStyleSheet("color:#8b949e; font-size:12px; margin-bottom:8px;")
        layout.addWidget(sub)

        layout.addWidget(QLabel("VID (7-stellig)"))
        self.vid_input = QLineEdit()
        self.vid_input.setPlaceholderText("z.B. 1000000")
        self.vid_input.setMaxLength(7)
        layout.addWidget(self.vid_input)

        layout.addWidget(QLabel("Passwort"))
        self.pw_input = QLineEdit()
        self.pw_input.setEchoMode(QLineEdit.EchoMode.Password)
        self.pw_input.setPlaceholderText("Dein Passwort")
        layout.addWidget(self.pw_input)

        self.save_creds_cb = QCheckBox("VID & Passwort speichern")
        layout.addWidget(self.save_creds_cb)

        self.error_label = QLabel("")
        self.error_label.setStyleSheet("color:#f85149; font-size:12px;")
        self.error_label.setWordWrap(True)
        layout.addWidget(self.error_label)

        self.login_btn = QPushButton("ANMELDEN")
        self.login_btn.setObjectName("loginBtn")
        self.login_btn.clicked.connect(self._do_login)
        layout.addWidget(self.login_btn)

        self.pw_input.returnPressed.connect(self._do_login)
        self.vid_input.returnPressed.connect(lambda: self.pw_input.setFocus())

        # Restore saved credentials
        saved_vid = self._settings.value("saved_vid", "")
        saved_pw = self._settings.value("saved_pw", "")
        if saved_vid:
            self.vid_input.setText(saved_vid)
        if saved_pw:
            self.pw_input.setText(saved_pw)
        if saved_vid and saved_pw:
            self.save_creds_cb.setChecked(True)

    def _do_login(self):
        vid = self.vid_input.text().strip()
        pw  = self.pw_input.text()
        self.error_label.setText("")
        if not vid or not pw:
            self.error_label.setText("VID und Passwort eingeben!")
            return
        self.login_btn.setEnabled(False)
        self.login_btn.setText("Verbinde…")
        try:
            import urllib.request as _ureq
            url = self.server_http + "/login"
            payload = json.dumps({"vid": vid, "password": pw}).encode()
            req = _ureq.Request(url, data=payload,
                                headers={"Content-Type": "application/json"})
            with _ureq.urlopen(req, timeout=10) as resp:
                data = json.loads(resp.read())
            if data.get("error"):
                self.error_label.setText(data["error"])
                self.login_btn.setEnabled(True)
                self.login_btn.setText("ANMELDEN")
                return
            if data.get("is_banned"):
                self.error_label.setText("Dein Account wurde gesperrt.")
                self.login_btn.setEnabled(True)
                self.login_btn.setText("ANMELDEN")
                return
            # Save or clear credentials
            if self.save_creds_cb.isChecked():
                self._settings.setValue("saved_vid", vid)
                self._settings.setValue("saved_pw", pw)
            else:
                self._settings.remove("saved_vid")
                self._settings.remove("saved_pw")
            self.user_data = data
            self.accept()
        except Exception as exc:
            self.error_label.setText(f"Fehler: {exc}")
            self.login_btn.setEnabled(True)
            self.login_btn.setText("ANMELDEN")


def main():
    app = QApplication(sys.argv)
    app.setStyle("Fusion")

    # ── Login Gate ──
    login = PilotLoginDialog()
    if login.exec() != QDialog.DialogCode.Accepted:
        sys.exit(0)

    user = login.user_data
    window = PilotWindow()
    window.setWindowTitle(f"NEXUS-ATC — {user.get('name','')} (VID {user.get('vid','')})")

    # Auto-Config: Name + Callsign aus Server-Daten, gesperrt
    window.txt_pilot_name.setText(user.get('name', ''))
    window.txt_pilot_name.setReadOnly(True)
    window.txt_pilot_name.setStyleSheet(
        window.txt_pilot_name.styleSheet() +
        "QLineEdit { background: #0d1117; color: #00ccff; border-color: #00ccff; }")
    window.txt_callsign.setText(user.get('vid', ''))
    window.txt_callsign.setReadOnly(True)
    window.txt_callsign.setStyleSheet(
        window.txt_callsign.styleSheet() +
        "QLineEdit { background: #0d1117; color: #00ccff; border-color: #00ccff; }")
    # Server-Feld auch sperren
    window.txt_server.setReadOnly(True)

    window.show()
    sys.exit(app.exec())


if __name__ == "__main__":
    main()
