/* ═══════════════════════════════════════════════════════════════════
 *  LotsenradarATC.js  –  InGame-Panel  v2.0
 *  Full feature-parity with pilot_client
 *  Coherent GT compatible (NO async/await, NO ES6 modules)
 * ═══════════════════════════════════════════════════════════════════ */

var API = "http://127.0.0.1:9001";
var POLL_MS = 1200;
var _lastFreqCount = 0;
var _lastCpdlcCount = 0;
var _lastAtcHash = "";
var _activeTab = "main";
var _cpdlcAlertBlink = null;
var _radioAlertBlink = null;

/* ── Helpers ──────────────────────────────────────────────────────── */

function $(id) { return document.getElementById(id); }

function httpGet(path, cb) {
    var xhr = new XMLHttpRequest();
    xhr.open("GET", API + path, true);
    xhr.timeout = 3000;
    xhr.onreadystatechange = function () {
        if (xhr.readyState === 4 && xhr.status === 200) {
            try { cb(JSON.parse(xhr.responseText)); } catch (e) { /* ignore */ }
        }
    };
    xhr.onerror = function () {};
    xhr.ontimeout = function () {};
    xhr.send();
}

function httpPost(path, payload, cb) {
    var xhr = new XMLHttpRequest();
    xhr.open("POST", API + path, true);
    xhr.setRequestHeader("Content-Type", "application/json");
    xhr.timeout = 5000;
    xhr.onreadystatechange = function () {
        if (xhr.readyState === 4) {
            try {
                var d = JSON.parse(xhr.responseText);
                if (cb) cb(d);
            } catch (e) { /* ignore */ }
        }
    };
    xhr.onerror = function () {};
    xhr.ontimeout = function () {};
    xhr.send(JSON.stringify(payload));
}

function escHtml(s) {
    if (!s) return "";
    return String(s).replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
}

function roleClass(station) {
    var s = String(station).toUpperCase();
    if (s.indexOf("_DEL") > -1) return "del";
    if (s.indexOf("_GND") > -1) return "gnd";
    if (s.indexOf("_TWR") > -1) return "twr";
    if (s.indexOf("_APP") > -1) return "app";
    if (s.indexOf("_DEP") > -1) return "dep";
    if (s.indexOf("_CTR") > -1) return "ctr";
    return "";
}

function roleLabel(station) {
    var s = String(station).toUpperCase();
    if (s.indexOf("_DEL") > -1) return "DEL";
    if (s.indexOf("_GND") > -1) return "GND";
    if (s.indexOf("_TWR") > -1) return "TWR";
    if (s.indexOf("_APP") > -1) return "APP";
    if (s.indexOf("_DEP") > -1) return "DEP";
    if (s.indexOf("_CTR") > -1) return "CTR";
    return "ATC";
}

function shortStation(station) {
    var parts = String(station).split("_");
    return parts[0] || station;
}

function fmtFreq(f) {
    if (!f || f < 100) return "----.---";
    return Number(f).toFixed(3);
}

function timeNow() {
    var d = new Date();
    var hh = ("0" + d.getUTCHours()).slice(-2);
    var mm = ("0" + d.getUTCMinutes()).slice(-2);
    return hh + ":" + mm + "Z";
}

/* ── Tab System ───────────────────────────────────────────────────── */

function switchTab(name) {
    _activeTab = name;
    var btns = document.querySelectorAll(".tab-btn");
    var tabs = document.querySelectorAll(".tab-content");
    for (var i = 0; i < btns.length; i++) {
        if (btns[i].getAttribute("data-tab") === name) {
            btns[i].classList.add("active");
            btns[i].classList.remove("alert");
        } else {
            btns[i].classList.remove("active");
        }
    }
    for (var j = 0; j < tabs.length; j++) {
        if (tabs[j].id === "tab-" + name) {
            tabs[j].classList.add("active");
        } else {
            tabs[j].classList.remove("active");
        }
    }
    // Clear alert on switched tab
    if (name === "cpdlc" && _cpdlcAlertBlink) { clearInterval(_cpdlcAlertBlink); _cpdlcAlertBlink = null; }
    if (name === "radio" && _radioAlertBlink) { clearInterval(_radioAlertBlink); _radioAlertBlink = null; }
}

// Bind tab buttons
document.addEventListener("DOMContentLoaded", function () {
    var btns = document.querySelectorAll(".tab-btn");
    for (var i = 0; i < btns.length; i++) {
        (function (btn) {
            btn.addEventListener("click", function () {
                switchTab(btn.getAttribute("data-tab"));
            });
        })(btns[i]);
    }
    // Enter key for chat
    var chatInput = $("chatInput");
    if (chatInput) {
        chatInput.addEventListener("keydown", function (e) {
            if (e.key === "Enter" || e.keyCode === 13) {
                e.preventDefault();
                sendFreqMsg();
            }
        });
    }
});

/* ── Alert Tab ────────────────────────────────────────────────────── */

function alertTab(tabName) {
    if (_activeTab === tabName) return;
    var btns = document.querySelectorAll('.tab-btn[data-tab="' + tabName + '"]');
    if (btns.length > 0) {
        btns[0].classList.add("alert");
    }
}

/* ── CPDLC Respond ────────────────────────────────────────────────── */

function cpdlcRespond(response) {
    $("btnWilco").disabled = true;
    $("btnUnable").disabled = true;
    $("btnStandby").disabled = true;
    httpPost("/cpdlc/respond", { response: response }, function (d) {
        if (d && d.ok) {
            var screen = $("dcduScreen");
            if (screen) {
                screen.innerHTML = '<div class="dcdu-sent">' + response + ' GESENDET</div>';
            }
        }
    });
}

/* ── Send Freq Message ────────────────────────────────────────────── */

function sendFreqMsg() {
    var input = $("chatInput");
    if (!input) return;
    var msg = input.value.trim();
    if (!msg) return;
    input.value = "";
    httpPost("/freq_msg", { message: msg }, function (d) {
        if (d && !d.ok) {
            // Show error in chat
            var log = $("chatLog");
            if (log) {
                var el = document.createElement("div");
                el.className = "chat-msg system";
                el.textContent = "Fehler: " + (d.error || "unbekannt");
                log.appendChild(el);
                log.scrollTop = log.scrollHeight;
            }
        }
    });
}

/* ── Tune Frequency ───────────────────────────────────────────────── */

function tuneFreq(freq) {
    httpPost("/tune", { frequency: freq }, function (d) {
        var el = $("freqTuneStatus");
        if (el) {
            if (d && d.ok) {
                el.textContent = "TUNED " + fmtFreq(freq);
                el.className = "freq-tune-status tuned";
            } else {
                el.textContent = "TUNE FAILED";
                el.className = "freq-tune-status failed";
            }
            setTimeout(function () { el.textContent = ""; el.className = "freq-tune-status"; }, 3000);
        }
    });
}

/* ══════════════════════════════════════════════════════════════════
 *  POLLING
 * ══════════════════════════════════════════════════════════════════ */

function pollAll() {
    pollStatus();
    pollAtc();
    pollFreq();
    pollCpdlc();
    pollHandoff();
    pollWeather();
    pollTraffic();
}

/* ── Status ──────────────────────────────────────────────────────── */

function pollStatus() {
    httpGet("/status", function (d) {
        // LEDs
        var ledSim = $("ledSim");
        var ledSrv = $("ledSrv");
        var ledPTT = $("ledPTT");
        if (ledSim) ledSim.className = "led " + (d.sim_connected ? "on-green" : "off");
        if (ledSrv) ledSrv.className = "led " + (d.server_connected ? "on-blue" : "off");
        if (ledPTT) ledPTT.className = "led " + (d.ptt_active ? "on-orange" : (d.voice_connected ? "on-dim" : "off"));

        // Status text
        var st = $("statusText");
        if (st) {
            if (d.server_connected) {
                st.textContent = d.callsign || "ONLINE";
                st.className = "status-text online";
            } else {
                st.textContent = "OFFLINE";
                st.className = "status-text offline";
            }
        }

        // Frequency
        var fv = $("freqValue");
        if (fv) fv.textContent = fmtFreq(d.frequency);

        // Squawk
        var sq = $("squawkValue");
        if (sq) sq.textContent = d.squawk || "----";

        var asq = $("assignedSquawk");
        if (asq) asq.textContent = d.assigned_squawk || "----";

        var alt = $("assignedAlt");
        if (alt) alt.textContent = d.assigned_alt || "---";

        // Cleared bar
        var cb = $("clearedBar");
        if (cb) {
            if (d.cleared) {
                cb.textContent = "✓ " + d.cleared;
                cb.style.display = "block";
            } else {
                cb.style.display = "none";
            }
        }

        // Voice bar
        var vs = $("voiceStatus");
        if (vs) {
            if (d.voice_connected) {
                vs.textContent = "VOICE ON";
                vs.className = "voice-indicator connected";
            } else {
                vs.textContent = "VOICE OFF";
                vs.className = "voice-indicator disconnected";
            }
        }
        var ps = $("pttStatus");
        if (ps) {
            if (d.ptt_active) {
                ps.textContent = "TX";
                ps.className = "ptt-indicator transmit";
            } else {
                ps.textContent = "STANDBY";
                ps.className = "ptt-indicator standby";
            }
        }

        // Model warnings
        if (d.model_warnings && d.model_warnings.length > 0) {
            var sec = $("modelWarnSection");
            var lst = $("modelWarnList");
            if (sec && lst) {
                sec.style.display = "block";
                var html = "";
                for (var i = 0; i < d.model_warnings.length; i++) {
                    html += '<div class="model-warn-item">' + escHtml(d.model_warnings[i]) + '</div>';
                }
                lst.innerHTML = html;
            }
        }

        // Signal bar placeholder
        var sig = $("signalBar");
        if (sig) {
            if (d.server_connected && d.sim_connected) {
                var bars = d.voice_connected ? 4 : 2;
                var sigHtml = '<span class="signal-label">SIGNAL</span>';
                for (var b = 1; b <= 5; b++) {
                    sigHtml += '<span class="signal-bar-item ' + (b <= bars ? "active" : "") + '"></span>';
                }
                sig.innerHTML = sigHtml;
                sig.style.display = "flex";
            } else {
                sig.style.display = "none";
            }
        }
    });
}

/* ── ATC List ────────────────────────────────────────────────────── */

function pollAtc() {
    httpGet("/atc", function (stations) {
        if (!stations || !stations.length) {
            var qk = $("atcListQuick");
            if (qk) qk.innerHTML = '<div class="empty">Keine Lotsen</div>';
            var fl = $("atcListFull");
            if (fl) fl.innerHTML = '<div class="empty">Keine Lotsen in Reichweite</div>';
            _lastAtcHash = "";
            return;
        }

        // Sort: DEL, GND, TWR, APP, DEP, CTR
        var order = { "DEL": 0, "GND": 1, "TWR": 2, "APP": 3, "DEP": 4, "CTR": 5 };
        stations.sort(function (a, b) {
            var ra = roleLabel(a.station);
            var rb = roleLabel(b.station);
            return (order[ra] || 9) - (order[rb] || 9);
        });

        // Check if changed
        var hash = "";
        for (var h = 0; h < stations.length; h++) hash += stations[h].station + ";";
        if (hash === _lastAtcHash) return;
        _lastAtcHash = hash;

        // Quick list (top 4)
        var quickHtml = "";
        var limit = Math.min(stations.length, 4);
        for (var i = 0; i < limit; i++) {
            var s = stations[i];
            var rc = roleClass(s.station);
            quickHtml += '<div class="atc-item ' + rc + '" onclick="tuneFreq(' + s.frequency + ')">'
                + '<span class="atc-role">' + roleLabel(s.station) + '</span>'
                + '<span class="atc-name">' + escHtml(shortStation(s.station)) + '</span>'
                + '<span class="atc-freq">' + fmtFreq(s.frequency) + '</span>'
                + '</div>';
        }
        var qk2 = $("atcListQuick");
        if (qk2) qk2.innerHTML = quickHtml;

        // Full list
        var fullHtml = "";
        for (var j = 0; j < stations.length; j++) {
            var st = stations[j];
            var rc2 = roleClass(st.station);
            fullHtml += '<div class="atc-item ' + rc2 + '" onclick="tuneFreq(' + st.frequency + ')">'
                + '<span class="atc-role">' + roleLabel(st.station) + '</span>'
                + '<span class="atc-name">' + escHtml(st.station) + '</span>'
                + '<span class="atc-freq">' + fmtFreq(st.frequency) + '</span>'
                + '<span class="atc-ctrl">' + escHtml(st.controller_name) + '</span>'
                + '</div>';
        }
        var fl2 = $("atcListFull");
        if (fl2) fl2.innerHTML = fullHtml;
    });
}

/* ── Freq Messages / Radio ────────────────────────────────────────── */

function pollFreq() {
    httpGet("/freq", function (msgs) {
        if (!msgs) return;

        // Alert if new messages
        if (msgs.length > _lastFreqCount && _lastFreqCount > 0) {
            alertTab("radio");
        }
        _lastFreqCount = msgs.length;

        var log = $("chatLog");
        if (!log) return;

        if (msgs.length === 0) {
            log.innerHTML = '<div class="empty">Keine Nachrichten</div>';
            return;
        }

        var html = "";
        for (var i = 0; i < msgs.length; i++) {
            var m = msgs[i];
            var who = escHtml(m.callsign || "???");
            var freq = m.frequency ? fmtFreq(m.frequency) : "";
            var text = escHtml(m.message || "");
            html += '<div class="chat-msg">'
                + '<span class="chat-who">' + who + '</span>'
                + (freq ? ' <span class="chat-freq">[' + freq + ']</span>' : '')
                + ' <span class="chat-text">' + text + '</span>'
                + '</div>';
        }
        log.innerHTML = html;
        log.scrollTop = log.scrollHeight;
    });
}

/* ── CPDLC ────────────────────────────────────────────────────────── */

function pollCpdlc() {
    httpGet("/cpdlc", function (d) {
        if (!d) return;

        // DCDU Screen
        var screen = $("dcduScreen");
        var btnW = $("btnWilco");
        var btnU = $("btnUnable");
        var btnS = $("btnStandby");

        if (d.pending && d.message) {
            if (screen) {
                screen.innerHTML = '<div class="dcdu-msg">' + escHtml(d.message) + '</div>';
            }
            if (btnW) btnW.disabled = false;
            if (btnU) btnU.disabled = false;
            if (btnS) btnS.disabled = false;
        } else if (d.has_message && d.message) {
            if (screen) {
                screen.innerHTML = '<div class="dcdu-msg ack">' + escHtml(d.message) + '</div>';
            }
            if (btnW) btnW.disabled = true;
            if (btnU) btnU.disabled = true;
            if (btnS) btnS.disabled = true;
        } else {
            if (screen) {
                screen.innerHTML = '<div class="dcdu-empty">— NO ACTIVE CPDLC —</div>';
            }
            if (btnW) btnW.disabled = true;
            if (btnU) btnU.disabled = true;
            if (btnS) btnS.disabled = true;
        }

        // CPDLC message history + ACARS
        var msgs = d.messages || [];

        // Alert if new
        if (msgs.length > _lastCpdlcCount && _lastCpdlcCount > 0) {
            alertTab("cpdlc");
        }
        _lastCpdlcCount = msgs.length;

        // ACARS printer (last 5)
        var printer = $("acarsPrinter");
        if (printer) {
            if (msgs.length === 0) {
                printer.innerHTML = '<div class="acars-empty">Kein ACARS-Verkehr</div>';
            } else {
                var acars = "";
                var start = Math.max(0, msgs.length - 5);
                for (var a = start; a < msgs.length; a++) {
                    var am = msgs[a];
                    acars += '<div class="acars-line">'
                        + '<span class="acars-time">' + escHtml(am.time || timeNow()) + '</span> '
                        + '<span class="acars-dir">' + (am.direction === "UP" ? "↑" : "↓") + '</span> '
                        + '<span class="acars-text">' + escHtml(am.text || am.message || "") + '</span>'
                        + '</div>';
                }
                printer.innerHTML = acars;
            }
        }

        // History
        var hist = $("cpdlcHistory");
        if (hist) {
            if (msgs.length === 0) {
                hist.innerHTML = '<div class="empty">Keine CPDLC Nachrichten</div>';
            } else {
                var hhtml = "";
                for (var i = msgs.length - 1; i >= 0; i--) {
                    var cm = msgs[i];
                    var dir = cm.direction === "UP" ? "uplink" : "downlink";
                    hhtml += '<div class="cpdlc-msg ' + dir + '">'
                        + '<span class="cpdlc-time">' + escHtml(cm.time || "") + '</span> '
                        + '<span class="cpdlc-dir">' + (cm.direction === "UP" ? "ATC→" : "→ATC") + '</span> '
                        + '<span class="cpdlc-text">' + escHtml(cm.text || cm.message || "") + '</span>'
                        + (cm.response ? ' <span class="cpdlc-resp">[' + escHtml(cm.response) + ']</span>' : '')
                        + '</div>';
                }
                hist.innerHTML = hhtml;
            }
        }

        // Main tab CPDLC summary
        var sumSec = $("cpdlcSummarySection");
        var sumEl = $("cpdlcSummary");
        if (sumSec && sumEl) {
            if (d.pending || (msgs.length > 0)) {
                sumSec.style.display = "block";
                if (d.pending) {
                    sumEl.innerHTML = '<span class="cpdlc-pending-badge">⚡ NEUE NACHRICHT</span> '
                        + escHtml(d.message || "").substring(0, 60);
                } else {
                    var last = msgs[msgs.length - 1];
                    sumEl.innerHTML = escHtml((last.text || last.message || "").substring(0, 60));
                }
            } else {
                sumSec.style.display = "none";
            }
        }
    });
}

/* ── Handoff ──────────────────────────────────────────────────────── */

function pollHandoff() {
    httpGet("/handoff", function (d) {
        var bar = $("handoffBar");
        if (!bar) return;
        if (d && d.pending && d.frequency > 0) {
            bar.innerHTML = '<span class="handoff-text">CONTACT ' + escHtml(d.station)
                + ' ' + fmtFreq(d.frequency) + '</span>'
                + '<button class="handoff-tune-btn" onclick="tuneFreq(' + d.frequency + ')">TUNE</button>';
            bar.style.display = "flex";
            bar.className = "handoff-bar active";
        } else {
            bar.style.display = "none";
            bar.className = "handoff-bar";
        }
    });
}

/* ── Weather ──────────────────────────────────────────────────────── */

function pollWeather() {
    httpGet("/weather", function (d) {
        var box = $("weatherBox");
        if (!box) return;
        if (!d || !d.metars) {
            box.innerHTML = '<div class="empty">Kein Wetter</div>';
            return;
        }

        var html = "";
        var metars = d.metars;
        var keys = Object.keys(metars);
        if (keys.length === 0) {
            box.innerHTML = '<div class="empty">Kein Wetter</div>';
            return;
        }

        for (var i = 0; i < keys.length; i++) {
            var icao = keys[i];
            var m = metars[icao];
            var raw = "";
            if (typeof m === "string") {
                raw = m;
            } else if (m && m.raw) {
                raw = m.raw;
            } else if (m && m.metar) {
                raw = m.metar;
            }
            html += '<div class="wx-metar">'
                + '<div class="wx-icao">' + escHtml(icao) + '</div>'
                + '<div class="wx-raw">' + escHtml(raw) + '</div>'
                + '</div>';
        }

        if (d.live_weather) {
            html += '<div class="wx-live-badge">🌍 LIVE WEATHER ACTIVE</div>';
        }

        box.innerHTML = html;
    });
}

/* ── AI Traffic ───────────────────────────────────────────────────── */

function pollTraffic() {
    httpGet("/ai_traffic", function (traffic) {
        var list = $("aiTrafficList");
        var count = $("aiCount");
        if (!list) return;

        if (!traffic || traffic.length === 0) {
            list.innerHTML = '<div class="empty">Kein AI-Traffic</div>';
            if (count) count.textContent = "";
            return;
        }

        if (count) count.textContent = "(" + traffic.length + ")";

        var html = "";
        for (var i = 0; i < traffic.length; i++) {
            var t = traffic[i];
            var cs = escHtml(t.callsign || t.title || "UNKNOWN");
            var alt = t.altitude ? Math.round(t.altitude) + " ft" : "---";
            var dist = t.distance ? t.distance.toFixed(1) + " nm" : "";
            var model = escHtml(t.model || "");
            html += '<div class="traffic-item">'
                + '<span class="traffic-cs">' + cs + '</span>'
                + '<span class="traffic-alt">' + alt + '</span>'
                + (dist ? '<span class="traffic-dist">' + dist + '</span>' : '')
                + (model ? '<span class="traffic-model">' + model + '</span>' : '')
                + '</div>';
        }
        list.innerHTML = html;
    });
}

/* ══════════════════════════════════════════════════════════════════
 *  ATC Message Banner (wake/contact)
 * ══════════════════════════════════════════════════════════════════ */

function pollWake() {
    httpGet("/wake", function (d) {
        if (!d || !d.type) return;
        var banner = $("atcMsgBanner");
        if (!banner) return;
        banner.textContent = d.message || d.text || JSON.stringify(d);
        banner.style.display = "block";
        banner.className = "atc-msg-banner active";
        setTimeout(function () {
            banner.style.display = "none";
            banner.className = "atc-msg-banner";
        }, 8000);
    });
}

/* ══════════════════════════════════════════════════════════════════
 *  INIT & MAIN LOOP
 * ══════════════════════════════════════════════════════════════════ */

function initPanel() {
    // Initial poll
    pollAll();
    pollWake();

    // Recurring poll
    setInterval(pollAll, POLL_MS);
    setInterval(pollWake, 5000);
}

// Start when DOM is ready
if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", initPanel);
} else {
    initPanel();
}
