"""
NEXUS-ATC – ATC Radar (Komplett)
====================================
PyQt6-Anwendung für Fluglotsen mit:
- Sektordaten (ARTCC, Runway, Taxiway, SID, STAR)
- OSM-Kartenkacheln als Hintergrund
- QPixmap-Cache für statische Hintergrundlinien (Performance)
- Full Data Block (Callsign / Alt / Speed / Squawk / Freq)
- Squawk-Matching (Assigned ↔ Actual)
- Rechtsklick-Kontextmenü (Assigned Squawk, Assigned Alt, Handover, CLEARED)
- Flight-Strip-Leiste (alle Flugzeuge auf eigener Frequenz)
- Controller-Registrierung (Station, Frequenz, Rolle)
- ICAO-Suche (Radar auf Flughafen zentrieren)
- Handover-Logik (Blink + Accept/Reject)
- Rollen-Dropdown (GND / TWR / APP / CTR)

Nutzung:
    python atc_radar.py
    python atc_radar.py --server ws://192.168.1.50:9000/ws/atc
    python atc_radar.py --sector sectors/eddf_example.sct
"""

from __future__ import annotations

import argparse
import json
import logging
import math
import os
import sys
import urllib.request

log = logging.getLogger(__name__)
from pathlib import Path
from typing import Dict, List, Optional

from PyQt6.QtCore import Qt, QTimer, QThread, pyqtSignal, QRectF, QUrl, QPointF, QEvent, QSettings
from PyQt6.QtGui import (
    QPainter, QColor, QFont, QPen, QBrush, QPixmap, QImage,
    QAction, QCursor, QPolygonF,
)
from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QStatusBar, QGroupBox, QTextEdit, QSplitter,
    QLineEdit, QPushButton, QFormLayout, QComboBox, QCheckBox,
    QMenu, QInputDialog, QScrollArea, QFrame, QMessageBox,
    QTreeWidget, QTreeWidgetItem, QDockWidget, QSlider, QDialog,
    QDialogButtonBox, QGraphicsDropShadowEffect,
)
from PyQt6.QtWebSockets import QWebSocket

from sct_parser import SectorData, parse_sct_file, GeoLine, Runway, Navaid, Label, Region
from airport_db import AirportDB
from weather_service import WeatherService, ParsedMetar, recommend_runway, generate_atis
from auto_sector import generate_simple_airport_layout
from nav_db import NavDB, AirwaySegment
from map_projection import MapProjection
from tile_manager import TileManager
from fir_boundaries import FIRManager, FIRRegion
from osm_overpass import OSMOverpass, OSMData
from gate_db import GateDB, GateInfo, get_gate_db, _haversine_m as _gate_haversine_m

import json as _json
import sqlite3
import threading


# ---------------------------------------------------------------------------
# Konfiguration
# ---------------------------------------------------------------------------
DEFAULT_SERVER = "ws://localhost:9000/ws/atc"
DEFAULT_CENTER_LAT = 50.0
DEFAULT_CENTER_LON = 10.0
DEFAULT_SCALE = 80.0

KM_PER_DEG_LAT = 111.32

# Globale Airport-DB (wird beim Start geladen)
try:
    _AIRPORT_DB = AirportDB()
except Exception:
    _AIRPORT_DB = None

# Wetter-Service
_WEATHER = WeatherService()

# Navigation-DB
try:
    _NAV_DB = NavDB()
except Exception:
    _NAV_DB = None

# FIR-Grenzen
try:
    _FIR_MGR = FIRManager()
except Exception:
    _FIR_MGR = None

# OSM Overpass (für Taxiway/Gate-Daten)
try:
    _OSM_OVERPASS = OSMOverpass()
except Exception:
    _OSM_OVERPASS = None

# Gate-Datenbank (SQLite)
try:
    _GATE_DB = get_gate_db(auto_download_top=False)
except Exception:
    _GATE_DB = None

# Projektions-Helfer
_PROJ = MapProjection()

# Globale Navigations-Datenbank (für Frequenz-Lookup)
_GLOBAL_NAV_DB = Path(__file__).parent / "data" / "global_nav.db"

# ── Bekannte Center-Stationen (Client-seitig, analog zu server.py) ──────
CENTER_STATIONS: Dict[str, Dict] = {
    # EDGG – Langen Center
    "EDGG_A_CTR": {"freq": 135.725, "name": "Langen Radar", "fir": "EDGG"},
    "EDGG_B_CTR": {"freq": 124.475, "name": "Langen Radar", "fir": "EDGG"},
    "EDGG_C_CTR": {"freq": 127.725, "name": "Langen Radar", "fir": "EDGG"},
    "EDGG_D_CTR": {"freq": 136.125, "name": "Langen Radar", "fir": "EDGG"},
    "EDGG_E_CTR": {"freq": 132.405, "name": "Langen Radar", "fir": "EDGG"},
    "EDGG_K_CTR": {"freq": 123.275, "name": "Langen Radar", "fir": "EDGG"},
    "EDGG_R_CTR": {"freq": 124.725, "name": "Rhein Radar",  "fir": "EDGG"},
    "EDGG_CTR":   {"freq": 135.725, "name": "Langen Radar", "fir": "EDGG"},
    # EDMM – München Center
    "EDMM_A_CTR": {"freq": 134.075, "name": "München Radar", "fir": "EDMM"},
    "EDMM_B_CTR": {"freq": 126.950, "name": "München Radar", "fir": "EDMM"},
    "EDMM_C_CTR": {"freq": 133.680, "name": "München Radar", "fir": "EDMM"},
    "EDMM_D_CTR": {"freq": 128.025, "name": "München Radar", "fir": "EDMM"},
    "EDMM_E_CTR": {"freq": 120.150, "name": "München Radar", "fir": "EDMM"},
    "EDMM_F_CTR": {"freq": 132.330, "name": "München Radar", "fir": "EDMM"},
    "EDMM_S_CTR": {"freq": 125.175, "name": "München Radar", "fir": "EDMM"},
    "EDMM_CTR":   {"freq": 134.075, "name": "München Radar", "fir": "EDMM"},
    # EDWW – Bremen Center
    "EDWW_A_CTR": {"freq": 124.825, "name": "Bremen Radar",  "fir": "EDWW"},
    "EDWW_B_CTR": {"freq": 126.650, "name": "Bremen Radar",  "fir": "EDWW"},
    "EDWW_C_CTR": {"freq": 134.150, "name": "Bremen Radar",  "fir": "EDWW"},
    "EDWW_D_CTR": {"freq": 121.350, "name": "Bremen Radar",  "fir": "EDWW"},
    "EDWW_E_CTR": {"freq": 123.900, "name": "Bremen Radar",  "fir": "EDWW"},
    "EDWW_H_CTR": {"freq": 129.725, "name": "Bremen Radar",  "fir": "EDWW"},
    "EDWW_K_CTR": {"freq": 135.725, "name": "Bremen Radar",  "fir": "EDWW"},
    "EDWW_CTR":   {"freq": 124.825, "name": "Bremen Radar",  "fir": "EDWW"},
    # LSAS – Swiss Radar
    "LSAS_CTR":   {"freq": 136.215, "name": "Swiss Radar",   "fir": "LSAS"},
    "LSAS_A_CTR": {"freq": 136.215, "name": "Swiss Radar",   "fir": "LSAS"},
    # LOVV – Wien Center
    "LOVV_CTR":   {"freq": 133.650, "name": "Wien Radar",    "fir": "LOVV"},
    # LIPP – Padova Center
    "LIPP_CTR":   {"freq": 124.450, "name": "Padova Radar",  "fir": "LIPP"},
    # LFFF – Paris Center
    "LFFF_CTR":   {"freq": 136.275, "name": "Paris Control",  "fir": "LFFF"},
    # EGTT – London Center
    "EGTT_CTR":   {"freq": 127.825, "name": "London Control", "fir": "EGTT"},
    # EHAA – Amsterdam Radar
    "EHAA_CTR":   {"freq": 132.350, "name": "Amsterdam Radar","fir": "EHAA"},
    # EBBU – Brussels Center
    "EBBU_CTR":   {"freq": 129.575, "name": "Brussels Radar", "fir": "EBBU"},
    # EKDK – Copenhagen Center
    "EKDK_CTR":   {"freq": 134.350, "name": "Copenhagen Control", "fir": "EKDK"},
    # ESMM – Malmö Center
    "ESMM_CTR":   {"freq": 133.900, "name": "Malmö Control", "fir": "ESMM"},
    # EPWW – Warszawa Center
    "EPWW_CTR":   {"freq": 133.200, "name": "Warszawa Control", "fir": "EPWW"},
    # LKAA – Praha Center
    "LKAA_CTR":   {"freq": 127.125, "name": "Praha Control",  "fir": "LKAA"},
}


def fetch_frequency_for_callsign(callsign: str):
    """Parst ein ATC-Callsign (z.B. LSZH_APP, EDGG_A_CTR) und sucht die Frequenz.

    Prüft zuerst CENTER_STATIONS, dann die airports/frequencies-DB,
    dann die firs-Tabelle als Fallback.

    Returns:
        (freq_mhz: float | None, icao: str | None, airport_name: str | None)
    """
    callsign = callsign.strip().upper()
    parts = callsign.split("_")
    if len(parts) < 2:
        return None, None, None

    icao = parts[0]
    role = parts[-1]  # Letztes Element = Rolle (z.B. EDGG_A_CTR → CTR)

    # ── 1) CENTER_STATIONS Direkt-Lookup ──────────────────────────────
    if role == "CTR":
        # Exakter Match
        ctr_info = CENTER_STATIONS.get(callsign)
        if ctr_info:
            return ctr_info["freq"], icao, ctr_info.get("name", "")

        # Prefix-Scan: EDGG_CTR → EDGG_A_CTR (erste passende)
        for cs_key, cs_val in CENTER_STATIONS.items():
            if cs_key.startswith(icao + "_") and cs_key.endswith("_CTR"):
                return cs_val["freq"], icao, cs_val.get("name", "")

    # Mapping: Suffix → mögliche 'kind'-Werte in der DB
    role_map = {
        "DEL": ["CLD", "CLR", "CLNC", "CLNC DEL", "DELIVERY", "CLRD", "DEL", "DLV",
                 "CLEARANCE", "C/D", "CLR DLVR", "CLRN", "DCL", "APP/DEL"],
        "GND": ["GND", "GROUND", "GMC", "SMC", "GRD", "GRN",
                 "GROUND CONTROL", "GROUND (EAST)", "GROUND (WEST)", "GROUND (MAIN)"],
        "TWR": ["TWR", "TOWER", "TOWER PRIMARY", "TOWER SECONDARY",
                 "TWR 1", "TWR 2", "MTWR", "AFIS-TOWER"],
        "APP": ["APP", "APPROACH", "APCH", "APPR", "APP/DEP", "APP/RAD",
                 "APP CONTROL", "RADAR", "RAD", "RAD/APP", "APP/RADAR",
                 "DIR", "DIRECTOR", "ARR", "ARRIVAL"],
        "DEP": ["DEP", "DEPARTURE", "DEPARTURES", "DEPT", "APP/DEP"],
        "CTR": ["CTR", "CENTER", "CNTR", "CTRL", "CONTROL", "ACC",
                 "AREA CONTROL", "AREA CTRL", "ACC SECTOR"],
    }

    search_types = role_map.get(role, [])
    if not search_types or not _GLOBAL_NAV_DB.exists():
        return None, icao, None

    try:
        conn = sqlite3.connect(str(_GLOBAL_NAV_DB))
        conn.row_factory = sqlite3.Row

        # Airport-Name holen
        ap_row = conn.execute(
            "SELECT name FROM airports WHERE icao = ?", (icao,)
        ).fetchone()
        airport_name = ap_row["name"] if ap_row else None

        # ── 2) Frequenz in airports/frequencies-Tabelle suchen ────────
        placeholders = ",".join("?" for _ in search_types)
        row = conn.execute(
            f"SELECT mhz FROM frequencies WHERE airport_icao = ?"
            f" AND kind IN ({placeholders})"
            f" AND mhz >= 118.0 AND mhz <= 137.0"
            f" ORDER BY mhz LIMIT 1",
            (icao, *search_types)
        ).fetchone()

        if row:
            conn.close()
            return row["mhz"], icao, airport_name

        # ── 3) Fallback für CTR: firs-Tabelle abfragen ───────────────
        if role == "CTR":
            fir_row = conn.execute(
                "SELECT frequency, name FROM firs WHERE icao = ? AND frequency > 0",
                (icao,)
            ).fetchone()
            if fir_row:
                conn.close()
                return fir_row["frequency"], icao, fir_row["name"] or airport_name

            # Noch ein Versuch: 2-Buchstaben-Prefix (ED → EDGG, EDMM, EDWW)
            prefix = icao[:2]
            fir_prefix_row = conn.execute(
                "SELECT frequency, name, icao FROM firs WHERE icao LIKE ? AND frequency > 0 LIMIT 1",
                (prefix + "%",)
            ).fetchone()
            if fir_prefix_row:
                conn.close()
                return fir_prefix_row["frequency"], icao, fir_prefix_row["name"] or airport_name

        conn.close()
        return None, icao, airport_name
    except Exception:
        return None, icao, None

# Model-Matching  (ICAO-Typ → MSFS-Modellname)
MODEL_MATCH: Dict[str, str] = {
    "A320": "Airbus A320 Neo Asobo",
    "A20N": "Airbus A320 Neo Asobo",
    "B738": "Boeing 737-800 Asobo",
    "B78X": "Boeing 787-10 Asobo",
    "A321": "Airbus A320 Neo Asobo",
    "B744": "Boeing 747-8i Asobo",
    "C172": "Cessna Skyhawk Asobo",
    "DA40": "DA40-NG Asobo",
    "DA62": "DA62 Asobo",
    "TBM9": "TBM 930 Asobo",
    "C208": "Cessna 208B Grand Caravan EX",
    "PC12": "Pilatus PC-12 NGX Asobo",
}


# ---------------------------------------------------------------------------
# LOD – Level of Detail  (Zoom-Stufen → sichtbare Elemente)
# ---------------------------------------------------------------------------
# Zoom-Stufe (OSM-äquivalent)    Sichtbare Elemente            Rolle
# 0  –  5                       Kontinente, FIR-Grenzen          Global
# 6  –  9                       High Airways, VORs               CTR
# 10 – 13                       Fixes, STARs/SIDs, Airport-Pos   APP
# 14+                           Runways, Taxiways, Gates, OSM    GND/TWR

def _osm_zoom_from_scale(scale: float) -> int:
    """Konvertiert den internen Scale-Wert in einen OSM-Zoom-Level."""
    if scale > 100000: return 21
    elif scale > 50000: return 20
    elif scale > 25000: return 19
    elif scale > 12000: return 18
    elif scale > 5000: return 17
    elif scale > 2400: return 16
    elif scale > 1200: return 15
    elif scale > 600: return 14
    elif scale > 300: return 13
    elif scale > 150: return 12
    elif scale > 80: return 11
    elif scale > 40: return 10
    elif scale > 20: return 9
    elif scale > 10: return 8
    elif scale > 5: return 7
    elif scale > 2.5: return 6
    else: return 5

def _lod_show_firs(scale: float) -> bool:
    """FIR-Grenzen: immer sichtbar wenn rausgezoomt (≤ scale 800)."""
    return scale < 800

def _lod_show_high_airways(scale: float) -> bool:
    """High Airways: nur bei Center-Zoom (scale < 30) – sehr dezent."""
    return scale < 30

def _lod_show_low_airways(scale: float) -> bool:
    """Low Airways: APP-Level (scale 20-100)."""
    return 20 <= scale <= 100

def _lod_show_vors(scale: float) -> bool:
    """VOR/VORDME: ab Center-Level (scale < 100)."""
    return scale < 100

def _lod_show_fixes(scale: float) -> bool:
    """Fixes/Waypoints: bei Approach und Center Level sichtbar."""
    return scale <= 300

def _lod_show_airport_detail(scale: float) -> bool:
    """Taxiways, Gates, etc.: ab TWR/GND-Level."""
    return _osm_zoom_from_scale(scale) >= 14

def _lod_show_gate_db(scale: float) -> bool:
    """GateDB-Gates: ab scale > 300 (Ground-Mode, < 5nm Range)."""
    return scale > 300

def _lod_navaid_kinds(scale: float) -> list:
    """Welche Navaid-Arten für den aktuellen Zoom sichtbar sind.
    scale > 500: nur VOR (weit rausgezoomt)
    scale 100-500: VOR + NDB + DME
    scale 20-100: VOR + NDB + DME + FIX (Approach)
    scale < 20: VOR + VORDME (Airways)"""
    if scale > 500:
        return ["VOR", "VORDME"]
    elif scale > 100:
        return ["VOR", "VORDME", "NDB", "DME"]
    elif scale > 20:
        return ["VOR", "VORDME", "NDB", "DME", "FIX"]
    else:
        return ["VOR", "VORDME"]


# ---------------------------------------------------------------------------
# Lat/Lon → Pixel
# ---------------------------------------------------------------------------
def lat_lon_to_pixel(
    lat: float, lon: float,
    center_lat: float, center_lon: float,
    zoom: float,
    screen_w: int, screen_h: int,
) -> tuple[float, float]:
    """Konvertiert GPS → Bildschirm-Pixel (equirect, kompatibel)."""
    km_per_deg_lon = KM_PER_DEG_LAT * math.cos(math.radians(center_lat))
    delta_lat = lat - center_lat
    delta_lon = lon - center_lon
    dy_km = delta_lat * KM_PER_DEG_LAT
    dx_km = delta_lon * km_per_deg_lon
    px_per_km = zoom / KM_PER_DEG_LAT
    x = screen_w / 2 + dx_km * px_per_km
    y = screen_h / 2 - dy_km * px_per_km
    return x, y


# ---------------------------------------------------------------------------
# Piloten-Datenstruktur
# ---------------------------------------------------------------------------
class PilotInfo:
    def __init__(self, callsign: str, lat: float, lon: float, alt: float = 0,
                 dep: str = "", arr: str = "", actype: str = "",
                 route: str = "", cruise_alt: str = "",
                 heading: float | None = None,
                 groundspeed: float = 0, com1_freq: float = 0.0,
                 squawk: int = 2000, squawk_ident: bool = False,
                 controller: str = "", controller_role: str = ""):
        self.callsign = callsign
        self.lat = lat
        self.lon = lon
        self.alt = alt
        self.dep = dep
        self.arr = arr
        self.actype = actype
        self.route = route
        self.cruise_alt = cruise_alt
        self.heading = heading
        self.groundspeed = groundspeed
        self.com1_freq = com1_freq
        self.squawk = squawk
        self.squawk_ident = squawk_ident
        self.controller = controller
        self.controller_role = controller_role

        # Erweiterte Telemetrie
        self.ias: float = 0.0           # Indicated Airspeed (kts)
        self.vs: float = 0.0            # Vertical Speed (ft/min)
        self.sel_alt: float = 0.0       # Selected / MCP Altitude (ft)

        # ATC-zugewiesene Werte
        self.assigned_squawk: int = 0
        self.assigned_alt: str = ""
        self.cleared: bool = False

        # Handover-Blink-State
        self.handover_blink: bool = False
        self.handover_from: str = ""

        # QNH / Kohlsman
        self.kohlsman_mb: float = 0.0

        # Transponder-Modus (STBY, ON, TA, TA/RA)
        self.transponder_mode: str = ""

        # CPDLC-Status
        self.cpdlc_pending: bool = False  # True = offene CPDLC-Nachricht
        self.cpdlc_last_status: str = ""  # Letzter Antwort-Status (WILCO/UNABLE/STANDBY)

        # Geofencing: Sektor-Grenzwarnung
        self.boundary_alert: bool = False  # True = Pilot nahe Sektorgrenze ohne Handover

        # STCA: Short Term Conflict Alert
        self.stca_alert: bool = False       # True = Pilot in Konflikt
        self.stca_partner: str = ""         # Callsign des Konflikt-Partners

        # Event / Slot-Daten
        self.slot_time: str = ""            # UTC-Zeit des gebuchten Slots (ISO)
        self.slot_status: str = ""          # ON_TIME, LATE, EARLY, oder leer

        # Gate-Docking (kommt vom Server)
        self.gate: str = ""                 # Gate-Bezeichnung (z.B. "A12", "B24")
        self.gate_terminal: str = ""        # Terminal (falls bekannt)
        self.gate_dist_m: float = 0.0       # Distanz zum Gate in Metern

        # Contact-Mismatch: Pilot im Zuständigkeitsbereich, falsche Frequenz
        self.contact_mismatch: bool = False  # True = Warn-Icon im Datablock
        self.contact_expected_freq: float = 0.0  # Soll-Frequenz
        self.contacted_flash: float = 0.0  # Zeitstempel für "CONTACTED" Bestätigung


# ---------------------------------------------------------------------------
# Radar-Widget
# ---------------------------------------------------------------------------
class RadarWidget(QWidget):
    """Schwarzes Radar-Widget mit QPixmap-Cache für statische Layer."""

    pilot_right_clicked = pyqtSignal(str, int, int)   # callsign, screen_x, screen_y

    def __init__(self, parent=None):
        super().__init__(parent)
        self.pilots: Dict[str, PilotInfo] = {}
        self.selected_callsign: Optional[str] = None

        # ATC
        self.atc_frequency: float = 0.0
        self.my_station: str = ""

        # Sektordaten
        self.sector: Optional[SectorData] = None
        self.show_artcc = True
        self.show_runways = True
        self.show_navaids = False
        self.show_geo = True
        self.show_taxiways = False
        self.show_sids = False
        self.show_stars = False
        self.show_labels = False
        self.show_map_tiles = False
        self.show_regions = False
        self.show_low_airways = False
        self.show_high_airways = False
        self.show_firs = True            # FIR-Grenzen (global)
        self.show_global_navaids = True  # Navaids aus NavDB (viewport)
        self.show_global_airways = True  # Airways aus NavDB (viewport)
        self.show_sectors = True         # Center-Sektorgrenzen (sectors.geojson)
        self.show_gates = True           # Gate-Positionen aus GateDB

        # Gate-DB (gecachte Gate-Daten pro ICAO)
        self._gate_db_cache: Dict[str, List[GateInfo]] = {}  # icao → gates
        self._occupied_gates: Dict[str, set] = {}  # icao → {ref, ref, ...}

        # Sektordaten – dynamisch vom Server oder Fallback aus GeoJSON
        self._sector_polygons: list[dict] = []
        self._sector_last_viewport: tuple = ()   # (lat_min, lat_max, lon_min, lon_max)
        self._sector_fetch_pending = False
        self._load_sector_geojson()  # Einmalig: statische GeoJSON als Fallback

        # Trend-Vektoren (Projektion in die Zukunft)
        self.show_trend_vectors = True   # Toggle
        self.trend_vector_minutes = [1, 2, 5]  # Projektions-Minuten (wählbar)

        # Rolle
        self.role: str = "CTR"

        # Karte
        self.center_lat = DEFAULT_CENTER_LAT
        self.center_lon = DEFAULT_CENTER_LON
        self.scale = DEFAULT_SCALE

        # Drag
        self._drag_start = None
        self._drag_center_lat = 0.0
        self._drag_center_lon = 0.0

        # Tile Manager (async)
        self._tile_mgr = TileManager()
        self._tile_mgr.set_ready_callback(self._on_tile_ready)

        # Voice: Talking-State (Callsign → True wenn spricht)
        self._talking_users: Dict[str, bool] = {}

        # OSM Overpass (gecachte Airport-Daten)
        self._osm_data: dict[str, OSMData] = {}
        self._osm_icao: Optional[str] = None
        if _OSM_OVERPASS:
            _OSM_OVERPASS.set_callback(self._on_osm_ready)

        # QPixmap-Cache für statische Hintergrundlinien
        self._bg_cache: Optional[QPixmap] = None
        self._bg_cache_key: tuple = ()   # (w, h, center_lat, center_lon, scale, flags)

        # Blink-Zähler (für Handover-Animation, 500ms Toggle)
        self._blink_on = True

        # Wind (für Pfeilanzeige im Overlay)
        self._wind_dir = 0
        self._wind_speed = 0

        # Server-Wetter (METAR-Daten vom ATC-Network)
        self._server_weather: Dict[str, dict] = {}  # icao → parsed metar
        self._metar_bar_text = ""  # Rohtext für METAR-Leiste

        # Smooth-Zoom-Animation
        self._zoom_timer: Optional[QTimer] = None
        self._zoom_target_lat = 0.0
        self._zoom_target_lon = 0.0
        self._zoom_target_scale = 0.0
        self._zoom_steps_left = 0

        # ── Search-Marker: Visueller Indikator nach ICAO-Sprung ──
        self._search_marker_lat: float = 0.0
        self._search_marker_lon: float = 0.0
        self._search_marker_icao: str = ""
        self._search_marker_timer: int = 0  # Countdown-Frames (≈ 3s)
        self._search_marker_freq: float = 0.0   # Suggested Frequency MHz
        self._search_marker_fir: str = ""        # FIR-Name

        self.setMinimumSize(600, 400)
        self.setMouseTracking(True)
        self.setFocusPolicy(Qt.FocusPolicy.StrongFocus)

    # ICAO-Prefix Farbkodierung für dynamische FIR-Sektoren
    _ICAO_COLOR_MAP = {
        "E": "#00ccff",    # Europa (ICAO E-Prefix: ED, EG, EH, EI, EK, ...)
        "L": "#3fb950",    # Südeuropa / Mittelmeer (LF, LI, LS, LE, ...)
        "K": "#58a6ff",    # USA (ICAO K-Prefix)
        "C": "#8957e5",    # Kanada
        "U": "#d29922",    # Russland / GUS
        "Z": "#f85149",    # China
        "R": "#f0883e",    # Japan / Korea
        "V": "#56d364",    # Südasien (Indien, Thailand, ...)
        "W": "#79c0ff",    # Südostasien (Indonesien, ...)
        "Y": "#d2a8ff",    # Australien
        "D": "#ffd700",    # Westafrika
        "H": "#ff7b72",    # Ostafrika
        "F": "#a5d6ff",    # Südafrika
        "S": "#7ee787",    # Südamerika
        "M": "#ff9bce",    # Mittelamerika / Karibik
        "O": "#e3b341",    # Naher Osten
        "B": "#6bc5d2",    # Island / Grönland
    }

    @classmethod
    def _icao_color(cls, icao: str) -> str:
        """Farbcode für einen ICAO-Prefix (erstes Zeichen)."""
        if not icao:
            return "#00ccff"
        return cls._ICAO_COLOR_MAP.get(icao[0].upper(), "#00ccff")

    def _load_sector_geojson(self):
        """Lädt Center-Sektorgrenzen aus sectors.geojson (statischer Fallback)."""
        geojson_path = Path(__file__).parent / "sectors.geojson"
        if not geojson_path.exists():
            return
        try:
            with open(geojson_path, "r", encoding="utf-8") as f:
                data = json.load(f)
            for feature in data.get("features", []):
                props = feature.get("properties", {})
                geom = feature.get("geometry", {})
                if geom.get("type") != "Polygon":
                    continue
                coords = geom.get("coordinates", [])
                if not coords or not coords[0]:
                    continue
                ring = coords[0]
                # GeoJSON = [lon, lat]
                boundary = [(pt[1], pt[0]) for pt in ring]
                lats = [p[0] for p in boundary]
                lons = [p[1] for p in boundary]
                icao = props.get("id", "")
                self._sector_polygons.append({
                    "id": icao,
                    "name": props.get("name", ""),
                    "fir": props.get("fir", ""),
                    "frequency": props.get("frequency", 0),
                    "color": props.get("color", self._icao_color(icao)),
                    "fl_lower": props.get("fl_lower", 0),
                    "fl_upper": props.get("fl_upper", 660),
                    "boundary": boundary,
                    "lat_min": min(lats), "lat_max": max(lats),
                    "lon_min": min(lons), "lon_max": max(lons),
                    "source": "geojson",
                })
            if self._sector_polygons:
                print(f"SectorLoader: {len(self._sector_polygons)} Sektoren geladen (GeoJSON)")
        except Exception as e:
            print(f"⚠ SectorLoader: {e}")

    def _fetch_sector_firs_from_api(self):
        """Holt FIR-Sektoren dynamisch vom Server-API (/api/nav/nearby_firs).

        Wird lazy bei Viewport-Wechsel aufgerufen, ersetzt die alten
        statischen Sektordaten durch weltweite FIR-Polygone.
        """
        if self._sector_fetch_pending:
            return
        lat_min, lat_max, lon_min, lon_max = self._viewport_bounds()

        # Prüfe ob Viewport sich wesentlich geändert hat (>0.5° Toleranz)
        if self._sector_last_viewport:
            olat_min, olat_max, olon_min, olon_max = self._sector_last_viewport
            if (abs(lat_min - olat_min) < 0.5 and abs(lat_max - olat_max) < 0.5 and
                    abs(lon_min - olon_min) < 0.5 and abs(lon_max - olon_max) < 0.5):
                return  # Viewport kaum geändert → kein neuer Fetch

        self._sector_fetch_pending = True
        self._sector_last_viewport = (lat_min, lat_max, lon_min, lon_max)

        # HTTP-URL vom WS-Server ableiten
        parent_win = self.parent()
        while parent_win and not isinstance(parent_win, QMainWindow):
            parent_win = parent_win.parent() if hasattr(parent_win, 'parent') else None
        base_url = "http://localhost:9000"
        if parent_win and hasattr(parent_win, 'server_url'):
            base_url = parent_win.server_url.replace("ws://", "http://").replace(
                "wss://", "https://").split("/ws/")[0]

        center_lat = (lat_min + lat_max) / 2
        center_lon = (lon_min + lon_max) / 2
        radius = max(abs(lat_max - lat_min), abs(lon_max - lon_min)) / 2 + 0.5

        def _do_fetch():
            try:
                url = (f"{base_url}/api/nav/nearby_firs"
                       f"?lat={center_lat:.4f}&lon={center_lon:.4f}"
                       f"&radius={radius:.2f}")
                req = urllib.request.Request(url)
                with urllib.request.urlopen(req, timeout=5) as resp:
                    data = json.loads(resp.read())
                firs = data.get("firs", [])
                new_sectors = []
                for f in firs:
                    poly_json = f.get("polygon_geojson", "")
                    if not poly_json:
                        continue
                    try:
                        geom = json.loads(poly_json)
                    except (json.JSONDecodeError, TypeError):
                        continue
                    coords = geom.get("coordinates", [])
                    if geom.get("type") == "Polygon":
                        rings = [coords[0]] if coords else []
                    elif geom.get("type") == "MultiPolygon":
                        rings = [p[0] for p in coords if p]
                    else:
                        continue
                    for ring in rings:
                        if len(ring) < 3:
                            continue
                        boundary = [(pt[1], pt[0]) for pt in ring]  # [lon,lat]→(lat,lon)
                        lats = [p[0] for p in boundary]
                        lons = [p[1] for p in boundary]
                        icao = f.get("icao", "")
                        new_sectors.append({
                            "id": icao,
                            "name": f.get("name", ""),
                            "fir": icao,
                            "frequency": f.get("frequency", 0),
                            "color": self._icao_color(icao),
                            "fl_lower": 0,
                            "fl_upper": 660,
                            "boundary": boundary,
                            "lat_min": min(lats), "lat_max": max(lats),
                            "lon_min": min(lons), "lon_max": max(lons),
                            "online": f.get("online", False),
                            "source": "api",
                        })
                if new_sectors:
                    # GeoJSON-Sektoren beibehalten, API-Sektoren ersetzen
                    keep = [s for s in self._sector_polygons
                            if s.get("source") == "geojson"]
                    self._sector_polygons = keep + new_sectors
                    self.invalidate_bg_cache()
                    self._fir_api_warned = False  # Reset bei Erfolg
            except Exception as e:
                if not getattr(self, '_fir_api_warned', False):
                    print(f"⚠ FIR-API-Fetch: {e} (weitere Fehler werden unterdrückt)")
                    self._fir_api_warned = True
            finally:
                self._sector_fetch_pending = False

        import threading
        threading.Thread(target=_do_fetch, daemon=True).start()

    def invalidate_bg_cache(self):
        self._bg_cache = None

    def smooth_goto(self, lat: float, lon: float, scale: float, steps: int = 20):
        """Animiert sanftes Zoomen/Pannen zu einem Ziel."""
        self._zoom_target_lat = lat
        self._zoom_target_lon = lon
        self._zoom_target_scale = scale
        self._zoom_steps_left = steps
        if self._zoom_timer is None:
            self._zoom_timer = QTimer(self)
            self._zoom_timer.timeout.connect(self._zoom_step)
        self._zoom_timer.start(30)  # ~33fps

    def _zoom_step(self):
        if self._zoom_steps_left <= 0:
            if self._zoom_timer:
                self._zoom_timer.stop()
            return
        f = 0.2  # Ease-Faktor
        self.center_lat += (self._zoom_target_lat - self.center_lat) * f
        self.center_lon += (self._zoom_target_lon - self.center_lon) * f
        self.scale += (self._zoom_target_scale - self.scale) * f
        self._zoom_steps_left -= 1
        if self._zoom_steps_left <= 0:
            self.center_lat = self._zoom_target_lat
            self.center_lon = self._zoom_target_lon
            self.scale = self._zoom_target_scale
            if self._zoom_timer:
                self._zoom_timer.stop()
        self.invalidate_bg_cache()
        self.update()

    def _bg_cache_valid(self) -> bool:
        key = (
            self.width(), self.height(),
            round(self.center_lat, 6), round(self.center_lon, 6),
            round(self.scale, 2),
            self.show_artcc, self.show_runways, self.show_geo,
            self.show_navaids, self.show_taxiways, self.show_sids,
            self.show_stars, self.show_labels, self.show_map_tiles,
            self.show_regions, self.show_low_airways, self.show_high_airways,
            self.show_firs, self.show_global_navaids, self.show_global_airways,
            self.show_sectors, self.show_gates,
        )
        if self._bg_cache is not None and self._bg_cache_key == key:
            return True
        self._bg_cache_key = key
        return False

    # -- Painting -----------------------------------------------------------
    def paintEvent(self, event):
        p = QPainter(self)
        p.setRenderHint(QPainter.RenderHint.Antialiasing)

        # Statischer Hintergrund (gecacht)
        if self._bg_cache_valid() and self._bg_cache is not None:
            p.drawPixmap(0, 0, self._bg_cache)
        else:
            bg = QPixmap(self.size())
            bg.fill(QColor(5, 5, 5))  # EuroScope Tiefschwarz
            bp = QPainter(bg)
            bp.setRenderHint(QPainter.RenderHint.Antialiasing)

            if self.show_map_tiles:
                self._draw_map_tiles(bp)

            self._draw_grid(bp)

            # ── Globale Layer (FIR, Navaids, Airways aus NavDB) ──
            if self.show_firs and _lod_show_firs(self.scale):
                self._draw_firs(bp)

            if self.show_sectors:
                self._draw_sectors(bp)

            if self.show_global_airways and _NAV_DB:
                self._draw_global_airways(bp)

            if self.show_global_navaids and _NAV_DB:
                self._draw_global_navaids(bp)

            # ── Erzwinge Navaid-Zeichnung bei APP/CTR auch ohne Sektor ──
            if not self.sector and self.show_navaids and _NAV_DB:
                self._draw_global_navaids(bp)

            # ── Sektor-Layer ──
            if self.sector:
                if self.show_geo:
                    self._draw_geo_lines(bp, self.sector.geo_lines, QColor(40, 50, 40))
                if self.show_taxiways and self.sector.taxiway_lines:
                    # Bei GND-Zoom (scale>1000) hellere, dickere Taxiway-Linien
                    twy_color = QColor(30, 140, 220) if self.scale > 1000 else QColor(0, 70, 120)
                    self._draw_geo_lines(bp, self.sector.taxiway_lines, twy_color)
                if self.show_sids and self.sector.sid_lines:
                    self._draw_geo_lines(bp, self.sector.sid_lines, QColor(0, 130, 210, 200))
                if self.show_stars and self.sector.star_lines:
                    self._draw_geo_lines(bp, self.sector.star_lines, QColor(210, 160, 0, 200))
                if self.show_artcc:
                    self._draw_geo_lines(bp, self.sector.artcc_lines, QColor(60, 70, 90))
                    self._draw_geo_lines(bp, self.sector.artcc_high_lines, QColor(50, 60, 80))
                    self._draw_geo_lines(bp, self.sector.artcc_low_lines, QColor(50, 60, 80))
                if self.show_runways:
                    self._draw_runways(bp, self.sector.runways)
                if self.show_navaids:
                    self._draw_navaids(bp, self.sector.all_navaids)
                if self.show_labels and self.sector.labels:
                    self._draw_labels(bp, self.sector.labels)
                if self.show_regions and self.sector.regions:
                    self._draw_regions(bp, self.sector.regions)
                if self.show_low_airways and self.sector.low_airway_lines:
                    self._draw_geo_lines(bp, self.sector.low_airway_lines, QColor(50, 50, 60, 120))
                if self.show_high_airways and self.sector.high_airway_lines:
                    self._draw_geo_lines(bp, self.sector.high_airway_lines, QColor(50, 50, 60, 120))

            # ── OSM Overpass Layer (echte Taxiways/Gates) ──
            if self._osm_data and _lod_show_airport_detail(self.scale):
                for icao, osm_d in self._osm_data.items():
                    self._draw_osm_data(bp, osm_d)

            # ── GateDB-Layer (Gates aus SQLite, wenn kein OSM) ──
            if self.show_gates and _lod_show_gate_db(self.scale):
                # Nur wenn OSM keine Gates hat (Fallback) oder immer für Occupancy
                self._draw_gate_db_gates(bp)

            # ── Runways aus globaler DB ──
            if self.show_runways and self._osm_icao and self.scale >= 150:
                self._draw_db_runways(bp, self._osm_icao)

            bp.end()
            self._bg_cache = bg
            p.drawPixmap(0, 0, bg)

        # Dynamische Layer (Piloten) – immer frisch gezeichnet
        self._pilot_label_rects = []  # Reset Anti-Overlap für jeden Frame
        self._update_occupied_gates()  # Gate-Occupancy für diesen Frame
        for pilot in self.pilots.values():
            self._draw_pilot(p, pilot)

        # Great Circle Linie zum Ziel (nur für selektiertes Flugzeug)
        if self.selected_callsign:
            sel_pilot = self.pilots.get(self.selected_callsign)
            if sel_pilot and sel_pilot.arr and _AIRPORT_DB:
                self._draw_great_circle(p, sel_pilot)

        # ── Search-Marker (visueller Airport-Indikator) ──
        self._draw_search_marker(p)

        self._draw_overlay(p)
        p.end()

    def _draw_grid(self, p: QPainter):
        pen = QPen(QColor(30, 50, 30))
        pen.setWidth(1)
        p.setPen(pen)
        p.setFont(QFont("Consolas", 8))
        w, h = self.width(), self.height()

        step = 1 if self.scale > 30 else 5
        if self.scale < 10:
            step = 10

        cos_lat = math.cos(math.radians(self.center_lat))
        km_per_deg_lon = KM_PER_DEG_LAT * cos_lat
        px_per_km = self.scale / KM_PER_DEG_LAT

        min_lon = self.center_lon - (w / 2) / (px_per_km * km_per_deg_lon)
        max_lon = self.center_lon + (w / 2) / (px_per_km * km_per_deg_lon)
        lon = int(min_lon // step) * step
        while lon <= max_lon:
            x, _ = lat_lon_to_pixel(self.center_lat, lon,
                                    self.center_lat, self.center_lon, self.scale, w, h)
            p.drawLine(int(x), 0, int(x), h)
            p.setPen(QColor(50, 80, 50))
            p.drawText(int(x) + 2, h - 4, f"{lon:.0f}°")
            p.setPen(pen)
            lon += step

        min_lat = self.center_lat - (h / 2) / (px_per_km * KM_PER_DEG_LAT)
        max_lat = self.center_lat + (h / 2) / (px_per_km * KM_PER_DEG_LAT)
        lat = int(min_lat // step) * step
        while lat <= max_lat:
            _, y = lat_lon_to_pixel(lat, self.center_lon,
                                    self.center_lat, self.center_lon, self.scale, w, h)
            p.drawLine(0, int(y), w, int(y))
            p.setPen(QColor(50, 80, 50))
            p.drawText(4, int(y) - 4, f"{lat:.0f}°")
            p.setPen(pen)
            lat += step

    def _draw_pilot(self, p: QPainter, pilot: PilotInfo):
        # Gate-Docking: Wenn Pilot am Gate (< 30m), Position ans Gate snappen
        is_docked = (pilot.gate and pilot.gate_dist_m < 30 and
                     pilot.groundspeed < 5 and pilot.alt < 300)
        if is_docked and _GATE_DB and self._osm_icao:
            # Versuche Gate-Position aus Cache zu holen
            cached = self._gate_db_cache.get(self._osm_icao, [])
            for g in cached:
                if g.ref == pilot.gate:
                    x, y = lat_lon_to_pixel(
                        g.lat, g.lon,
                        self.center_lat, self.center_lon,
                        self.scale, self.width(), self.height(),
                    )
                    break
            else:
                is_docked = False  # Gate nicht im Cache → kein Snapping
                x, y = lat_lon_to_pixel(
                    pilot.lat, pilot.lon,
                    self.center_lat, self.center_lon,
                    self.scale, self.width(), self.height(),
                )
        else:
            x, y = lat_lon_to_pixel(
                pilot.lat, pilot.lon,
                self.center_lat, self.center_lon,
                self.scale, self.width(), self.height(),
            )

        is_selected = pilot.callsign == self.selected_callsign
        is_ident = pilot.squawk_ident
        is_handover_blink = pilot.handover_blink and self._blink_on
        is_boundary_alert = pilot.boundary_alert and self._blink_on
        is_stca = pilot.stca_alert and self._blink_on

        # FL245-Geofencing: CTR-Rolle → unter FL245 kleiner, über FL245 normal
        fl245_ft = 24500
        is_below_fl245 = pilot.alt < fl245_ft
        if self.role == "CTR" and is_below_fl245:
            dot_size = 8 if not is_selected else 12   # klein: unterhalb CTR-Bereich
        else:
            dot_size = 16 if is_selected else 12

        # Squawk-Matching
        squawk_matched = (pilot.assigned_squawk > 0
                          and pilot.assigned_squawk == pilot.squawk)
        squawk_unmatched = (pilot.assigned_squawk > 0
                            and pilot.assigned_squawk != pilot.squawk)

        # Farblogik - EuroScope-Style
        if is_stca:
            color = QColor(255, 0, 0)         # ROT blinkend: STCA Konflikt
        elif is_handover_blink:
            color = QColor(255, 255, 0)       # Gelb blinkend: Handover
        elif is_boundary_alert:
            color = QColor(255, 200, 0)       # Gelb/Orange blinkend: Grenzwarnung
        elif is_ident:
            color = QColor(255, 165, 0)       # Orange: IDENT
        elif self.atc_frequency > 0 and abs(pilot.com1_freq - self.atc_frequency) < 0.01:
            color = QColor(200, 215, 230)     # Stahl-Blau: auf meiner Frequenz
        elif self.atc_frequency > 0:
            color = QColor(60, 60, 60)        # Dunkelgrau: andere Frequenz
        elif is_selected:
            color = QColor(255, 255, 0)       # Gelb: selektiert
        else:
            color = QColor(200, 215, 230)     # Standard-Blau

        # Wenn Controller: eigene = gruen, andere = dunkelgrau
        if self.my_station and pilot.controller:
            if pilot.controller == self.my_station:
                color = QColor(200, 215, 230) # Aktiv bei mir
            else:
                color = QColor(60, 60, 60)    # Passiv

        # Punkt (dünnere Linien) / Quadrat wenn am Gate gedockt
        p.setBrush(QBrush(color))
        p.setPen(Qt.PenStyle.NoPen)
        if is_docked:
            # Quadrat = am Gate geparkt
            p.drawRect(QRectF(x - dot_size/2, y - dot_size/2, dot_size, dot_size))
        else:
            p.drawEllipse(QRectF(x - dot_size/2, y - dot_size/2, dot_size, dot_size))

        # Heading (Linie) + Trend-Vektoren
        if pilot.heading is not None:
            hrad = math.radians(pilot.heading)

            if self.show_trend_vectors and pilot.groundspeed > 10:
                # ── Trend-Vektoren: Projektion 1/2/5 min ──
                # GS in kts → nm/min = GS/60
                nm_per_min = pilot.groundspeed / 60.0
                cos_lat = math.cos(math.radians(pilot.lat))
                km_per_deg_lon = KM_PER_DEG_LAT * cos_lat
                px_per_km = self.scale / KM_PER_DEG_LAT

                prev_px, prev_py = x, y
                for i, minutes in enumerate(self.trend_vector_minutes):
                    nm = nm_per_min * minutes
                    km = nm * 1.852
                    # Projektion in lat/lon
                    proj_lat = pilot.lat + (km / KM_PER_DEG_LAT) * math.cos(hrad)
                    proj_lon = pilot.lon + (km / km_per_deg_lon) * math.sin(hrad)
                    px_proj, py_proj = lat_lon_to_pixel(
                        proj_lat, proj_lon,
                        self.center_lat, self.center_lon,
                        self.scale, self.width(), self.height(),
                    )
                    # Farbe: zunehmend transparenter
                    alpha = max(50, 200 - i * 50)
                    # Handover-Vektoren in Magenta
                    if pilot.handover_blink:
                        vec_color = QColor(255, 0, 255, alpha)   # Magenta
                    elif pilot.stca_alert:
                        vec_color = QColor(255, 0, 0, alpha)     # Rot
                    else:
                        vec_color = QColor(color.red(), color.green(), color.blue(), alpha)
                    pen_w = 1.5 if i == 0 else 1.0
                    p.setPen(QPen(vec_color, pen_w))
                    p.drawLine(int(prev_px), int(prev_py), int(px_proj), int(py_proj))
                    # Tick-Markierung am Ende jedes Segments
                    tick_len = 4
                    perp = hrad + math.pi / 2
                    tx1 = px_proj + tick_len * math.sin(perp)
                    ty1 = py_proj - tick_len * math.cos(perp)
                    tx2 = px_proj - tick_len * math.sin(perp)
                    ty2 = py_proj + tick_len * math.cos(perp)
                    p.drawLine(int(tx1), int(ty1), int(tx2), int(ty2))
                    prev_px, prev_py = px_proj, py_proj
            else:
                # Fallback: einfache 26px Heading-Linie
                line_len = 26
                hx = x + line_len * math.sin(hrad)
                hy = y - line_len * math.cos(hrad)
                pen = QPen(color, 1.5)
                p.setPen(pen)
                p.drawLine(int(x), int(y), int(hx), int(hy))

        # Data Block – Anti-Overlap: berechne Tag-Position + Collision
        tag_x = int(x) + 14
        tag_y = int(y) - 20
        line_h = 12

        # Anti-Overlap: Falls ein Label dort schon existiert, verschiebe nach unten
        font_main = QFont("Consolas", 8)
        fm = p.fontMetrics()
        block_w = 120
        block_h = line_h * 3 + 4
        label_rect = QRectF(tag_x, tag_y - 2, block_w, block_h)

        # Prüfe gegen bereits gezeichnete Labels
        if not hasattr(self, '_pilot_label_rects'):
            self._pilot_label_rects = []
        max_shift = 5  # max 5 Versuche nach unten
        for _ in range(max_shift):
            collision = False
            for existing_rect in self._pilot_label_rects:
                if label_rect.intersects(existing_rect):
                    collision = True
                    break
            if not collision:
                break
            tag_y += 20  # 20px Offset nach unten
            label_rect = QRectF(tag_x, tag_y - 2, block_w, block_h)
        self._pilot_label_rects.append(label_rect)

        # Hintergrund-Box mit 80% Transparenz
        p.setBrush(QBrush(QColor(5, 5, 5, 200)))  # 80% opak schwarz
        p.setPen(Qt.PenStyle.NoPen)
        p.drawRect(label_rect)

        # Leader Line (vom Dot zum Data Block)
        p.setPen(QPen(QColor(50, 50, 50), 1))
        p.drawLine(int(x) + dot_size // 2, int(y), tag_x - 1, tag_y + line_h)

        # Zeile 1: CALLSIGN  (+UNCORRELATED)
        font = QFont("Consolas", 8)
        font.setBold(is_selected or is_ident or squawk_matched)
        p.setFont(font)
        p.setPen(color)
        cs_extra = ""
        if squawk_unmatched:
            cs_extra = " UNCRL"
            p.setPen(QColor(255, 80, 80))

        # Talking-LED: grüner Kreis neben dem Callsign wenn Pilot spricht
        is_talking = self._talking_users.get(pilot.callsign, False)
        if is_talking:
            p.setBrush(QBrush(QColor(63, 185, 80)))
            p.setPen(QPen(QColor(63, 185, 80, 160), 1))
            p.drawEllipse(tag_x - 7, tag_y - 8, 6, 6)
            p.setPen(color)

        p.drawText(tag_x + 2, tag_y, pilot.callsign + cs_extra)

        # CPDLC-Envelope-Icon: ✉ neben Callsign wenn offene Nachricht
        if pilot.cpdlc_pending:
            cs_width = p.fontMetrics().horizontalAdvance(pilot.callsign + cs_extra)
            env_x = tag_x + 4 + cs_width
            # Blinkeffekt: gelb / orange im Wechsel
            import time as _t_mod
            blink = int(_t_mod.time() * 2) % 2
            env_color = QColor(255, 200, 0) if blink else QColor(255, 140, 0)
            p.setPen(env_color)
            p.setFont(QFont("Consolas", 7, QFont.Weight.Bold))
            p.drawText(env_x, tag_y, "✉")
            p.setFont(font)
            p.setPen(color)
        elif pilot.cpdlc_last_status:
            cs_width = p.fontMetrics().horizontalAdvance(pilot.callsign + cs_extra)
            env_x = tag_x + 4 + cs_width
            stat_colors = {"WILCO": QColor(0, 200, 80), "UNABLE": QColor(255, 80, 80),
                           "STANDBY": QColor(0, 180, 255)}
            sc = stat_colors.get(pilot.cpdlc_last_status, QColor(150, 150, 150))
            p.setPen(sc)
            p.setFont(QFont("Consolas", 6))
            p.drawText(env_x, tag_y, pilot.cpdlc_last_status[:1])
            p.setFont(font)
            p.setPen(color)

        # STCA-Tag: rot blinkend neben Callsign
        if pilot.stca_alert:
            cs_width_full = p.fontMetrics().horizontalAdvance(pilot.callsign + cs_extra)
            stca_x = tag_x + 6 + cs_width_full
            if pilot.cpdlc_pending or pilot.cpdlc_last_status:
                stca_x += 14  # Platz für CPDLC-Icon
            if self._blink_on:
                p.setPen(QColor(255, 0, 0))
                p.setFont(QFont("Consolas", 7, QFont.Weight.Bold))
                p.drawText(stca_x, tag_y, "STCA")
                p.setFont(font)
                p.setPen(color)

        # Contact-Confirmed: grünes "OK" neben Callsign (3 Sekunden)
        import time as _t_now
        if pilot.contacted_flash and (_t_now.time() - pilot.contacted_flash) < 3.0:
            cs_width_full = p.fontMetrics().horizontalAdvance(pilot.callsign + cs_extra)
            ok_x = tag_x + 6 + cs_width_full
            if pilot.cpdlc_pending or pilot.cpdlc_last_status:
                ok_x += 14
            if pilot.stca_alert:
                ok_x += 32
            p.setPen(QColor(63, 185, 80))  # Grün
            p.setFont(QFont("Consolas", 7, QFont.Weight.Bold))
            p.drawText(ok_x, tag_y, "✓CONTACT")
            p.setFont(font)
            p.setPen(color)
        elif pilot.contacted_flash and (_t_now.time() - pilot.contacted_flash) >= 3.0:
            pilot.contacted_flash = 0.0  # Flash abgelaufen

        # Contact-Mismatch: gelbes Warn-Icon ⚠ neben Callsign
        # (Pilot im eigenen Bereich, falsche Frequenz)
        if pilot.contact_mismatch and self._blink_on:
            cs_width_full = p.fontMetrics().horizontalAdvance(pilot.callsign + cs_extra)
            warn_x = tag_x + 6 + cs_width_full
            # Versatz wenn CPDLC oder STCA-Icons davor
            if pilot.cpdlc_pending or pilot.cpdlc_last_status:
                warn_x += 14
            if pilot.stca_alert:
                warn_x += 32  # Platz für "STCA"
            p.setPen(QColor(255, 220, 0))  # Gelb
            p.setFont(QFont("Consolas", 8, QFont.Weight.Bold))
            p.drawText(warn_x, tag_y, "⚠")
            # Soll-Frequenz klein daneben
            if pilot.contact_expected_freq > 0:
                p.setFont(QFont("Consolas", 6))
                p.setPen(QColor(255, 200, 0, 180))
                p.drawText(warn_x + 12, tag_y,
                           f"{pilot.contact_expected_freq:.3f}")
            p.setFont(font)
            p.setPen(color)

        # Zeile 2: ALT / GS / IAS  (+ VS-Pfeil)  –  oder Gate wenn gedockt
        alt_hundreds = int(pilot.alt / 100)
        gs = int(pilot.groundspeed)
        ias = int(pilot.ias) if pilot.ias else 0
        vs = int(pilot.vs) if pilot.vs else 0

        if is_docked and pilot.gate:
            # Am Gate: "G-B24" statt Höhe
            term_tag = f"T{pilot.gate_terminal}" if pilot.gate_terminal else ""
            line2 = f"G-{pilot.gate} {term_tag}".strip()
            dim = QColor(255, 165, 0)  # Orange für Gate-Info
        else:
            # VS-Pfeil: ↑ steigend, ↓ sinkend, = level
            if vs > 200:
                vs_arrow = "↑"
            elif vs < -200:
                vs_arrow = "↓"
            else:
                vs_arrow = "="
            if ias > 0:
                line2 = f"{alt_hundreds:03d}{vs_arrow} N{ias:03d} G{gs:03d}"
            else:
                line2 = f"{alt_hundreds:03d}{vs_arrow} / {gs:03d}"
            dim = QColor(int(color.red() * 0.7), int(color.green() * 0.7),
                          int(color.blue() * 0.7))
        small_font = QFont("Consolas", 7)
        p.setFont(small_font)
        p.setPen(dim)
        p.drawText(tag_x + 2, tag_y + line_h, line2)

        # Zeile 3: SQUAWK / FREQ (+ Controller-Station)
        freq_str = f"{pilot.com1_freq:.2f}" if pilot.com1_freq > 0 else "---"
        sq_str = f"{pilot.squawk:04d}"
        if pilot.assigned_squawk > 0:
            sq_str += f"→{pilot.assigned_squawk:04d}"
        ctrl_tag = ""
        if pilot.controller and pilot.controller != self.my_station:
            # Kurzform: z.B. "EDGG_A" statt "EDGG_A_CTR"
            ctrl_tag = f" {pilot.controller.replace('_CTR', '').replace('_APP', '').replace('_TWR', '')}"
        line3 = f"{sq_str} / {freq_str}{ctrl_tag}"
        p.drawText(tag_x + 2, tag_y + line_h * 2, line3)

        # Zeile 4: Assigned Alt + Cleared + XPDR Mode + SEL ALT
        if pilot.assigned_alt or pilot.cleared or pilot.transponder_mode or pilot.sel_alt > 0:
            parts = []
            if pilot.assigned_alt:
                parts.append(f"↑{pilot.assigned_alt}")
            if pilot.sel_alt > 0:
                sel_fl = int(pilot.sel_alt / 100)
                parts.append(f"SEL:{sel_fl:03d}")
            if pilot.cleared:
                parts.append("CLR")
            if pilot.transponder_mode:
                parts.append(pilot.transponder_mode)
            p.setPen(QColor(0, 200, 255))
            p.drawText(tag_x + 2, tag_y + line_h * 3, " ".join(parts))

        # Zeile 5: QNH-Warnung
        if pilot.kohlsman_mb > 0 and hasattr(self, '_wind_dir'):
            parent_win = self.parent()
            while parent_win and not isinstance(parent_win, QMainWindow):
                parent_win = parent_win.parent() if hasattr(parent_win, 'parent') else None
            if parent_win and hasattr(parent_win, '_current_metar'):
                mt = parent_win._current_metar
                if mt and mt.qnh and abs(pilot.kohlsman_mb - mt.qnh) > 2:
                    warn_color = QColor(255, 50, 50)
                    p.setPen(warn_color)
                    p.setFont(QFont("Consolas", 7, QFont.Weight.Bold))
                    p.drawText(tag_x + 2, tag_y + line_h * 4,
                               f"⚠QNH {pilot.kohlsman_mb:.0f}≠{mt.qnh:.0f}")

        # Zeile 6: Event-Slot (grün=pünktlich, gelb=verspätet)
        if pilot.slot_time:
            try:
                slot_short = pilot.slot_time[11:16] + "Z" if len(pilot.slot_time) > 16 else pilot.slot_time
            except Exception:
                slot_short = pilot.slot_time
            slot_label = f"SLOT {slot_short}"
            if pilot.slot_status == "LATE":
                slot_color = QColor(255, 200, 0)  # gelb
            elif pilot.slot_status == "EARLY":
                slot_color = QColor(100, 200, 255)  # hellblau
            else:
                slot_color = QColor(0, 220, 100)  # grün
            p.setPen(slot_color)
            p.setFont(QFont("Consolas", 7, QFont.Weight.Bold))
            p.drawText(tag_x + 2, tag_y + line_h * 5, slot_label)

    # -- Sektor-Zeichnung ---------------------------------------------------
    def _draw_geo_lines(self, p: QPainter, lines: List[GeoLine], color: QColor):
        pen = QPen(color)
        # Bei hohem Zoom (GND-Level) dickere Linien für bessere Sichtbarkeit
        pen.setWidth(2 if self.scale > 800 else 1)
        p.setPen(pen)
        w, h = self.width(), self.height()
        for gl in lines:
            x1, y1 = lat_lon_to_pixel(gl.start.lat, gl.start.lon,
                                       self.center_lat, self.center_lon, self.scale, w, h)
            x2, y2 = lat_lon_to_pixel(gl.end.lat, gl.end.lon,
                                       self.center_lat, self.center_lon, self.scale, w, h)
            if (-500 < x1 < w+500 and -500 < y1 < h+500) or \
               (-500 < x2 < w+500 and -500 < y2 < h+500):
                p.drawLine(int(x1), int(y1), int(x2), int(y2))

    def _draw_runways(self, p: QPainter, runways: List[Runway]):
        w, h = self.width(), self.height()
        for rwy in runways:
            x1, y1 = lat_lon_to_pixel(rwy.start.lat, rwy.start.lon,
                                       self.center_lat, self.center_lon, self.scale, w, h)
            x2, y2 = lat_lon_to_pixel(rwy.end.lat, rwy.end.lon,
                                       self.center_lat, self.center_lon, self.scale, w, h)
            if not ((-500 < x1 < w+500 and -500 < y1 < h+500) or
                    (-500 < x2 < w+500 and -500 < y2 < h+500)):
                continue

            # Runway-Richtung und Breite berechnen
            dx, dy = x2 - x1, y2 - y1
            length = max(1.0, (dx*dx + dy*dy) ** 0.5)
            nx, ny = -dy / length, dx / length
            rwy_width_m = 45
            if _AIRPORT_DB and hasattr(rwy, 'width_m') and rwy.width_m > 0:
                rwy_width_m = rwy.width_m
            half_w_px = max(2.0, rwy_width_m * self.scale / (KM_PER_DEG_LAT * 1000) * 0.5)

            # Runway-Rechteck – gut sichtbar
            pts = [
                QPointF(x1 + nx * half_w_px, y1 + ny * half_w_px),
                QPointF(x2 + nx * half_w_px, y2 + ny * half_w_px),
                QPointF(x2 - nx * half_w_px, y2 - ny * half_w_px),
                QPointF(x1 - nx * half_w_px, y1 - ny * half_w_px),
            ]
            p.setBrush(QBrush(QColor(70, 70, 78, 250)))   # Deutlich helles Grau
            p.setPen(QPen(QColor(140, 140, 150), 2.0))     # Kräftiger Rand
            p.drawPolygon(QPolygonF(pts))

            # Weiße Mittellinie – sehr kräftig
            center_pen = QPen(QColor(255, 255, 255, 230))
            center_pen.setStyle(Qt.PenStyle.DashLine)
            center_pen.setWidthF(1.5)
            p.setPen(center_pen)
            p.drawLine(int(x1), int(y1), int(x2), int(y2))

            # Schwellen-Nummern
            if self.scale > 40:
                p.setPen(QColor(255, 255, 255))
                p.setFont(QFont("Consolas", 11, QFont.Weight.Bold))
                p.drawText(int(x1) - 14, int(y1) - 8, rwy.name1)
                p.drawText(int(x2) + 6, int(y2) + 14, rwy.name2)

    def _draw_db_runways(self, p: QPainter, icao: str):
        """Zeichnet Runways aus der globalen Airport-DB – Anthrazit-Style.
        Wind-Farbcodierung: Grün=Gegenwind, Rot=Rückenwind, Grau=kein Wind."""
        if not _AIRPORT_DB:
            return
        ap = _AIRPORT_DB.get(icao)
        if not ap or not ap.runways:
            return
        w, h = self.width(), self.height()
        wind_dir = self._wind_dir
        wind_spd = self._wind_speed

        for rwy in ap.runways:
            if rwy.le_lat == 0 and rwy.le_lon == 0:
                continue
            if rwy.he_lat == 0 and rwy.he_lon == 0:
                continue

            x1, y1 = lat_lon_to_pixel(rwy.le_lat, rwy.le_lon,
                                       self.center_lat, self.center_lon, self.scale, w, h)
            x2, y2 = lat_lon_to_pixel(rwy.he_lat, rwy.he_lon,
                                       self.center_lat, self.center_lon, self.scale, w, h)
            if not ((-500 < x1 < w+500 and -500 < y1 < h+500) or
                    (-500 < x2 < w+500 and -500 < y2 < h+500)):
                continue

            dx, dy = x2 - x1, y2 - y1
            pxlen = max(1.0, (dx*dx + dy*dy) ** 0.5)
            nx, ny = -dy / pxlen, dx / pxlen

            half_w_px = max(2.0, rwy.width_m * self.scale / (KM_PER_DEG_LAT * 1000) * 0.5)
            if half_w_px < 1.5:
                half_w_px = 1.5

            pts = [
                QPointF(x1 + nx * half_w_px, y1 + ny * half_w_px),
                QPointF(x2 + nx * half_w_px, y2 + ny * half_w_px),
                QPointF(x2 - nx * half_w_px, y2 - ny * half_w_px),
                QPointF(x1 - nx * half_w_px, y1 - ny * half_w_px),
            ]

            # ── Wind-abhängige Farbgebung ──
            fill = QColor(65, 65, 70, 240)  # Default
            rwy_border = QColor(140, 140, 150)
            le_color = QColor(255, 255, 255)
            he_color = QColor(255, 255, 255)

            if wind_spd >= 3:
                # Runway-Heading berechnen (LE → HE)
                rwy_hdg = math.degrees(math.atan2(
                    rwy.he_lon - rwy.le_lon,
                    rwy.he_lat - rwy.le_lat)) % 360
                rwy_hdg_rev = (rwy_hdg + 180) % 360

                # Headwind-Komponente für LE-Schwelle (Landung aus LE-Richtung)
                # Wind kommt AUS wind_dir → Gegenwind wenn Diff nahe 0°
                diff_le = abs((wind_dir - rwy_hdg_rev + 180) % 360 - 180)
                diff_he = abs((wind_dir - rwy_hdg + 180) % 360 - 180)

                # LE-Schwelle: Gegenwind wenn Wind aus der Gegenrichtung kommt
                if diff_le < 70:
                    le_color = QColor(50, 220, 50)   # Grün = Gegenwind (gut)
                elif diff_le > 110:
                    le_color = QColor(220, 50, 50)   # Rot = Rückenwind (schlecht)
                else:
                    le_color = QColor(220, 220, 50)  # Gelb = Seitenwind

                if diff_he < 70:
                    he_color = QColor(50, 220, 50)
                elif diff_he > 110:
                    he_color = QColor(220, 50, 50)
                else:
                    he_color = QColor(220, 220, 50)

                # Runway-Füllung: Grün/Rot je nach bester Richtung
                best_diff = min(diff_le, diff_he)
                if best_diff < 70:
                    fill = QColor(25, 60, 30, 220)    # Dunkelgrün
                    rwy_border = QColor(50, 180, 50)
                elif best_diff > 110:
                    fill = QColor(60, 25, 25, 220)    # Dunkelrot
                    rwy_border = QColor(180, 50, 50)
                else:
                    fill = QColor(60, 55, 20, 220)    # Dunkelgelb
                    rwy_border = QColor(180, 180, 50)
            else:
                # Oberfläche → Standard-Farbschema
                surface = (rwy.surface or "").upper()
                if "ASP" in surface or "CON" in surface or "BIT" in surface:
                    fill = QColor(70, 70, 78, 250)
                elif "GRS" in surface or "GRASS" in surface:
                    fill = QColor(35, 70, 35, 220)

            p.setBrush(QBrush(fill))
            p.setPen(QPen(rwy_border, 2.0))
            p.drawPolygon(QPolygonF(pts))

            # Weiße Mittellinie – sehr kräftig
            center = QPen(QColor(255, 255, 255, 230))
            center.setStyle(Qt.PenStyle.DashLine)
            center.setWidthF(1.5)
            p.setPen(center)
            p.drawLine(int(x1), int(y1), int(x2), int(y2))

            # Schwellen-Nummern (farbcodiert nach Wind)
            if self.scale > 100:
                p.setFont(QFont("Consolas", 11, QFont.Weight.Bold))
                p.setPen(le_color)
                p.drawText(int(x1) - 14, int(y1) - 8, rwy.le_ident)
                p.setPen(he_color)
                p.drawText(int(x2) + 6, int(y2) + 14, rwy.he_ident)
                if self.scale > 400:
                    mx, my = (x1+x2)/2, (y1+y2)/2
                    p.setFont(QFont("Consolas", 8))
                    p.setPen(QColor(200, 200, 200, 220))
                    p.drawText(int(mx)+6, int(my)-8,
                               f"{rwy.length_m}m x {rwy.width_m}m")

    def _draw_navaids(self, p: QPainter, navaids: List[Navaid]):
        """Sektor-Navaids – gut sichtbar auf dunklem Hintergrund."""
        w, h = self.width(), self.height()
        for nav in navaids:
            x, y = lat_lon_to_pixel(nav.position.lat, nav.position.lon,
                                     self.center_lat, self.center_lon, self.scale, w, h)
            if not (0 < x < w and 0 < y < h):
                continue
            if nav.kind == "VOR":
                color, size = QColor(0, 180, 220), 8
                p.setPen(QPen(color, 1.5))
                p.setBrush(Qt.BrushStyle.NoBrush)
                p.drawEllipse(QRectF(x-size, y-size, size*2, size*2))
                p.setBrush(QBrush(QColor(0, 220, 255)))
                p.setPen(Qt.PenStyle.NoPen)
                p.drawEllipse(QRectF(x-2, y-2, 4, 4))
            elif nav.kind == "NDB":
                color, size = QColor(180, 120, 255), 7
                p.setBrush(QBrush(color))
                p.setPen(Qt.PenStyle.NoPen)
                p.drawEllipse(QRectF(x-2.5, y-2.5, 5, 5))
                p.setPen(QPen(color, 1))
                for ang in (0, 60, 120, 180, 240, 300):
                    rad = math.radians(ang)
                    p.drawLine(int(x+3*math.cos(rad)), int(y+3*math.sin(rad)),
                               int(x+size*math.cos(rad)), int(y+size*math.sin(rad)))
            else:
                # FIX/Waypoint – deutlich sichtbar
                color, size = QColor(0, 200, 120), 6
                tri = QPolygonF([
                    QPointF(x, y - size),
                    QPointF(x - size * 0.87, y + size * 0.5),
                    QPointF(x + size * 0.87, y + size * 0.5),
                ])
                p.setPen(QPen(color, 1.5))
                p.setBrush(QBrush(QColor(0, 200, 120, 40)))
                p.drawPolygon(tri)
            # Label – gut lesbar
            if self.scale > 40:
                p.setPen(QColor(180, 200, 220))
                p.setFont(QFont("Consolas", 8, QFont.Weight.Bold))
                p.drawText(int(x) + size + 3, int(y) + 4, nav.ident)

    def _draw_labels(self, p: QPainter, labels: List[Label]):
        """Sektor-Labels – bei hohem Zoom (GND) größer und heller."""
        if self.scale < 200:  # Erst bei hohem Zoom anzeigen
            return
        w, h = self.width(), self.height()
        # Bei GND-Level (scale > 1000) größere, hellere Labels
        if self.scale > 1000:
            p.setFont(QFont("Consolas", 9, QFont.Weight.Bold))
            label_color = QColor(120, 160, 200)  # Hellblau
            gate_color = QColor(255, 165, 0)    # Orange für Gates
        else:
            p.setFont(QFont("Consolas", 7))
            label_color = QColor(68, 68, 68)
            gate_color = QColor(255, 140, 0)
        used_rects: list = []
        for lb in labels:
            x, y = lat_lon_to_pixel(lb.position.lat, lb.position.lon,
                                     self.center_lat, self.center_lon, self.scale, w, h)
            if 0 < x < w and 0 < y < h:
                fm = p.fontMetrics()
                tw = fm.horizontalAdvance(lb.text)
                th = fm.height()
                r = QRectF(x, y - th, tw, th)
                overlap = False
                for ur in used_rects:
                    if r.intersects(ur):
                        overlap = True
                        break
                if overlap:
                    continue
                used_rects.append(r)
                # Gate-Labels gelb, Rest standard
                is_gate = lb.text.startswith("Gate") or lb.text.isdigit()
                p.setPen(gate_color if is_gate else label_color)
                p.drawText(int(x), int(y), lb.text)

    def _draw_regions(self, p: QPainter, regions: List[Region]):
        """Draw filled polygonal regions (terminals, aprons, buildings)."""
        w, h = self.width(), self.height()
        for reg in regions:
            if len(reg.points) < 3:
                continue
            # Convert all points
            pts = []
            any_visible = False
            for gp in reg.points:
                x, y = lat_lon_to_pixel(gp.lat, gp.lon,
                                        self.center_lat, self.center_lon, self.scale, w, h)
                pts.append(QPointF(x, y))
                if -500 < x < w + 500 and -500 < y < h + 500:
                    any_visible = True
            if not any_visible:
                continue
            poly = QPolygonF(pts)
            # Parse color if available, default to dark gray fill
            fill = QColor(80, 80, 90, 120)
            outline = QColor(100, 100, 110)
            if reg.color:
                parts = reg.color.strip().split()
                if len(parts) >= 3:
                    try:
                        r, g, b = int(parts[0]), int(parts[1]), int(parts[2])
                        fill = QColor(r, g, b, 100)
                        outline = QColor(min(r + 40, 255), min(g + 40, 255), min(b + 40, 255))
                    except ValueError:
                        pass
            p.setBrush(QBrush(fill))
            pen = QPen(outline)
            pen.setWidth(1)
            p.setPen(pen)
            p.drawPolygon(poly)
            # Draw region name at centroid if zoomed in enough
            if self.scale > 200 and reg.name:
                cx = sum(pt.x() for pt in pts) / len(pts)
                cy = sum(pt.y() for pt in pts) / len(pts)
                p.setPen(QColor(180, 180, 180))
                p.setFont(QFont("Consolas", 7))
                p.drawText(int(cx), int(cy), reg.name)

    # -- Globale Layer (FIR, Navaids, Airways) ──────────────────────────

    def _viewport_bounds(self) -> tuple[float, float, float, float]:
        """Gibt (lat_min, lat_max, lon_min, lon_max) des sichtbaren Bereichs zurück."""
        w, h = self.width(), self.height()
        px_per_km = self.scale / KM_PER_DEG_LAT
        cos_lat = math.cos(math.radians(self.center_lat))
        km_per_deg_lon = KM_PER_DEG_LAT * cos_lat
        half_lat = (h / 2) / (px_per_km * KM_PER_DEG_LAT)
        half_lon = (w / 2) / (px_per_km * km_per_deg_lon)
        return (self.center_lat - half_lat, self.center_lat + half_lat,
                self.center_lon - half_lon, self.center_lon + half_lon)

    def _draw_firs(self, p: QPainter):
        """Zeichnet FIR-Grenzen als echte Polygone – sichtbar auf dunklem Hintergrund.
        Durchgehende Linien, eigener Sektor hervorgehoben."""
        if not _FIR_MGR:
            return
        lat_min, lat_max, lon_min, lon_max = self._viewport_bounds()
        firs = _FIR_MGR.in_viewport(lat_min, lat_max, lon_min, lon_max)
        w, h = self.width(), self.height()

        # Sichtbare Linien – grau/blau getönt
        pen = QPen(QColor(50, 60, 80))  # Dezentes Blaugrau
        pen.setWidth(1)
        pen.setStyle(Qt.PenStyle.SolidLine)
        p.setPen(pen)
        p.setBrush(Qt.BrushStyle.NoBrush)

        # Eigener Sektor (wenn eingeloggt)
        my_icao = ""
        parent_win = self.parent()
        while parent_win and not isinstance(parent_win, QMainWindow):
            parent_win = parent_win.parent() if hasattr(parent_win, 'parent') else None
        if parent_win and hasattr(parent_win, 'txt_station'):
            st = parent_win.txt_station.text().strip()
            if st and '_' in st:
                my_icao = st.split('_')[0].upper()

        for fir in firs:
            if len(fir.boundary) < 3:
                continue
            pts = []
            for lat, lon in fir.boundary:
                x, y = lat_lon_to_pixel(lat, lon, self.center_lat,
                                        self.center_lon, self.scale, w, h)
                pts.append(QPointF(x, y))
            poly = QPolygonF(pts)

            # Eigener FIR-Bereich hervorheben
            is_my_sector = my_icao and (fir.icao.startswith(my_icao[:2]) or fir.icao == my_icao)
            if is_my_sector:
                # Glow: breite helle Linie + Füllung
                glow = QPen(QColor(0, 150, 255, 50), 5)
                glow.setStyle(Qt.PenStyle.SolidLine)
                p.setPen(glow)
                p.setBrush(QBrush(QColor(0, 120, 255, 20)))
                p.drawPolygon(poly)
                # Scharfe Hauptlinie
                my_pen = QPen(QColor(40, 140, 255, 180), 2)
                my_pen.setStyle(Qt.PenStyle.SolidLine)
                p.setPen(my_pen)
                p.setBrush(Qt.BrushStyle.NoBrush)
                p.drawPolygon(poly)
            else:
                p.setPen(pen)
                p.setBrush(Qt.BrushStyle.NoBrush)
                p.drawPolygon(poly)

            # Label: nur eigener Sektor groß, andere dezent
            cx = sum(pt.x() for pt in pts) / len(pts)
            cy = sum(pt.y() for pt in pts) / len(pts)
            if 0 < cx < w and 0 < cy < h:
                if is_my_sector:
                    # Eigener Sektor: groß und gut sichtbar
                    p.setPen(QColor(60, 100, 180, 160))
                    p.setFont(QFont("Consolas", 18, QFont.Weight.Bold))
                    fm = p.fontMetrics()
                    txt = fir.name.upper() if fir.name else fir.icao
                    tw = fm.horizontalAdvance(txt)
                    p.drawText(int(cx) - tw // 2, int(cy) + 6, txt)
                else:
                    # Andere Sektoren: dezent aber lesbar
                    p.setPen(QColor(60, 70, 80))
                    p.setFont(QFont("Consolas", 8))
                    p.drawText(int(cx) - 12, int(cy) + 3, fir.icao)

    def _draw_sectors(self, p: QPainter):
        """Zeichnet Center-Sektorgrenzen (dynamisch aus API + statisch aus GeoJSON).

        Cyan/Orange/Grüne Polygonlinien mit transparenter Füllung, Beschriftung
        mit Sektor-ID und Frequenz. Online-Sektoren werden hervorgehoben.
        ICAO-Prefix Farbkodierung für weltweite Unterscheidbarkeit.
        """
        # Lazy-Fetch: bei Viewport-Wechsel neue FIR-Daten vom Server holen
        self._fetch_sector_firs_from_api()

        if not self._sector_polygons:
            return
        lat_min, lat_max, lon_min, lon_max = self._viewport_bounds()
        w, h = self.width(), self.height()

        # Online ATC-Stationen ermitteln (für Hervorhebung)
        parent_win = self.parent()
        while parent_win and not isinstance(parent_win, QMainWindow):
            parent_win = parent_win.parent() if hasattr(parent_win, 'parent') else None
        online_stations: set[str] = set()
        if parent_win and hasattr(parent_win, '_online_atc_stations'):
            for atc_st in parent_win._online_atc_stations:
                online_stations.add(atc_st.get('station', ''))

        # Eigene Station
        my_station = self.my_station

        for sec in self._sector_polygons:
            # Viewport-Culling
            if (sec["lat_max"] < lat_min or sec["lat_min"] > lat_max or
                    sec["lon_max"] < lon_min or sec["lon_min"] > lon_max):
                continue

            boundary = sec["boundary"]
            if len(boundary) < 3:
                continue

            is_online = (sec["id"] in online_stations or
                        sec.get("online", False))
            # Flexible Matching: EDGG_A_CTR ↔ EDGG, oder EDGG_CTR ↔ EDGG
            is_mine = False
            if my_station:
                sid = sec["id"]
                # Exakt
                if sid == my_station:
                    is_mine = True
                else:
                    # Prefix-Match: Station EDGG_A_CTR → ICAO EDGG
                    my_icao = my_station.split("_")[0]
                    sec_icao = sid.split("_")[0]
                    if my_icao == sec_icao:
                        is_mine = True
                    # FIR-Match: sec.fir == ICAO-Prefix
                    elif sec.get("fir", "") == my_icao:
                        is_mine = True

            # Farbe: ICAO-Prefix-Kodierung oder GeoJSON-Eigenschaft
            try:
                base_color = QColor(sec.get("color", self._icao_color(sec["id"])))
            except Exception:
                base_color = QColor(0, 200, 255)

            # Polygon-Punkte berechnen
            pts = []
            for lat, lon in boundary:
                x, y = lat_lon_to_pixel(lat, lon, self.center_lat,
                                        self.center_lon, self.scale, w, h)
                pts.append(QPointF(x, y))
            poly = QPolygonF(pts)

            # Füllung: transparent getönt
            if is_mine:
                fill = QColor(base_color.red(), base_color.green(), base_color.blue(), 45)
            elif is_online:
                fill = QColor(base_color.red(), base_color.green(), base_color.blue(), 20)
            else:
                fill = QColor(base_color.red(), base_color.green(), base_color.blue(), 6)

            p.setBrush(QBrush(fill))

            # Linienfarbe
            if is_mine:
                # Glow-Effekt: breiter, heller Hintergrund + scharfe Linie
                glow_pen = QPen(QColor(base_color.red(), base_color.green(), base_color.blue(), 60), 6)
                glow_pen.setStyle(Qt.PenStyle.SolidLine)
                p.setPen(glow_pen)
                p.drawPolygon(poly)
                pen = QPen(QColor(base_color.red(), base_color.green(), base_color.blue(), 230), 3)
                pen.setStyle(Qt.PenStyle.SolidLine)
            elif is_online:
                pen = QPen(QColor(base_color.red(), base_color.green(), base_color.blue(), 140), 1.5)
                pen.setStyle(Qt.PenStyle.SolidLine)
            else:
                pen = QPen(QColor(base_color.red(), base_color.green(), base_color.blue(), 50), 1)
                pen.setStyle(Qt.PenStyle.DashLine)

            p.setPen(pen)
            p.drawPolygon(poly)

            # Label im Zentrum
            cx = sum(pt.x() for pt in pts) / len(pts)
            cy = sum(pt.y() for pt in pts) / len(pts)
            if not (0 < cx < w and 0 < cy < h):
                continue

            freq = sec.get("frequency", 0)
            name = sec.get("name", sec["id"])

            if is_mine:
                # Eigener Sektor: groß + Frequenz + helle Farbe
                p.setPen(QColor(base_color.red(), base_color.green(), base_color.blue(), 220))
                p.setFont(QFont("Consolas", 16, QFont.Weight.Bold))
                fm = p.fontMetrics()
                txt = name.upper()
                tw = fm.horizontalAdvance(txt)
                p.drawText(int(cx) - tw // 2, int(cy) - 4, txt)
                # Frequenz darunter
                p.setFont(QFont("Consolas", 11, QFont.Weight.Bold))
                freq_txt = f"{freq:.3f}" if freq else ""
                ftw = p.fontMetrics().horizontalAdvance(freq_txt)
                p.setPen(QColor(220, 220, 100, 200))
                p.drawText(int(cx) - ftw // 2, int(cy) + 16, freq_txt)
            elif is_online:
                # Online-Sektor: mittelgroß, Farbe
                p.setPen(QColor(base_color.red(), base_color.green(), base_color.blue(), 160))
                p.setFont(QFont("Consolas", 10, QFont.Weight.Bold))
                fm = p.fontMetrics()
                txt = name
                tw = fm.horizontalAdvance(txt)
                p.drawText(int(cx) - tw // 2, int(cy) - 2, txt)
                # Frequenz
                p.setFont(QFont("Consolas", 8))
                freq_txt = f"{freq:.3f}" if freq else ""
                ftw = p.fontMetrics().horizontalAdvance(freq_txt)
                p.setPen(QColor(200, 200, 200, 120))
                p.drawText(int(cx) - ftw // 2, int(cy) + 12, freq_txt)
            else:
                # Offline-Sektor: dezent
                p.setPen(QColor(base_color.red(), base_color.green(), base_color.blue(), 40))
                p.setFont(QFont("Consolas", 7))
                short_id = sec["id"].replace("_CTR", "").replace("_", " ")
                p.drawText(int(cx) - 20, int(cy) + 3, short_id)

    def _draw_global_navaids(self, p: QPainter):
        """Zeichnet Navaids aus der globalen NavDB – gut sichtbar.
        Farbcodierte Symbole: VOR=Cyan, NDB=Lila, DME=Orange, FIX=Grün."""
        if not _NAV_DB:
            return
        kinds = _lod_navaid_kinds(self.scale)
        if not kinds:
            return

        lat_min, lat_max, lon_min, lon_max = self._viewport_bounds()
        navaids = _NAV_DB.in_viewport(lat_min, lat_max, lon_min, lon_max,
                                       kinds=kinds, limit=800)
        w, h = self.width(), self.height()
        used_rects: list = []

        for nav in navaids:
            x, y = lat_lon_to_pixel(nav.lat, nav.lon,
                                     self.center_lat, self.center_lon,
                                     self.scale, w, h)
            if not (-50 < x < w + 50 and -50 < y < h + 50):
                continue

            if nav.kind in ("VOR", "VORDME"):
                color = QColor(0, 180, 220)
                size = 8
                p.setPen(QPen(color, 1.5))
                p.setBrush(Qt.BrushStyle.NoBrush)
                p.drawEllipse(QRectF(x - size, y - size, size * 2, size * 2))
                p.setBrush(QBrush(QColor(0, 220, 255)))
                p.setPen(Qt.PenStyle.NoPen)
                p.drawEllipse(QRectF(x - 2, y - 2, 4, 4))
            elif nav.kind == "NDB":
                color = QColor(180, 120, 255)
                size = 7
                p.setBrush(QBrush(color))
                p.setPen(Qt.PenStyle.NoPen)
                p.drawEllipse(QRectF(x - 2.5, y - 2.5, 5, 5))
                p.setPen(QPen(color, 1))
                for ang in (0, 60, 120, 180, 240, 300):
                    rad = math.radians(ang)
                    p.drawLine(int(x + 3*math.cos(rad)), int(y + 3*math.sin(rad)),
                               int(x + size*math.cos(rad)), int(y + size*math.sin(rad)))
            elif nav.kind == "DME":
                color = QColor(220, 160, 0)
                size = 6
                p.setPen(QPen(color, 1.5))
                p.setBrush(Qt.BrushStyle.NoBrush)
                p.drawRect(QRectF(x - size/2, y - size/2, size, size))
            else:
                # FIX/Waypoint – deutlich sichtbar
                color = QColor(0, 200, 120)
                size = 5
                tri = QPolygonF([
                    QPointF(x, y - size),
                    QPointF(x - size * 0.87, y + size * 0.5),
                    QPointF(x + size * 0.87, y + size * 0.5),
                ])
                p.setPen(QPen(color, 1.5))
                p.setBrush(QBrush(QColor(0, 200, 120, 40)))
                p.drawPolygon(tri)

            # Label – gut lesbar, Anti-Overlap
            show_label = (nav.kind in ("VOR", "VORDME") and self.scale < 300) or \
                         (nav.kind in ("NDB", "DME") and self.scale < 300) or \
                         (nav.kind == "FIX" and self.scale > 20 and self.scale < 250)
            if show_label:
                font = QFont("Consolas", 8, QFont.Weight.Bold)
                p.setFont(font)
                fm = p.fontMetrics()
                lbl_text = nav.ident
                tw = fm.horizontalAdvance(lbl_text)
                th = fm.height()
                lx, ly = int(x) + size + 3, int(y) + 3
                r = QRectF(lx, ly - th, tw, th)
                overlap = False
                for ur in used_rects:
                    if r.intersects(ur):
                        overlap = True
                        break
                if not overlap:
                    used_rects.append(r)
                    p.setPen(QColor(180, 200, 220))
                    p.drawText(lx, ly, lbl_text)

    def _draw_global_airways(self, p: QPainter):
        """Zeichnet NUR echte benannte Airway-Segmente aus der NavDB.
        Keine generischen Punkt-zu-Punkt Verbindungen.
        Dunkelgrau #222222 mit 0.5px Breite, 10% Opacity."""
        if not _NAV_DB:
            return
        # Airways nur bei Center-Zoom (scale < 30)
        if self.scale >= 30:
            return
        lat_min, lat_max, lon_min, lon_max = self._viewport_bounds()
        w, h = self.width(), self.height()

        # Nur benannte Airways (haben einen ident wie L884, UL984, etc.)
        if _lod_show_high_airways(self.scale):
            segments = _NAV_DB.airways_in_viewport(
                lat_min, lat_max, lon_min, lon_max,
                kind="HIGH", limit=1500)
            pen = QPen(QColor(34, 34, 34, 25))  # #222222 @ ~10% Opacity
            pen.setWidthF(0.5)
            p.setPen(pen)
            for seg in segments:
                # NUR echte Airways mit Kennung zeichnen
                if not seg.airway or seg.airway.strip() == "":
                    continue
                x1, y1 = lat_lon_to_pixel(seg.fix1_lat, seg.fix1_lon,
                                           self.center_lat, self.center_lon,
                                           self.scale, w, h)
                x2, y2 = lat_lon_to_pixel(seg.fix2_lat, seg.fix2_lon,
                                           self.center_lat, self.center_lon,
                                           self.scale, w, h)
                if ((-500 < x1 < w + 500 and -500 < y1 < h + 500) or
                        (-500 < x2 < w + 500 and -500 < y2 < h + 500)):
                    p.drawLine(int(x1), int(y1), int(x2), int(y2))

        if _lod_show_low_airways(self.scale):
            segments = _NAV_DB.airways_in_viewport(
                lat_min, lat_max, lon_min, lon_max,
                kind="LOW", limit=1500)
            pen = QPen(QColor(34, 34, 34, 25))  # #222222 @ ~10% Opacity
            pen.setWidthF(0.5)
            p.setPen(pen)
            for seg in segments:
                if not seg.airway or seg.airway.strip() == "":
                    continue
                x1, y1 = lat_lon_to_pixel(seg.fix1_lat, seg.fix1_lon,
                                           self.center_lat, self.center_lon,
                                           self.scale, w, h)
                x2, y2 = lat_lon_to_pixel(seg.fix2_lat, seg.fix2_lon,
                                           self.center_lat, self.center_lon,
                                           self.scale, w, h)
                if ((-500 < x1 < w + 500 and -500 < y1 < h + 500) or
                        (-500 < x2 < w + 500 and -500 < y2 < h + 500)):
                    p.drawLine(int(x1), int(y1), int(x2), int(y2))

    def _draw_osm_data(self, p: QPainter, osm: OSMData):
        """Zeichnet echte OSM-Taxiways, Aprons, Gates und Terminals – EuroScope-Style."""
        w, h = self.width(), self.height()

        # Aprons (gefüllte Flächen – sichtbar)
        fill = QColor(30, 30, 35, 100)
        outline = QColor(60, 60, 70)
        for apron in osm.aprons:
            if len(apron.points) < 3:
                continue
            pts = []
            for lat, lon in apron.points:
                x, y = lat_lon_to_pixel(lat, lon, self.center_lat,
                                        self.center_lon, self.scale, w, h)
                pts.append(QPointF(x, y))
            p.setBrush(QBrush(fill))
            p.setPen(QPen(outline, 1))
            p.drawPolygon(QPolygonF(pts))

        # Terminals (sichtbar)
        fill_t = QColor(35, 35, 45, 120)
        outline_t = QColor(70, 70, 80)
        for term in osm.terminals:
            if len(term.points) < 3:
                continue
            pts = []
            for lat, lon in term.points:
                x, y = lat_lon_to_pixel(lat, lon, self.center_lat,
                                        self.center_lon, self.scale, w, h)
                pts.append(QPointF(x, y))
            p.setBrush(QBrush(fill_t))
            p.setPen(QPen(outline_t, 1))
            p.drawPolygon(QPolygonF(pts))
            if self.scale > 500 and term.name:
                cx = sum(pt.x() for pt in pts) / len(pts)
                cy = sum(pt.y() for pt in pts) / len(pts)
                p.setPen(QColor(220, 220, 220))
                p.setFont(QFont("Consolas", 8, QFont.Weight.Bold))
                p.drawText(int(cx), int(cy), term.name)

        # Taxiways (Linien – sehr deutlich)
        pen_twy = QPen(QColor(30, 150, 220, 220))
        pen_twy.setWidth(3)
        p.setPen(pen_twy)
        for twy in osm.taxiways:
            for i in range(len(twy.points) - 1):
                lat1, lon1 = twy.points[i]
                lat2, lon2 = twy.points[i + 1]
                x1, y1 = lat_lon_to_pixel(lat1, lon1, self.center_lat,
                                           self.center_lon, self.scale, w, h)
                x2, y2 = lat_lon_to_pixel(lat2, lon2, self.center_lat,
                                           self.center_lon, self.scale, w, h)
                if ((-500 < x1 < w + 500 and -500 < y1 < h + 500) or
                        (-500 < x2 < w + 500 and -500 < y2 < h + 500)):
                    p.drawLine(int(x1), int(y1), int(x2), int(y2))
            # Taxiway-Name am Mittelpunkt
            if self.scale > 400 and twy.ref:
                mid_idx = len(twy.points) // 2
                mid_lat, mid_lon = twy.points[mid_idx]
                mx, my = lat_lon_to_pixel(mid_lat, mid_lon, self.center_lat,
                                           self.center_lon, self.scale, w, h)
                if 0 < mx < w and 0 < my < h:
                    p.setPen(QColor(100, 200, 255))
                    p.setFont(QFont("Consolas", 9, QFont.Weight.Bold))
                    p.drawText(int(mx) + 3, int(my) - 2, twy.ref)
                    p.setPen(pen_twy)

        # Gates (Punkte + Labels – sehr deutlich)
        if self.scale > 300:
            p.setFont(QFont("Consolas", 10, QFont.Weight.Bold))
            for gate in osm.gates:
                x, y = lat_lon_to_pixel(gate.lat, gate.lon, self.center_lat,
                                        self.center_lon, self.scale, w, h)
                if 0 < x < w and 0 < y < h:
                    # Occupancy-Check: belegt → rot, frei → grün
                    icao_key = osm.icao if hasattr(osm, 'icao') else (self._osm_icao or "")
                    occupied = self._occupied_gates.get(icao_key, set())
                    is_occupied = gate.ref in occupied
                    if is_occupied:
                        fill_c = QColor(255, 60, 60)       # Rot: belegt
                        border_c = QColor(200, 0, 0)
                        label_c = QColor(255, 120, 120)
                    else:
                        fill_c = QColor(255, 165, 0)        # Orange: frei
                        border_c = QColor(200, 120, 0)
                        label_c = QColor(255, 200, 100)
                    p.setBrush(QBrush(fill_c))
                    p.setPen(QPen(border_c, 1))
                    r = 5 if self.scale > 800 else 4
                    p.drawEllipse(QRectF(x - r, y - r, r * 2, r * 2))
                    # Label
                    p.setPen(label_c)
                    p.drawText(int(x) + 6, int(y) + 5, gate.ref)

    def _draw_gate_db_gates(self, p: QPainter):
        """Zeichnet Gates aus der GateDB (SQLite) – unabhängig von OSM-Overpass.

        Wird nur aufgerufen wenn show_gates=True und Zoom ausreichend (scale > 300).
        Nutzt _gate_db_cache für Performance.
        """
        if not _GATE_DB:
            return

        w, h = self.width(), self.height()

        # Welcher Flughafen ist im Blickfeld?
        icao = self._osm_icao
        if not icao:
            return

        # Cache: Gates nur einmal laden
        if icao not in self._gate_db_cache:
            try:
                gates = _GATE_DB.get_gates(icao)
                self._gate_db_cache[icao] = gates
            except Exception:
                self._gate_db_cache[icao] = []
                return

        gates = self._gate_db_cache.get(icao, [])
        if not gates:
            return

        # Occupancy-Daten holen
        occupied = self._occupied_gates.get(icao, set())

        p.setFont(QFont("Consolas", 10, QFont.Weight.Bold))

        for gate in gates:
            x, y = lat_lon_to_pixel(gate.lat, gate.lon, self.center_lat,
                                    self.center_lon, self.scale, w, h)
            if not (0 < x < w and 0 < y < h):
                continue

            is_occupied = gate.ref in occupied

            if is_occupied:
                fill_c = QColor(255, 60, 60)       # Rot: belegt
                border_c = QColor(200, 0, 0)
                label_c = QColor(255, 120, 120)
            else:
                fill_c = QColor(255, 165, 0)        # Orange: frei
                border_c = QColor(200, 120, 0)
                label_c = QColor(255, 200, 100)

            # Punkt
            p.setBrush(QBrush(fill_c))
            p.setPen(QPen(border_c, 1))
            r = 5 if self.scale > 800 else 4
            p.drawEllipse(QRectF(x - r, y - r, r * 2, r * 2))

            # Label (nur bei tiefem Zoom)
            if self.scale > 500:
                p.setPen(label_c)
                p.drawText(int(x) + 6, int(y) + 5, gate.ref)

    def _update_occupied_gates(self):
        """Berechnet welche Gates besetzt sind (Piloten am Gate stehen)."""
        occupied: Dict[str, set] = {}
        for pilot in self.pilots.values():
            if pilot.gate and pilot.gate_dist_m < 50:
                # Wir brauchen den ICAO-Code des nächsten Flughafens
                icao = ""
                # Versuch: nearest_airport aus dem Kontext
                fp_dep = pilot.dep
                fp_arr = pilot.arr
                # Pilot am Boden + Gate → wahrscheinlich am nearest Airport
                if self._osm_icao:
                    icao = self._osm_icao
                if icao:
                    if icao not in occupied:
                        occupied[icao] = set()
                    occupied[icao].add(pilot.gate)
        self._occupied_gates = occupied

    def _on_tile_ready(self, z: int, x: int, y: int):
        """Callback vom TileManager wenn ein Tile fertig heruntergeladen ist."""
        self.invalidate_bg_cache()
        self.update()

    # -- Great Circle & Distanz ---------------------------------------------
    @staticmethod
    def _haversine_nm(lat1: float, lon1: float, lat2: float, lon2: float) -> float:
        """Berechnet Distanz in NM über die Haversine-Formel."""
        R = 3440.065  # Erdradius in NM
        dlat = math.radians(lat2 - lat1)
        dlon = math.radians(lon2 - lon1)
        a = (math.sin(dlat/2)**2
             + math.cos(math.radians(lat1)) * math.cos(math.radians(lat2))
             * math.sin(dlon/2)**2)
        return R * 2 * math.atan2(math.sqrt(a), math.sqrt(1-a))

    def _draw_great_circle(self, p: QPainter, pilot: PilotInfo):
        """Zeichnet Great Circle Linie zum Zielflughafen."""
        dest = _AIRPORT_DB.get(pilot.arr)
        if not dest:
            return
        w, h = self.width(), self.height()

        # Interpolierte Punkte auf dem Großkreis
        lat1, lon1 = math.radians(pilot.lat), math.radians(pilot.lon)
        lat2, lon2 = math.radians(dest.lat), math.radians(dest.lon)
        d = self._haversine_nm(pilot.lat, pilot.lon, dest.lat, dest.lon)
        if d < 1:
            return

        # Anzahl Segmente abhängig von Distanz
        n_segs = min(200, max(20, int(d / 50)))
        d_rad = d / 3440.065  # Winkel in Radiant

        pen = QPen(QColor(255, 140, 0, 120))
        pen.setWidth(1)
        pen.setStyle(Qt.PenStyle.DashDotLine)
        p.setPen(pen)

        prev_xy = None
        for i in range(n_segs + 1):
            f = i / n_segs
            # Sphärische Interpolation
            A = math.sin((1-f) * d_rad) / max(1e-10, math.sin(d_rad))
            B = math.sin(f * d_rad) / max(1e-10, math.sin(d_rad))
            x_gc = A * math.cos(lat1) * math.cos(lon1) + B * math.cos(lat2) * math.cos(lon2)
            y_gc = A * math.cos(lat1) * math.sin(lon1) + B * math.cos(lat2) * math.sin(lon2)
            z_gc = A * math.sin(lat1) + B * math.sin(lat2)
            gc_lat = math.degrees(math.atan2(z_gc, math.sqrt(x_gc*x_gc + y_gc*y_gc)))
            gc_lon = math.degrees(math.atan2(y_gc, x_gc))

            sx, sy = lat_lon_to_pixel(gc_lat, gc_lon,
                                       self.center_lat, self.center_lon,
                                       self.scale, w, h)
            if prev_xy is not None:
                px, py = prev_xy
                if ((-1000 < px < w+1000 and -1000 < py < h+1000) or
                        (-1000 < sx < w+1000 and -1000 < sy < h+1000)):
                    p.drawLine(int(px), int(py), int(sx), int(sy))
            prev_xy = (sx, sy)

        # Ziel-Marker
        dx, dy = lat_lon_to_pixel(dest.lat, dest.lon,
                                   self.center_lat, self.center_lon,
                                   self.scale, w, h)
        if 0 < dx < w and 0 < dy < h:
            p.setBrush(Qt.BrushStyle.NoBrush)
            p.setPen(QPen(QColor(255, 140, 0), 2))
            p.drawEllipse(QRectF(dx-6, dy-6, 12, 12))
            p.setFont(QFont("Consolas", 7, QFont.Weight.Bold))
            p.drawText(int(dx)+9, int(dy)+4, pilot.arr)

    def _on_osm_ready(self, icao: str, osm_data: OSMData):
        """Callback vom OSMOverpass wenn Airport-Daten geladen sind."""
        self._osm_data[icao] = osm_data
        self.invalidate_bg_cache()
        self.update()

    # -- OSM-Tiles (via TileManager) ----------------------------------------
    @staticmethod
    def _latlon_to_tile(lat: float, lon: float, zoom: int) -> tuple[int, int]:
        lat = max(-85.05, min(85.05, lat))
        n = 2 ** zoom
        x = int((lon + 180.0) / 360.0 * n)
        lat_rad = math.radians(lat)
        y = int((1.0 - math.log(math.tan(lat_rad)
                + 1.0 / math.cos(lat_rad)) / math.pi) / 2.0 * n)
        return x, y

    @staticmethod
    def _tile_to_latlon(tx: int, ty: int, zoom: int) -> tuple[float, float]:
        n = 2 ** zoom
        lon = tx / n * 360.0 - 180.0
        lat = math.degrees(math.atan(math.sinh(math.pi * (1 - 2.0 * ty / n))))
        return lat, lon

    def _zoom_for_scale(self) -> int:
        return _osm_zoom_from_scale(self.scale)

    def _draw_map_tiles(self, p: QPainter):
        z = self._zoom_for_scale()
        w, h = self.width(), self.height()
        px_per_km = self.scale / KM_PER_DEG_LAT
        cos_lat = math.cos(math.radians(self.center_lat))
        km_per_deg_lon = KM_PER_DEG_LAT * cos_lat
        half_w = (w / 2) / (px_per_km * km_per_deg_lon)
        half_h = (h / 2) / (px_per_km * KM_PER_DEG_LAT)

        tx_min, ty_max = self._latlon_to_tile(self.center_lat - half_h,
                                               self.center_lon - half_w, z)
        tx_max, ty_min = self._latlon_to_tile(self.center_lat + half_h,
                                               self.center_lon + half_w, z)
        tx_min -= 1; ty_min -= 1; tx_max += 1; ty_max += 1
        if (tx_max - tx_min + 1) * (ty_max - ty_min + 1) > 100:
            return

        p.setOpacity(0.4)
        for tx in range(tx_min, tx_max + 1):
            for ty in range(ty_min, ty_max + 1):
                pm = self._tile_mgr.get_tile(z, tx, ty)
                if pm is None:
                    continue
                tlat, tlon = self._tile_to_latlon(tx, ty, z)
                tlat2, tlon2 = self._tile_to_latlon(tx+1, ty+1, z)
                sx, sy = lat_lon_to_pixel(tlat, tlon, self.center_lat,
                                           self.center_lon, self.scale, w, h)
                sx2, sy2 = lat_lon_to_pixel(tlat2, tlon2, self.center_lat,
                                             self.center_lon, self.scale, w, h)
                p.drawPixmap(QRectF(sx, sy, sx2-sx, sy2-sy).toRect(), pm)
        p.setOpacity(1.0)

    # ── Search-Marker: Visueller Flughafen-Indikator ──────────────────

    def _draw_search_marker(self, p: QPainter):
        """Zeichnet einen pulsierenden Kreis + Label um den Ziel-Flughafen."""
        if self._search_marker_timer <= 0 or not self._search_marker_icao:
            return

        w, h = self.width(), self.height()
        sx, sy = lat_lon_to_pixel(
            self._search_marker_lat, self._search_marker_lon,
            self.center_lat, self.center_lon, self.scale, w, h)

        # Außerhalb Bildschirm? → nicht zeichnen
        if sx < -100 or sx > w + 100 or sy < -100 or sy > h + 100:
            return

        # Pulsierender Radius (3s Animation, ~90 Frames bei 30fps)
        progress = 1.0 - (self._search_marker_timer / 90.0)
        base_radius = 25
        pulse = 8 * math.sin(progress * math.pi * 6)  # 3 Pulse
        radius = int(base_radius + pulse)

        # Alpha-Fade (langsam ausblenden in letzter Sekunde)
        alpha = 255 if self._search_marker_timer > 30 else int(255 * self._search_marker_timer / 30)

        # Äußerer Ring (Cyan, pulsierend)
        p.setPen(QPen(QColor(0, 220, 255, alpha), 2))
        p.setBrush(QColor(0, 220, 255, alpha // 8))
        p.drawEllipse(QPointF(sx, sy), radius, radius)

        # Innerer Punkt
        p.setPen(Qt.PenStyle.NoPen)
        p.setBrush(QColor(0, 220, 255, alpha))
        p.drawEllipse(QPointF(sx, sy), 4, 4)

        # ICAO-Label
        p.setPen(QColor(0, 220, 255, alpha))
        p.setFont(QFont("Consolas", 11, QFont.Weight.Bold))
        label = self._search_marker_icao
        p.drawText(int(sx) + radius + 6, int(sy) - 8, label)

        # Suggested Frequency (unterhalb ICAO)
        if self._search_marker_freq > 0:
            p.setFont(QFont("Consolas", 9))
            p.setPen(QColor(80, 255, 80, alpha))
            p.drawText(int(sx) + radius + 6, int(sy) + 6,
                        f"▸ {self._search_marker_freq:.3f} MHz")

        # FIR-Name (darunter)
        if self._search_marker_fir:
            p.setFont(QFont("Consolas", 8))
            p.setPen(QColor(180, 180, 220, alpha))
            p.drawText(int(sx) + radius + 6, int(sy) + 19,
                        f"FIR: {self._search_marker_fir}")

        # Fadenkreuz (dezent)
        line_len = radius + 10
        p.setPen(QPen(QColor(0, 220, 255, alpha // 3), 1, Qt.PenStyle.DashLine))
        p.drawLine(int(sx) - line_len, int(sy), int(sx) + line_len, int(sy))
        p.drawLine(int(sx), int(sy) - line_len, int(sx), int(sy) + line_len)

        # Timer dekrementieren
        self._search_marker_timer -= 1

    def _draw_overlay(self, p: QPainter):
        p.setPen(QColor(140, 180, 220))
        p.setFont(QFont("Consolas", 9))
        parts = [f"Piloten: {len(self.pilots)}",
                 f"Zentrum: {self.center_lat:.2f}°N {self.center_lon:.2f}°E",
                 f"Zoom: {self.scale:.0f}",
                 f"Rolle: {self.role}"]
        if self.my_station:
            parts.append(f"Station: {self.my_station}")
        if self.show_map_tiles:
            z = self._zoom_for_scale()
            parts.append(f"Map: Z{z}")
        if self.show_firs:
            parts.append("FIR")
        if self.show_sectors:
            parts.append("SEC")
        if self.show_global_navaids:
            parts.append("NavDB")
        if self.show_global_airways:
            parts.append("AWY")
        if self._osm_icao and self._osm_icao in self._osm_data:
            parts.append(f"OSM:{self._osm_icao}")
        if self.show_gates and _GATE_DB:
            parts.append("GATE")
        p.drawText(8, 16, "  |  ".join(parts))

        # ── METAR-Leiste (zweite Zeile) ──
        if self._metar_bar_text:
            p.setFont(QFont("Consolas", 10, QFont.Weight.Bold))
            p.setPen(QColor(80, 220, 255))
            p.drawText(8, 32, f"☁ {self._metar_bar_text}")

        # ── Voice-Kanal-Leiste (dritte Zeile) ──
        voice_ch = getattr(self.parent(), '_voice_channel', '') if self.parent() else ''
        voice_freq = getattr(self.parent(), '_voice_freq', 0) if self.parent() else 0
        voice_st = getattr(self.parent(), '_voice_status', '') if self.parent() else ''
        if voice_ch:
            y_voice = 48 if self._metar_bar_text else 32
            p.setFont(QFont("Consolas", 10, QFont.Weight.Bold))
            if voice_st == "joined":
                p.setPen(QColor(63, 185, 80))     # grün
                freq_s = f"  [{voice_freq:.3f} MHz]" if voice_freq > 0 else ""
                p.drawText(8, y_voice, f"🎤 VOICE: {voice_ch}{freq_s}  verbunden")
            elif voice_st == "pending":
                p.setPen(QColor(210, 153, 34))     # gelb
                p.drawText(8, y_voice, f"🎤 VOICE: {voice_ch}  ausstehend …")
            else:
                p.setPen(QColor(248, 81, 73))      # rot
                p.drawText(8, y_voice, f"🎤 VOICE: {voice_ch}  Fehler")

        # Wind-Pfeil oben rechts zeichnen
        if self._wind_speed > 0:
            self._draw_wind_arrow(p)

    def _draw_wind_arrow(self, p: QPainter):
        """Zeichnet einen Wind-Pfeil oben rechts im Radar."""
        cx = self.width() - 50
        cy = 50
        radius = 30
        # Kreis
        p.setPen(QPen(QColor(100, 100, 100), 1))
        p.drawEllipse(cx - radius, cy - radius, radius * 2, radius * 2)
        # Nord-Markierung
        p.setPen(QColor(100, 150, 100))
        p.setFont(QFont("Consolas", 7))
        p.drawText(cx - 3, cy - radius - 3, "N")
        # Pfeilrichtung: Wind kommt AUS _wind_dir, Pfeil zeigt IN Richtung
        angle_rad = math.radians(self._wind_dir)
        # Pfeil zeigt windabwärts (Wind kommt aus dir -> zeigt dir + 180)
        tip_x = cx + radius * 0.8 * math.sin(angle_rad)
        tip_y = cy - radius * 0.8 * math.cos(angle_rad)
        tail_x = cx - radius * 0.8 * math.sin(angle_rad)
        tail_y = cy + radius * 0.8 * math.cos(angle_rad)
        # Pfeil-Linie
        color = QColor(255, 255, 80) if self._wind_speed >= 15 else QColor(80, 200, 255)
        p.setPen(QPen(color, 2))
        p.drawLine(int(tail_x), int(tail_y), int(tip_x), int(tip_y))
        # Pfeilspitze
        arr_len = 8
        arr1_x = tip_x - arr_len * math.sin(angle_rad + 0.4)
        arr1_y = tip_y + arr_len * math.cos(angle_rad + 0.4)
        arr2_x = tip_x - arr_len * math.sin(angle_rad - 0.4)
        arr2_y = tip_y + arr_len * math.cos(angle_rad - 0.4)
        p.drawLine(int(tip_x), int(tip_y), int(arr1_x), int(arr1_y))
        p.drawLine(int(tip_x), int(tip_y), int(arr2_x), int(arr2_y))
        # Label: Richtung/Stärke
        p.setPen(color)
        p.setFont(QFont("Consolas", 8))
        p.drawText(cx - 25, cy + radius + 14, f"{self._wind_dir:03d}/{self._wind_speed}kt")

    # -- Maus ---------------------------------------------------------------
    def mousePressEvent(self, event):
        if event.button() == Qt.MouseButton.LeftButton:
            self._try_select_pilot(event.pos().x(), event.pos().y())
        if event.button() == Qt.MouseButton.MiddleButton:
            self._drag_start = event.pos()
            self._drag_center_lat = self.center_lat
            self._drag_center_lon = self.center_lon
        if event.button() == Qt.MouseButton.RightButton:
            # Rechtsklick: erst prüfen ob Pilot getroffen
            clicked = self._find_pilot_at(event.pos().x(), event.pos().y())
            if clicked:
                self.selected_callsign = clicked
                self.pilot_right_clicked.emit(clicked, event.globalPosition().x(),
                                               event.globalPosition().y())
            else:
                self._drag_start = event.pos()
                self._drag_center_lat = self.center_lat
                self._drag_center_lon = self.center_lon

    def mouseMoveEvent(self, event):
        if self._drag_start is not None:
            dx = event.pos().x() - self._drag_start.x()
            dy = event.pos().y() - self._drag_start.y()
            px_per_km = self.scale / KM_PER_DEG_LAT
            cos_lat = math.cos(math.radians(self._drag_center_lat))
            km_per_deg_lon = KM_PER_DEG_LAT * cos_lat
            self.center_lon = self._drag_center_lon - dx / (px_per_km * km_per_deg_lon)
            self.center_lat = self._drag_center_lat + dy / (px_per_km * KM_PER_DEG_LAT)
            self.invalidate_bg_cache()
            self.update()

    def mouseReleaseEvent(self, event):
        if event.button() in (Qt.MouseButton.RightButton, Qt.MouseButton.MiddleButton):
            self._drag_start = None

    def wheelEvent(self, event):
        delta = event.angleDelta().y()
        factor = 1.15 if delta > 0 else 1 / 1.15
        self.scale = max(2.0, min(200000.0, self.scale * factor))
        self.invalidate_bg_cache()
        self.update()

    def keyPressEvent(self, event):
        key = event.key()
        toggle_map = {
            Qt.Key.Key_A: "show_artcc", Qt.Key.Key_R: "show_runways",
            Qt.Key.Key_N: "show_navaids", Qt.Key.Key_G: "show_geo",
            Qt.Key.Key_T: "show_taxiways", Qt.Key.Key_S: "show_sids",
            Qt.Key.Key_D: "show_stars", Qt.Key.Key_L: "show_labels",
            Qt.Key.Key_M: "show_map_tiles",
            Qt.Key.Key_P: "show_regions",
            Qt.Key.Key_W: "show_low_airways",
            Qt.Key.Key_H: "show_high_airways",
            Qt.Key.Key_I: "show_firs",
            Qt.Key.Key_V: "show_global_navaids",
            Qt.Key.Key_Y: "show_global_airways",
            Qt.Key.Key_X: "show_trend_vectors",
            Qt.Key.Key_C: "show_sectors",
            Qt.Key.Key_B: "show_gates",
        }
        attr = toggle_map.get(key)
        if attr:
            setattr(self, attr, not getattr(self, attr))
            self.invalidate_bg_cache()
            self.update()
        elif key == Qt.Key.Key_F:
            if self.pilots:
                p = next(iter(self.pilots.values()))
                self.center_lat = p.lat
                self.center_lon = p.lon
                self.invalidate_bg_cache()
                self.update()

    def _find_pilot_at(self, mx: int, my: int) -> Optional[str]:
        best, best_dist = None, 20.0
        for pilot in self.pilots.values():
            x, y = lat_lon_to_pixel(pilot.lat, pilot.lon, self.center_lat,
                                     self.center_lon, self.scale,
                                     self.width(), self.height())
            dist = ((x - mx)**2 + (y - my)**2) ** 0.5
            if dist < best_dist:
                best = pilot.callsign
                best_dist = dist
        return best

    def _try_select_pilot(self, mx: int, my: int):
        self.selected_callsign = self._find_pilot_at(mx, my)
        self.update()


# ---------------------------------------------------------------------------
# Flight-Strip Widget (einzelner Strip)
# ---------------------------------------------------------------------------
class StripItem(QFrame):
    """Ein einzelner Flight-Strip in der Seitenleiste."""

    def __init__(self, callsign: str, parent=None):
        super().__init__(parent)
        self.callsign = callsign
        self.setFrameStyle(QFrame.Shape.Box)
        self.setFixedHeight(90)
        self.setStyleSheet("""
            QFrame {
                background: #0e1225;
                border: 1px solid #0f2040;
                border-radius: 4px;
                margin: 1px;
            }
        """)
        layout = QVBoxLayout(self)
        layout.setContentsMargins(6, 4, 6, 4)
        layout.setSpacing(2)

        self.lbl_top = QLabel(callsign)
        self.lbl_top.setStyleSheet("color: #c8d6e5; font-weight: bold; font-size: 13px;")
        layout.addWidget(self.lbl_top)

        self.lbl_route = QLabel("")
        self.lbl_route.setStyleSheet("color: #58a6ff; font-size: 11px; font-weight: bold;")
        layout.addWidget(self.lbl_route)

        self.lbl_info = QLabel("")
        self.lbl_info.setStyleSheet("color: #8ba4b8; font-size: 11px;")
        layout.addWidget(self.lbl_info)

    def update_data(self, pilot: PilotInfo):
        alt_str = f"FL{int(pilot.alt/100):03d}" if pilot.alt >= 10000 else f"{int(pilot.alt)}ft"
        dep_arr = f"{pilot.dep} → {pilot.arr}" if pilot.dep else "---"
        sq = f"SQ:{pilot.squawk:04d}"
        clr = " CLR" if pilot.cleared else ""
        xpdr_mode = ""
        if hasattr(pilot, 'transponder_mode') and pilot.transponder_mode:
            xpdr_mode = f"  [{pilot.transponder_mode}]"
        self.lbl_route.setText(dep_arr)
        self.lbl_info.setText(f"{pilot.actype}  {alt_str}  {sq}{clr}{xpdr_mode}")

        # Farbe je nach Status
        if pilot.cleared:
            self.setStyleSheet(self.styleSheet().replace("#0f2040", "#0f2a4a"))
        elif pilot.handover_blink:
            self.setStyleSheet(self.styleSheet().replace("#0f2040", "#2a3a5a"))
        else:
            self.setStyleSheet(self.styleSheet().replace("#0f2a4a", "#0f2040").replace("#2a3a5a", "#0f2040"))


# ---------------------------------------------------------------------------
# Auto-Strip (kompakt: INBOUND / OUTBOUND)
# ---------------------------------------------------------------------------
class AutoStripItem(QFrame):
    """Kompakter Strip für Frequenz-basiertes Auto-Routing."""

    clicked = pyqtSignal(str)   # callsign

    def __init__(self, callsign: str, direction: str = "INBOUND", parent=None):
        super().__init__(parent)
        self.callsign = callsign
        self.direction = direction  # "INBOUND" or "OUTBOUND"
        self._blink_on = True
        self._acknowledged = False
        self.setFrameStyle(QFrame.Shape.Box)
        self.setFixedHeight(52)
        self.setCursor(Qt.CursorShape.PointingHandCursor)

        base_border = "#cc0000" if direction == "INBOUND" else "#00aa00"
        self._base_border = base_border
        self._base_bg = "#120008" if direction == "INBOUND" else "#001208"
        self.setStyleSheet(f"""
            QFrame {{
                background: {self._base_bg};
                border: 2px solid {base_border};
                border-radius: 4px;
                margin: 1px;
            }}
        """)

        layout = QVBoxLayout(self)
        layout.setContentsMargins(6, 3, 6, 3)
        layout.setSpacing(1)

        # Zeile 1: Callsign (fett)
        self.lbl_callsign = QLabel(callsign)
        cs_color = "#ff4444" if direction == "INBOUND" else "#44ff44"
        self.lbl_callsign.setStyleSheet(
            f"color: {cs_color}; font-weight: bold; font-size: 13px;"
            " font-family: Consolas; border: none;")
        layout.addWidget(self.lbl_callsign)

        # Zeile 2: DEP→ARR  |  Freq
        self.lbl_info = QLabel("")
        self.lbl_info.setStyleSheet(
            "color: #8ba4b8; font-size: 10px; font-family: Consolas; border: none;")
        layout.addWidget(self.lbl_info)

    def set_info(self, dep: str = "", arr: str = "", freq: float = 0,
                 actype: str = "", alt: float = 0):
        route = f"{dep}→{arr}" if dep or arr else "---"
        freq_s = f"{freq:.3f}" if freq > 0 else "---"
        alt_s = f"FL{int(alt/100):03d}" if alt >= 10000 else f"{int(alt)}ft"
        self.lbl_info.setText(f"{route}  {actype}  {alt_s}  {freq_s}")

    def acknowledge(self):
        """Strip wurde angeklickt → hört auf zu blinken."""
        self._acknowledged = True
        self._blink_on = True
        self.setStyleSheet(f"""
            QFrame {{
                background: {self._base_bg};
                border: 1px solid {self._base_border};
                border-radius: 4px;
                margin: 1px;
            }}
        """)

    def do_blink(self):
        """Wird vom Timer aufgerufen (toggle)."""
        if self._acknowledged:
            return
        self._blink_on = not self._blink_on
        if self._blink_on:
            self.setStyleSheet(f"""
                QFrame {{
                    background: #2a0010;
                    border: 2px solid #ff2222;
                    border-radius: 4px;
                    margin: 1px;
                }}
            """)
        else:
            self.setStyleSheet(f"""
                QFrame {{
                    background: {self._base_bg};
                    border: 2px solid #440000;
                    border-radius: 4px;
                    margin: 1px;
                }}
            """)

    def mousePressEvent(self, event):
        super().mousePressEvent(event)
        self.acknowledge()
        self.clicked.emit(self.callsign)


# ---------------------------------------------------------------------------
# Flugplan-Detail-Panel
# ---------------------------------------------------------------------------
class FlightStripWidget(QGroupBox):
    """Detailpanel für den ausgewählten Piloten + Steuerbuttons."""

    cleared_signal = pyqtSignal(str)            # callsign
    assign_alt_signal = pyqtSignal(str, str)    # callsign, alt
    transfer_signal = pyqtSignal(str, str)      # callsign, target_station

    def __init__(self, parent=None):
        super().__init__("Flugplan / Strip", parent)
        layout = QVBoxLayout(self)

        self.lbl_callsign = QLabel("---")
        self.lbl_callsign.setStyleSheet("font-size: 16px; font-weight: bold; color: #58a6ff;")
        layout.addWidget(self.lbl_callsign)

        self.lbl_position = QLabel("")
        self.lbl_position.setStyleSheet("color: #8ba4b8;")
        layout.addWidget(self.lbl_position)

        self.lbl_flightplan = QLabel("")
        self.lbl_flightplan.setStyleSheet("color: #a0b4c8;")
        layout.addWidget(self.lbl_flightplan)

        # Distanz / ETA Panel
        self.lbl_dist_eta = QLabel("")
        self.lbl_dist_eta.setStyleSheet(
            "color: #ff8844; font-family: Consolas; font-size: 11px; font-weight: bold;")
        layout.addWidget(self.lbl_dist_eta)

        # Temp Altitude
        alt_row = QHBoxLayout()
        self.txt_temp_alt = QLineEdit()
        self.txt_temp_alt.setPlaceholderText("z.B. FL100")
        self.txt_temp_alt.setMaximumWidth(90)
        self.txt_temp_alt.setStyleSheet(
            "background: #16213e; border: 1px solid #0f3460; color: #c8d6e5; padding: 2px;")
        alt_row.addWidget(QLabel("Temp Alt:"))
        alt_row.addWidget(self.txt_temp_alt)
        btn_alt = QPushButton("SET")
        btn_alt.setFixedWidth(40)
        btn_alt.clicked.connect(self._on_assign_alt)
        alt_row.addWidget(btn_alt)
        layout.addLayout(alt_row)

        # CLEARED Button
        self.btn_cleared = QPushButton("✓ CLEARED")
        self.btn_cleared.setStyleSheet(
            "background: #0f2a4a; color: #c8d6e5; padding: 6px; font-weight: bold; "
            "border: 1px solid #c8d6e5; border-radius: 3px;")
        self.btn_cleared.clicked.connect(self._on_cleared)
        layout.addWidget(self.btn_cleared)

        # TRANSFER Button
        transfer_row = QHBoxLayout()
        self.txt_transfer = QLineEdit()
        self.txt_transfer.setPlaceholderText("Ziel-Station")
        self.txt_transfer.setMaximumWidth(110)
        self.txt_transfer.setStyleSheet(
            "background: #16213e; border: 1px solid #0f3460; color: #ffcc44; padding: 2px;")
        transfer_row.addWidget(self.txt_transfer)
        self.btn_transfer = QPushButton("✈ Transfer")
        self.btn_transfer.setStyleSheet(
            "background: #3a2a00; color: #ffcc44; padding: 5px; font-weight: bold; "
            "border: 1px solid #ffcc44; border-radius: 3px;")
        self.btn_transfer.clicked.connect(self._on_transfer)
        transfer_row.addWidget(self.btn_transfer)
        layout.addLayout(transfer_row)

        layout.addStretch()

        self.setStyleSheet("""
            QGroupBox {
                color: #4a9edd; border: 1px solid #1a2a44;
                border-radius: 4px; margin-top: 8px; padding: 12px;
                background: #0a0a0a;
            }
            QGroupBox::title { subcontrol-origin: margin; left: 10px; }
            QLabel { color: #8ba4b8; }
            QPushButton { background: #0f3460; color: #c8d6e5; padding: 4px 8px;
                          border: 1px solid #c8d6e5; border-radius: 3px; }
        """)
        self._current_cs: str = ""

    def update_info(self, pilot: Optional[PilotInfo]):
        if pilot is None:
            self.lbl_callsign.setText("---")
            self.lbl_position.setText("Klicke auf einen Piloten")
            self.lbl_flightplan.setText("")
            self._current_cs = ""
            return

        self._current_cs = pilot.callsign
        self.lbl_callsign.setText(pilot.callsign)
        alt_str = f"FL{int(pilot.alt/100):03d}" if pilot.alt >= 10000 else f"{int(pilot.alt)} ft"
        gs_str = f"{int(pilot.groundspeed)} kn" if pilot.groundspeed else "0 kn"
        freq_str = f"{pilot.com1_freq:.3f}" if pilot.com1_freq > 0 else "---"
        sq_str = f"{pilot.squawk:04d}"
        ident_str = " [IDENT]" if pilot.squawk_ident else ""
        xpdr_str = ""
        if pilot.transponder_mode:
            xpdr_str = f"\nXPDR: {pilot.transponder_mode}"
        ctrl_str = ""
        if pilot.controller:
            # Frequenz des Controllers aus Online-Stationen holen
            ctrl_freq_str = ""
            parent_main = self.parent()
            while parent_main and not isinstance(parent_main, QMainWindow):
                parent_main = parent_main.parent() if hasattr(parent_main, 'parent') else None
            if parent_main and hasattr(parent_main, '_online_atc_stations'):
                for atc_st in parent_main._online_atc_stations:
                    if atc_st.get('station', '') == pilot.controller:
                        cf = atc_st.get('frequency', 0)
                        if cf > 0:
                            ctrl_freq_str = f" {cf:.3f}"
                        ctrl_full = atc_st.get('full_name', '')
                        if ctrl_full:
                            ctrl_freq_str += f" ({ctrl_full})"
                        break
            ctrl_str = f"\nOwner: {pilot.controller} ({pilot.controller_role}){ctrl_freq_str}"
        asq = f"\nAssigned SQ: {pilot.assigned_squawk:04d}" if pilot.assigned_squawk else ""
        aalt = f"\nAssigned Alt: {pilot.assigned_alt}" if pilot.assigned_alt else ""
        clr = "\nStatus: CLEARED" if pilot.cleared else ""

        self.lbl_position.setText(
            f"Lat: {pilot.lat:.5f}°\nLon: {pilot.lon:.5f}°\nAlt: {alt_str}\n"
            f"GS: {gs_str}\nFreq: {freq_str}\nSquawk: {sq_str}{ident_str}"
            f"{xpdr_str}{ctrl_str}{asq}{aalt}{clr}"
        )
        fp_parts = []
        if pilot.actype: fp_parts.append(f"Typ: {pilot.actype}")
        if pilot.dep: fp_parts.append(f"DEP: {pilot.dep}")
        if pilot.arr: fp_parts.append(f"ARR: {pilot.arr}")
        if pilot.cruise_alt: fp_parts.append(f"Cruise: {pilot.cruise_alt}")
        if pilot.route: fp_parts.append(f"Route: {pilot.route}")
        self.lbl_flightplan.setText("\n".join(fp_parts) if fp_parts else "Kein Flugplan")

        # Distanz & ETA zum Ziel berechnen
        dist_eta = ""
        if pilot.arr and _AIRPORT_DB:
            dest = _AIRPORT_DB.get(pilot.arr)
            if dest:
                dist_nm = RadarWidget._haversine_nm(
                    pilot.lat, pilot.lon, dest.lat, dest.lon)
                dist_eta = f"✈ {pilot.arr}: {dist_nm:.0f} NM"
                if pilot.groundspeed > 30:
                    eta_h = dist_nm / pilot.groundspeed
                    eta_min = int(eta_h * 60)
                    h, m = divmod(eta_min, 60)
                    dist_eta += f"  |  ETA: {h}h{m:02d}min"
        self.lbl_dist_eta.setText(dist_eta)

    def _on_cleared(self):
        if self._current_cs:
            self.cleared_signal.emit(self._current_cs)

    def _on_assign_alt(self):
        alt = self.txt_temp_alt.text().strip().upper()
        if self._current_cs and alt:
            self.assign_alt_signal.emit(self._current_cs, alt)
            self.txt_temp_alt.clear()

    def _on_transfer(self):
        target = self.txt_transfer.text().strip().upper()
        if self._current_cs and target:
            self.transfer_signal.emit(self._current_cs, target)
            self.txt_transfer.clear()


# ---------------------------------------------------------------------------
# Hauptfenster
# ---------------------------------------------------------------------------
class RadarWindow(QMainWindow):
    def __init__(self, server_url: str, sector_file: Optional[str] = None):
        super().__init__()
        self.setWindowTitle("NEXUS-ATC – ATC Radar")
        self.setMinimumSize(1000, 650)
        self.resize(1400, 800)
        self.setStyleSheet("background: #0a0a0a;")

        self.radar = RadarWidget()
        self.strip = FlightStripWidget()
        self.strip.cleared_signal.connect(self._send_cleared)
        self.strip.assign_alt_signal.connect(self._send_assigned_alt)
        self.strip.transfer_signal.connect(self._on_strip_transfer)
        self.radar.pilot_right_clicked.connect(self._show_context_menu)

        # Sektordatei laden
        self._sector_file = sector_file
        if sector_file:
            self._load_sector(sector_file)

        # ---- Layout aufbauen ----
        central = QWidget()
        main_layout = QHBoxLayout(central)
        main_layout.setContentsMargins(4, 4, 4, 4)

        # Flight Strip Leiste (ganz links, scrollbar) — GRÖSSER
        self.strip_area = QScrollArea()
        self.strip_area.setWidgetResizable(True)
        self.strip_area.setFixedWidth(260)
        self.strip_area.setStyleSheet("QScrollArea { border: 1px solid #0f2040; background: #080810; }")

        strip_outer = QWidget()
        strip_outer_layout = QVBoxLayout(strip_outer)
        strip_outer_layout.setContentsMargins(2, 2, 2, 2)
        strip_outer_layout.setSpacing(2)

        # ── INBOUND Header ──
        lbl_inbound = QLabel("▼  INBOUND")
        lbl_inbound.setStyleSheet(
            "color: #ff4444; font-weight: bold; font-family: Consolas;"
            " font-size: 12px; padding: 4px 6px; background: #1a0000;"
            " border: 1px solid #330000; border-radius: 3px;")
        strip_outer_layout.addWidget(lbl_inbound)

        self.inbound_container = QWidget()
        self.inbound_layout = QVBoxLayout(self.inbound_container)
        self.inbound_layout.setContentsMargins(0, 0, 0, 0)
        self.inbound_layout.setSpacing(2)
        self.inbound_layout.addStretch()
        strip_outer_layout.addWidget(self.inbound_container)

        # ── Separator ──
        sep = QFrame()
        sep.setFrameShape(QFrame.Shape.HLine)
        sep.setStyleSheet("color: #1a2a44;")
        strip_outer_layout.addWidget(sep)

        # ── OUTBOUND Header ──
        lbl_outbound = QLabel("▲  OUTBOUND")
        lbl_outbound.setStyleSheet(
            "color: #44ff44; font-weight: bold; font-family: Consolas;"
            " font-size: 12px; padding: 4px 6px; background: #001a00;"
            " border: 1px solid #003300; border-radius: 3px;")
        strip_outer_layout.addWidget(lbl_outbound)

        self.outbound_container = QWidget()
        self.outbound_layout = QVBoxLayout(self.outbound_container)
        self.outbound_layout.setContentsMargins(0, 0, 0, 0)
        self.outbound_layout.setSpacing(2)
        self.outbound_layout.addStretch()
        strip_outer_layout.addWidget(self.outbound_container)

        # ── Separator ──
        sep2 = QFrame()
        sep2.setFrameShape(QFrame.Shape.HLine)
        sep2.setStyleSheet("color: #1a2a44;")
        strip_outer_layout.addWidget(sep2)

        # ── ALL / Legacy strip list ──
        lbl_all = QLabel("▪  AKTIV")
        lbl_all.setStyleSheet(
            "color: #4a9edd; font-weight: bold; font-family: Consolas;"
            " font-size: 11px; padding: 3px 6px;")
        strip_outer_layout.addWidget(lbl_all)

        self.strip_container = QWidget()
        self.strip_layout = QVBoxLayout(self.strip_container)
        self.strip_layout.setContentsMargins(0, 0, 0, 0)
        self.strip_layout.setSpacing(3)
        self.strip_layout.addStretch()
        strip_outer_layout.addWidget(self.strip_container)

        strip_outer_layout.addStretch()
        self.strip_area.setWidget(strip_outer)

        self._strip_items: Dict[str, StripItem] = {}
        self._inbound_items: Dict[str, 'AutoStripItem'] = {}
        self._outbound_items: Dict[str, 'AutoStripItem'] = {}
        self._inbound_blink_timer = QTimer(self)
        self._inbound_blink_timer.timeout.connect(self._blink_inbound_strips)
        self._inbound_blink_timer.start(500)
        main_layout.addWidget(self.strip_area)

        # ---- Mitte: Radar oben + Bottom-Center-Panel unten ----
        center_vbox = QVBoxLayout()
        center_vbox.setContentsMargins(0, 0, 0, 0)
        center_vbox.setSpacing(4)
        center_vbox.addWidget(self.radar, stretch=1)

        # ---- Bottom-Center-Panel (Tabs: Flugplan/Strip, Funk-Chat) ----
        from PyQt6.QtWidgets import QTabWidget
        self.bottom_tabs = QTabWidget()
        self.bottom_tabs.setFixedHeight(220)
        self.bottom_tabs.setStyleSheet("""
            QTabWidget::pane { border: 1px solid #1a2a44; background: #0a0a0a; }
            QTabBar::tab { background: #0f1a2e; color: #4a9edd; padding: 6px 14px;
                           border: 1px solid #1a2a44; border-bottom: none;
                           border-top-left-radius: 4px; border-top-right-radius: 4px;
                           font-family: Consolas; font-size: 11px; min-width: 100px; }
            QTabBar::tab:selected { background: #0a0a0a; color: #58a6ff; font-weight: bold; }
            QTabBar::tab:hover { background: #16213e; }
        """)

        # --- Tab 1: Flugplan / Strip ---
        self.bottom_tabs.addTab(self.strip, "✈ Flugplan / Strip")

        # --- Tab 2: Frequenz-Chat ---
        chat_widget = QWidget()
        chat_vbox = QVBoxLayout(chat_widget)
        chat_vbox.setContentsMargins(6, 6, 6, 6)
        chat_vbox.setSpacing(4)

        self.txt_freq_chat = QTextEdit()
        self.txt_freq_chat.setReadOnly(True)
        self.txt_freq_chat.setStyleSheet(
            "QTextEdit { background: #0d1117; color: #c8d6e5; border: 1px solid #30363d;"
            " font-family: Consolas; font-size: 11px; border-radius: 3px; }")
        chat_vbox.addWidget(self.txt_freq_chat, 1)

        chat_input_row = QHBoxLayout()
        chat_input_row.setSpacing(3)
        self.txt_freq_input = QLineEdit()
        self.txt_freq_input.setPlaceholderText("Nachricht auf Frequenz...")
        self.txt_freq_input.setStyleSheet(
            "QLineEdit { background: #161b22; color: #c8d6e5; border: 1px solid #30363d;"
            " padding: 5px; font-size: 12px; border-radius: 3px; }"
            " QLineEdit:focus { border-color: #58a6ff; }")
        self.txt_freq_input.returnPressed.connect(self._send_freq_chat)
        chat_input_row.addWidget(self.txt_freq_input, 1)

        btn_freq_send = QPushButton("▶")
        btn_freq_send.setFixedWidth(36)
        btn_freq_send.setStyleSheet(
            "QPushButton { background: #21262d; color: #58a6ff; border: 1px solid #30363d;"
            " border-radius: 3px; font-weight: bold; font-size: 14px; }"
            " QPushButton:hover { background: #30363d; }")
        btn_freq_send.clicked.connect(self._send_freq_chat)
        chat_input_row.addWidget(btn_freq_send)
        chat_vbox.addLayout(chat_input_row)

        self.bottom_tabs.addTab(chat_widget, "📻 Funk-Chat")

        # --- Tab 3: CPDLC-Log ---
        cpdlc_widget = QWidget()
        cpdlc_vbox = QVBoxLayout(cpdlc_widget)
        cpdlc_vbox.setContentsMargins(6, 6, 6, 6)
        cpdlc_vbox.setSpacing(4)

        self.txt_cpdlc_log = QTextEdit()
        self.txt_cpdlc_log.setReadOnly(True)
        self.txt_cpdlc_log.setStyleSheet(
            "QTextEdit { background: #050810; color: #00e064; border: 1px solid #1a2a44;"
            " font-family: Consolas; font-size: 11px; border-radius: 3px; }")
        cpdlc_vbox.addWidget(self.txt_cpdlc_log, 1)

        self.bottom_tabs.addTab(cpdlc_widget, "📡 CPDLC")

        center_vbox.addWidget(self.bottom_tabs)
        main_layout.addLayout(center_vbox, stretch=1)

        # ---- Rechtes Panel (kompakt: Controller, Suche, Wetter, Ansicht) ----
        right_panel = QVBoxLayout()
        PANEL_STYLE = """
            QGroupBox { color: #4a9edd; border: 1px solid #1a2a44;
                        border-radius: 4px; padding: 10px; margin-top: 8px;
                        background: #0a0a0a; }
            QGroupBox::title { subcontrol-origin: margin; left: 10px; }
            QLineEdit { background: #16213e; border: 1px solid #0f3460;
                        color: #c8d6e5; padding: 3px; font-family: Consolas; }
            QPushButton { background: #0f3460; color: #c8d6e5; padding: 4px 8px;
                          border: 1px solid #c8d6e5; border-radius: 3px; }
            QPushButton:hover { background: #1a5276; }
            QComboBox { background: #16213e; color: #c8d6e5;
                        border: 1px solid #0f3460; padding: 3px; font-family: Consolas; }
            QLabel { color: #8ba4b8; }
        """

        # --- Controller-Registrierung ---
        ctrl_group = QGroupBox("Controller")
        ctrl_group.setStyleSheet(PANEL_STYLE)
        ctrl_form = QFormLayout(ctrl_group)
        self.txt_ctrl_name = QLineEdit()
        self.txt_ctrl_name.setPlaceholderText("Dein Name")
        ctrl_form.addRow("Name:", self.txt_ctrl_name)
        self.txt_station = QLineEdit()
        self.txt_station.setPlaceholderText("z.B. LSZH_APP")
        self.txt_station.textChanged.connect(self._on_station_text_changed)
        ctrl_form.addRow("Station:", self.txt_station)

        # Airport-Info Label (wird automatisch befüllt)
        self.lbl_airport_info = QLabel("")
        self.lbl_airport_info.setWordWrap(True)
        self.lbl_airport_info.setStyleSheet(
            "color: #00ccff; font-family: Consolas; font-size: 10px;"
            " padding: 2px; background: #0d1b2a; border-radius: 3px;")
        self.lbl_airport_info.setVisible(False)
        ctrl_form.addRow(self.lbl_airport_info)

        self.txt_ctrl_freq = QLineEdit()
        self.txt_ctrl_freq.setPlaceholderText("wird automatisch gefüllt")
        ctrl_form.addRow("Frequenz:", self.txt_ctrl_freq)
        self.cmb_role = QComboBox()
        self.cmb_role.addItems(["DEL", "GND", "TWR", "APP", "DEP", "CTR"])
        self.cmb_role.setCurrentText("CTR")
        ctrl_form.addRow("Rolle:", self.cmb_role)
        btn_row = QHBoxLayout()
        self.btn_register = QPushButton("✓ Registrieren")
        self.btn_register.clicked.connect(self._on_register_controller)
        btn_row.addWidget(self.btn_register)
        self.btn_logout = QPushButton("✗ Logout")
        self.btn_logout.setStyleSheet(
            "background: #3e1010; color: #ff4444; border: 1px solid #ff4444;")
        self.btn_logout.clicked.connect(self._on_logout_controller)
        self.btn_logout.setEnabled(False)
        btn_row.addWidget(self.btn_logout)
        ctrl_form.addRow(btn_row)
        self.lbl_ctrl_status = QLabel("Nicht registriert")
        self.lbl_ctrl_status.setStyleSheet("color: #888; font-size: 10px;")
        ctrl_form.addRow(self.lbl_ctrl_status)
        right_panel.addWidget(ctrl_group)

        # --- ICAO-Suche ---
        icao_group = QGroupBox("Flughafen suchen")
        icao_group.setStyleSheet(PANEL_STYLE)
        icao_form = QHBoxLayout(icao_group)
        self.txt_icao = QLineEdit()
        self.txt_icao.setPlaceholderText("ICAO")
        self.txt_icao.setMaximumWidth(70)
        self.txt_icao.returnPressed.connect(self._on_search_icao)
        icao_form.addWidget(self.txt_icao)
        btn_icao = QPushButton("GO")
        btn_icao.setFixedWidth(40)
        btn_icao.clicked.connect(self._on_search_icao)
        icao_form.addWidget(btn_icao)
        right_panel.addWidget(icao_group)

        # --- Wetter-Panel ---
        wx_group = QGroupBox("Wetter")
        wx_group.setStyleSheet(PANEL_STYLE)
        wx_form = QVBoxLayout(wx_group)
        wx_form.setSpacing(2)
        self.lbl_wx_icao = QLabel("---")
        self.lbl_wx_icao.setStyleSheet("color: #c8d6e5; font-weight: bold; font-size: 12px;")
        wx_form.addWidget(self.lbl_wx_icao)
        self.lbl_wx_wind = QLabel("Wind: ---")
        self.lbl_wx_wind.setStyleSheet("color: #dddd44; font-family: Consolas; font-size: 11px;")
        wx_form.addWidget(self.lbl_wx_wind)
        self.lbl_wx_qnh = QLabel("QNH: ---")
        self.lbl_wx_qnh.setStyleSheet("color: #44ddff; font-family: Consolas; font-size: 11px;")
        wx_form.addWidget(self.lbl_wx_qnh)
        self.lbl_wx_temp = QLabel("Temp: ---")
        self.lbl_wx_temp.setStyleSheet("color: #dddddd; font-family: Consolas; font-size: 12px;")
        wx_form.addWidget(self.lbl_wx_temp)
        self.lbl_wx_vis = QLabel("Sicht: ---")
        self.lbl_wx_vis.setStyleSheet("color: #dddddd; font-family: Consolas; font-size: 12px;")
        wx_form.addWidget(self.lbl_wx_vis)
        self.lbl_wx_rwy = QLabel("Empf. RWY: ---")
        self.lbl_wx_rwy.setStyleSheet("color: #ff8844; font-family: Consolas; font-size: 12px; font-weight: bold;")
        wx_form.addWidget(self.lbl_wx_rwy)
        self.lbl_wx_atis = QLabel("")
        self.lbl_wx_atis.setWordWrap(True)
        self.lbl_wx_atis.setStyleSheet("color: #b0c4d8; font-family: Consolas; font-size: 11px;")
        wx_form.addWidget(self.lbl_wx_atis)
        right_panel.addWidget(wx_group)

        # Wetter-Zustand
        self._current_metar: Optional[ParsedMetar] = None
        self._current_weather_icao: Optional[str] = None
        self._recommended_rwy: Optional[str] = None

        # Voice-Kanal-Zustand (ATC Auto-Sector-Move)
        self._voice_channel: str = ""
        self._voice_freq: float = 0.0
        self._voice_center_name: str = ""
        self._voice_status: str = ""       # "joined" | "pending" | ""

        # --- Ansicht / Rolle ---
        mode_group = QGroupBox("Ansicht")
        mode_group.setStyleSheet(PANEL_STYLE)
        mode_layout = QVBoxLayout(mode_group)

        self.chk_map = QCheckBox("OSM-Karte (M)")
        self.chk_map.setStyleSheet("color: #4a9edd;")
        self.chk_map.toggled.connect(self._on_map_toggled)
        mode_layout.addWidget(self.chk_map)

        btn_row = QHBoxLayout()
        for role in ["GND", "TWR", "APP", "CTR"]:
            b = QPushButton(role)
            b.clicked.connect(lambda _, r=role: self._apply_role(r))
            btn_row.addWidget(b)
        mode_layout.addLayout(btn_row)
        right_panel.addWidget(mode_group)

        right_panel.addStretch()

        right_widget = QWidget()
        right_widget.setLayout(right_panel)
        right_widget.setFixedWidth(270)
        main_layout.addWidget(right_widget)
        self.setCentralWidget(central)

        # --- Voice-Variablen initialisieren (VOR dem Dock-Aufbau) ---
        self._voice_state: dict = {}
        self._voice_known_users: dict[str, set[str]] = {}
        self._voice_config = self._load_voice_config()
        self._voice_rx_volume: int = self._voice_config.get("rx_volume", 80)
        self._voice_tx_volume: int = self._voice_config.get("tx_volume", 80)
        self._ptt_active = False
        self._ping_wav_path = self._generate_ping_wav()

        # PTT-Taste aus Config (Standardmäßig V)
        self._ptt_key_name: str = self._voice_config.get("atc_ptt_key", "V")
        self._ptt_qt_key = self._resolve_ptt_key(self._ptt_key_name)

        # Globaler Event-Filter: PTT funktioniert überall, egal welches
        # Widget gerade den Fokus hat.
        QApplication.instance().installEventFilter(self)

        # Joystick-PTT (optional)
        self._joy_id: int = self._voice_config.get("atc_joystick_id", -1)
        self._joy_button: int = self._voice_config.get("atc_joystick_button", -1)
        self._joy_thread = None  # wird in _setup_voice_dock gestartet

        # --- Voice Control Panel (QDockWidget, rechts angedockt) ---
        self._setup_voice_dock()

        # Joystick-PTT Thread starten (falls konfiguriert)
        self._start_atc_joy_thread()

        # Statusbar
        self.status = QStatusBar()
        self.status.setStyleSheet("color: #4a9edd; background: #111;")
        self.setStatusBar(self.status)
        self.status.showMessage("Verbinde …")

        # WebSocket
        self.ws = QWebSocket()
        self.ws.textMessageReceived.connect(self._on_message)
        self.ws.connected.connect(self._on_connected)
        self.ws.disconnected.connect(self._on_disconnected)
        self.ws.open(QUrl(server_url))
        self.server_url = server_url

        # Voice-WebSocket (für VOICE_STATE)
        voice_url = server_url.replace("/ws/atc", "/ws/voice")
        self.voice_ws = QWebSocket()
        self.voice_ws.textMessageReceived.connect(self._on_voice_message)
        self.voice_ws.connected.connect(self._on_voice_connected)
        self.voice_ws.disconnected.connect(self._on_voice_disconnected)
        self.voice_ws.open(QUrl(voice_url))
        # (Voice-Variablen bereits oben vor _setup_voice_dock initialisiert)

        # Timer
        self.timer = QTimer(self)
        self.timer.timeout.connect(self._tick)
        self.timer.start(100)

        # Blink-Timer (500ms für Handover-Blinken)
        self.blink_timer = QTimer(self)
        self.blink_timer.timeout.connect(self._blink_tick)
        self.blink_timer.start(500)

        # Reconnect-Timer
        self.reconnect_timer = QTimer(self)
        self.reconnect_timer.timeout.connect(self._try_reconnect)

        # Pending handover incoming
        self._handover_incoming: Dict[str, dict] = {}
        self._online_atc_stations: list = []  # Online ATC für Handover-Menü

    def _load_sector(self, path: str):
        try:
            self.radar.sector = parse_sct_file(path)
            n = (len(self.radar.sector.artcc_lines)
                 + len(self.radar.sector.artcc_high_lines)
                 + len(self.radar.sector.artcc_low_lines))
            nr = len(self.radar.sector.regions) if self.radar.sector.regions else 0
            nla = len(self.radar.sector.low_airways) if self.radar.sector.low_airways else 0
            nha = len(self.radar.sector.high_airways) if self.radar.sector.high_airways else 0
            print(f"Sektor geladen: {path}  ({n} ARTCC-Linien, "
                  f"{len(self.radar.sector.runways)} Runways, "
                  f"{len(self.radar.sector.all_navaids)} Navaids, "
                  f"{nr} Regions, {nla} Low Airways, {nha} High Airways)")
            # Auto-center on sector info if available
            info = self.radar.sector.info
            if info and (info.center_lat != 0 or info.center_lon != 0):
                self.radar.center_lat = info.center_lat
                self.radar.center_lon = info.center_lon
                if info.nm_range > 0:
                    self.radar.scale = max(30, min(50000, 60 * 100 / info.nm_range))
            self.radar.invalidate_bg_cache()
        except Exception as e:
            print(f"Warnung: Sektordatei konnte nicht geladen werden: {e}")
            import traceback; traceback.print_exc()

    # -- WebSocket -----------------------------------------------------------
    def _on_connected(self):
        self.status.showMessage("Verbunden mit Server")
        self.reconnect_timer.stop()

    def _on_disconnected(self):
        self.status.showMessage("Verbindung verloren – Reconnect in 3 s …")
        self.reconnect_timer.start(3000)

    def _try_reconnect(self):
        try:
            self.ws.abort()  # Alte Verbindung sauber abbrechen
        except Exception:
            pass
        self.ws.open(QUrl(self.server_url))

    def _on_message(self, text: str):
        try:
            data = json.loads(text)
        except json.JSONDecodeError:
            return

        # Typed Messages
        if isinstance(data, dict) and "type" in data:
            self._handle_typed_message(data)
            return

        if isinstance(data, list):
            for item in data:
                self._update_pilot(item)
        else:
            self._update_pilot(data)

    def _handle_typed_message(self, data: dict):
        msg_type = data.get("type", "")

        if msg_type == "ACTIVE_ATC":
            stations = data.get("stations", [])
            self._online_atc_stations = stations
            self.status.showMessage(f"Online ATC: {len(stations)}")
            # Sektor-Hervorhebung aktualisieren
            self.radar.invalidate_bg_cache()
            self.radar.update()

            # Auto-Fill: Eigene CTR-Frequenz vom Server aktualisieren
            # (Server ergänzt automatisch CENTER_STATIONS / globalDB-Freq)
            my_station = self.radar.my_station
            if my_station and (self.radar.atc_frequency == 0.0
                               or not self.radar.atc_frequency):
                for st in stations:
                    if st.get("station") == my_station:
                        srv_freq = st.get("frequency", 0.0)
                        if srv_freq and srv_freq > 0:
                            self.radar.atc_frequency = srv_freq
                            # UI aktualisieren
                            if hasattr(self, 'txt_ctrl_freq'):
                                self.txt_ctrl_freq.setText(f"{srv_freq:.3f}")
                            if hasattr(self, 'lbl_ctrl_status'):
                                self.lbl_ctrl_status.setText(
                                    f"✓ {my_station} ({srv_freq:.3f} MHz)")
                            self.status.showMessage(
                                f"🌍 CTR-Frequenz auto-zugewiesen: "
                                f"{srv_freq:.3f} MHz")
                            break

        elif msg_type == "HANDOVER_INCOMING":
            cs = data.get("callsign", "")
            from_st = data.get("from_station", "")
            if cs:
                self._handover_incoming[cs] = data
                pilot = self.radar.pilots.get(cs)
                if pilot:
                    pilot.handover_blink = True
                    pilot.handover_from = from_st
                self.status.showMessage(f"⚠ HANDOVER eingehend: {cs} von {from_st}")
                # Notification Popup
                info = f"Incoming Traffic: {cs}"
                if pilot:
                    info += f"\nAlt: {int(pilot.alt)} ft  |  GS: {int(pilot.groundspeed)} kn"
                    if pilot.dep or pilot.arr:
                        info += f"\n{pilot.dep} → {pilot.arr}"
                info += f"\n\nVon: {from_st}"
                info += "\n\nRechtsklick auf das Flugzeug → 'Handover ANNEHMEN'"
                msg_box = QMessageBox(self)
                msg_box.setWindowTitle("⚠ Incoming Traffic")
                msg_box.setText(info)
                msg_box.setIcon(QMessageBox.Icon.Information)
                msg_box.setStyleSheet(
                    "QMessageBox { background: #0a1a2a; color: #ffcc44; }"
                    "QPushButton { background: #0f3460; color: #c8d6e5; padding: 6px 16px; }")
                msg_box.setStandardButtons(QMessageBox.StandardButton.Ok)
                msg_box.show()

        elif msg_type == "HANDOVER_COMPLETE":
            cs = data.get("callsign", "")
            new_ctrl = data.get("new_controller", "")
            self._handover_incoming.pop(cs, None)
            pilot = self.radar.pilots.get(cs)
            if pilot:
                pilot.handover_blink = False
                pilot.controller = new_ctrl
            self.status.showMessage(f"✓ Handover abgeschlossen: {cs} → {new_ctrl}")

        elif msg_type == "HANDOVER_REJECTED":
            cs = data.get("callsign", "")
            by = data.get("by", "")
            self.status.showMessage(f"✗ Handover abgelehnt: {cs} von {by}")

        elif msg_type == "ATC_UPDATE_ACK":
            cs = data.get("callsign", "")
            pilot = self.radar.pilots.get(cs)
            if pilot:
                if "assigned_squawk" in data:
                    pilot.assigned_squawk = data["assigned_squawk"]
                if "assigned_alt" in data:
                    pilot.assigned_alt = data["assigned_alt"]
                if "cleared" in data:
                    pilot.cleared = data["cleared"]

        elif msg_type == "FREQ_MSG":
            sender = data.get("from", "?")
            message = data.get("message", "")
            freq = data.get("frequency", 0)
            self._append_freq_chat(sender, message, freq)

        elif msg_type == "WEATHER_SYNC":
            metars = data.get("metars", {})
            self.radar._server_weather = metars
            log.info("🌦️  %d METARs vom Server empfangen", len(metars))
            # Wenn ein Flughafen-ICAO gesetzt ist, METAR-Leiste aktualisieren
            icao = self._current_weather_icao
            if icao and icao in metars:
                m = metars[icao]
                raw = m.get("raw", "")
                self.radar._metar_bar_text = raw
                wd = m.get("wind_dir", 0)
                ws = m.get("wind_speed", 0)
                self.radar._wind_dir = wd
                self.radar._wind_speed = ws
            elif metars:
                # Kein spezifischer ICAO → ersten METAR nehmen
                first = next(iter(metars.values()))
                self.radar._metar_bar_text = first.get("raw", "")
            self.radar.invalidate_bg_cache()
            self.radar.update()

        elif msg_type == "CPDLC_SENT_ACK":
            # Bestätigung: Nachricht wurde an Piloten gesendet
            cs = data.get("callsign", "")
            mid = data.get("msg_id", "")
            pilot = self.radar.pilots.get(cs)
            if pilot:
                pilot.cpdlc_pending = True
                pilot.cpdlc_last_status = ""
            self._cpdlc_log_append(cs, "SENT", data.get("text", ""), mid)
            self.status.showMessage(f"📡 CPDLC → {cs}: gesendet [{mid}]")

        elif msg_type == "CPDLC_RESPONSE":
            # Antwort vom Piloten (WILCO/UNABLE/STANDBY)
            cs = data.get("callsign", "")
            resp = data.get("response", "")
            mid = data.get("msg_id", "")
            pilot = self.radar.pilots.get(cs)
            if pilot:
                pilot.cpdlc_pending = (resp == "STANDBY")
                pilot.cpdlc_last_status = resp
            self._cpdlc_log_append(cs, resp, data.get("text", ""), mid)
            symbols = {"WILCO": "✓", "UNABLE": "✗", "STANDBY": "⏳"}
            sym = symbols.get(resp, "?")
            self.status.showMessage(f"📡 CPDLC {cs}: {sym} {resp} [{mid}]")

        elif msg_type == "FREQ_CONTACT":
            # Pilot hat unsere Frequenz gerastet → INBOUND-Strip erstellen
            cs = data.get("callsign", "")
            if cs:
                self._add_inbound_strip(
                    cs,
                    dep=data.get("dep", ""),
                    arr=data.get("arr", ""),
                    actype=data.get("actype", ""),
                    freq=data.get("frequency", 0),
                    alt=data.get("alt", 0),
                )
                self.status.showMessage(
                    f"📡 FREQ_CONTACT: {cs} auf deiner Frequenz!")

        elif msg_type == "PILOT_TUNED_IN":
            # Pilot hat TUNE-Button gedrückt → visuelles Feedback
            cs = data.get("callsign", "")
            freq = data.get("frequency", 0)
            if cs:
                # Pilot im Radar grün markieren (contacted_flash)
                if cs in self.radar.pilots:
                    self.radar.pilots[cs].contacted_flash = 60  # ≈3s
                    self.radar.pilots[cs].contact_mismatch = False
                self.status.showMessage(
                    f"✅ {cs} hat auf {freq:.3f} MHz getuned — Voice aktiv!")
                # Sound-Feedback
                try:
                    import winsound
                    winsound.Beep(1200, 150)
                    winsound.Beep(1600, 150)
                except Exception:
                    pass

        elif msg_type == "FREQ_LEAVE":
            # Pilot hat Frequenz verlassen → Strip entfernen
            cs = data.get("callsign", "")
            if cs:
                self._remove_inbound_strip(cs)
                self._remove_outbound_strip(cs)
                self.status.showMessage(
                    f"📡 FREQ_LEAVE: {cs} hat Frequenz verlassen")

        elif msg_type == "STCA_ALERT":
            # STCA-Konflikte aktualisieren
            alerts = data.get("alerts", [])
            # Alle bisherigen STCA-Flags zurücksetzen
            for pilot in self.radar.pilots.values():
                pilot.stca_alert = False
                pilot.stca_partner = ""
            # Neue Konflikte setzen
            for a in alerts:
                cs_a = a.get("callsign_a", "")
                cs_b = a.get("callsign_b", "")
                if cs_a in self.radar.pilots:
                    self.radar.pilots[cs_a].stca_alert = True
                    self.radar.pilots[cs_a].stca_partner = cs_b
                if cs_b in self.radar.pilots:
                    self.radar.pilots[cs_b].stca_alert = True
                    self.radar.pilots[cs_b].stca_partner = cs_a
            if alerts:
                self.status.showMessage(
                    f"🔴 STCA: {len(alerts)} Konflikte erkannt!")

        elif msg_type == "EVENT_SLOT_UPDATE":
            # Slot-Status-Update für Piloten
            cs = data.get("callsign", "")
            slot_time = data.get("slot_time", "")
            slot_status = data.get("slot_status", "")
            if cs and cs in self.radar.pilots:
                self.radar.pilots[cs].slot_time = slot_time
                self.radar.pilots[cs].slot_status = slot_status
                self.status.showMessage(
                    f"🎫 Slot-Update: {cs} → {slot_time} ({slot_status})")

        elif msg_type == "SLOT_BOOKED":
            # Neuer Slot gebucht
            cs = data.get("callsign", "")
            slot_time = data.get("slot_time", "")
            if cs and cs in self.radar.pilots:
                self.radar.pilots[cs].slot_time = slot_time
                self.radar.pilots[cs].slot_status = "ON_TIME"
                self.status.showMessage(
                    f"🎫 Slot gebucht: {cs} → {slot_time}")

        elif msg_type == "FLOW_RATE":
            apt = data.get("airport", "")
            dep = data.get("dep_rate_h", 0)
            arr = data.get("arr_rate_h", 0)
            self.status.showMessage(
                f"📊 Flow-Rate: {apt} → DEP {dep}/h, ARR {arr}/h")

        elif msg_type == "PILOT_METAR":
            # Pilot hat sich eingeloggt – METAR + Runway-Empfehlung
            cs = data.get("callsign", "")
            icao = data.get("icao", "")
            raw = data.get("metar_raw", "")
            rwy = data.get("recommended_runway", "")
            wind_dir = data.get("wind_dir", 0)
            wind_spd = data.get("wind_speed_kt", 0)
            qnh = data.get("qnh", 0)
            vis = data.get("visibility_m", 9999)
            wx = data.get("wx", "")

            # METAR-Daten speichern (für ATIS-Nutzung)
            if not hasattr(self, '_pilot_metars'):
                self._pilot_metars = {}
            self._pilot_metars[icao] = data

            # Kompakte Status-Meldung
            rwy_text = f" RWY {rwy}" if rwy else ""
            wx_text = f" {wx}" if wx else ""
            vis_text = f" VIS {vis}m" if vis < 9999 else ""
            self.status.showMessage(
                f"🌦️ {cs} LOGIN → {icao}: "
                f"{wind_dir:03d}°/{wind_spd}kt QNH {qnh}"
                f"{vis_text}{wx_text}{rwy_text}")

            # Log im Frequenz-Chat
            import time as _t
            ts = _t.strftime("%H:%M:%S")
            self.txt_freq_chat.append(
                f'<span style="color:#8b949e">[{ts}]</span> '
                f'<span style="color:#64c8ff;font-weight:bold">🌦️ METAR {icao}:</span> '
                f'<span style="color:#c8d6e5">{raw[:80]}</span>')
            if rwy:
                self.txt_freq_chat.append(
                    f'<span style="color:#8b949e">[{ts}]</span> '
                    f'<span style="color:#3fb950;font-weight:bold">🛬 RWY-Empfehlung {icao}:</span> '
                    f'<span style="color:#c8d6e5">{rwy} (Wind {wind_dir:03d}°/{wind_spd}kt)</span>')
            sb = self.txt_freq_chat.verticalScrollBar()
            sb.setValue(sb.maximum())

        elif msg_type == "CONTACT_MISMATCH":
            # Pilot im eigenen Bereich, aber falsche Frequenz
            cs = data.get("callsign", "")
            expected_freq = data.get("expected_freq", 0.0)
            pilot_freq = data.get("pilot_freq", 0.0)
            dist = data.get("distance_nm", 0)
            if cs and cs in self.radar.pilots:
                self.radar.pilots[cs].contact_mismatch = True
                self.radar.pilots[cs].contact_expected_freq = expected_freq
                self.status.showMessage(
                    f"⚠ CONTACT MISMATCH: {cs} auf {pilot_freq:.3f}"
                    f" (soll: {expected_freq:.3f}) – {dist:.0f}nm")

        elif msg_type == "voice_status":
            # ===== ATC Auto-Sector-Move: Voice-Kanal Bestätigung =====
            channel = data.get("channel", "")
            freq = data.get("frequency", 0.0)
            center_name = data.get("center_name", "")
            status = data.get("status", "")
            voice_msg = data.get("message", "")

            # Interne State speichern
            self._voice_channel = channel
            self._voice_freq = freq
            self._voice_center_name = center_name
            self._voice_status = status

            # Label-Text aufbauen
            if status == "joined":
                freq_str = f"{freq:.3f}" if freq > 0 else ""
                name_str = f" ({center_name})" if center_name else ""
                label_txt = f"🎤 VOICE: {channel}{name_str}"
                if freq_str:
                    label_txt += f"  [{freq_str} MHz]"
                label_txt += "  verbunden"
                color = "#3fb950"  # grün
            elif status == "pending":
                label_txt = f"🎤 VOICE: {channel} — Kanal bereit (Mumble-Beitritt ausstehend)"
                color = "#d29922"  # gelb
            else:
                label_txt = f"🎤 VOICE: {voice_msg}"
                color = "#f85149"  # rot

            self.lbl_voice_status.setText(label_txt)
            self.lbl_voice_status.setStyleSheet(
                f"color:{color}; font-weight:bold; padding:4px;")

            log.info("🎙️  Voice-Status: %s (status=%s)", label_txt, status)
            self.status.showMessage(f"🎤 {label_txt}", 8000)

    def _update_pilot(self, data: dict):
        cs = data.get("callsign")
        if not cs:
            return

        if data.get("event") == "disconnect":
            self.radar.pilots.pop(cs, None)
            self._remove_strip(cs)
            self._remove_inbound_strip(cs)
            self._remove_outbound_strip(cs)
            if self.radar.selected_callsign == cs:
                self.radar.selected_callsign = None
            return

        pilot = self.radar.pilots.get(cs)
        if pilot is None:
            pilot = PilotInfo(cs, data["lat"], data["lon"], data.get("alt", 0))
            self.radar.pilots[cs] = pilot
            if len(self.radar.pilots) == 1:
                self.radar.center_lat = data["lat"]
                self.radar.center_lon = data["lon"]
                self.radar.scale = 200.0
                self.radar.invalidate_bg_cache()
        else:
            pilot.lat = data["lat"]
            pilot.lon = data["lon"]
            pilot.alt = data.get("alt", 0)

        for field in ("dep", "arr", "actype", "route", "cruise_alt",
                       "heading", "groundspeed", "com1_freq", "squawk",
                       "squawk_ident", "controller", "controller_role",
                       "ias", "vs", "sel_alt"):
            if field in data:
                setattr(pilot, field, data[field])

        # ATC-Anweisungen
        if "assigned_squawk" in data:
            pilot.assigned_squawk = data["assigned_squawk"]
        if "assigned_alt" in data:
            pilot.assigned_alt = data["assigned_alt"]
        if "cleared" in data:
            pilot.cleared = data["cleared"]
        if "kohlsman_mb" in data:
            pilot.kohlsman_mb = data["kohlsman_mb"]
        if "transponder_mode" in data:
            pilot.transponder_mode = data["transponder_mode"]

        # Event / Slot-Daten
        if "slot_time" in data:
            pilot.slot_time = data["slot_time"]
        if "slot_status" in data:
            pilot.slot_status = data["slot_status"]

        # ── Gate-Daten (vom Server) ──
        if "gate" in data:
            pilot.gate = data["gate"]
        if "gate_terminal" in data:
            pilot.gate_terminal = data["gate_terminal"]
        if "gate_dist_m" in data:
            pilot.gate_dist_m = float(data["gate_dist_m"])

        # ── Contact-Mismatch auto-clear ──
        # Wenn Pilot jetzt auf meiner Frequenz ist → Mismatch aufheben
        if (pilot.contact_mismatch
                and self.radar.atc_frequency > 0
                and abs(pilot.com1_freq - self.radar.atc_frequency) < 0.01):
            pilot.contact_mismatch = False
            pilot.contact_expected_freq = 0.0
            # Grüner "CONTACTED" Flash (3s)
            import time as _t_contacted
            pilot.contacted_flash = _t_contacted.time()
            self.status.showMessage(
                f"✅ {cs} hat Frequenz gewechselt – CONTACTED", 5000)

        # ── Geofencing: Grenzwarnung ──
        # Pilot auf meiner Frequenz, der sich FL245 nähert (±2000ft) ohne Handover
        if self.radar.role == "CTR" and self.radar.atc_frequency > 0:
            on_my_freq = abs(pilot.com1_freq - self.radar.atc_frequency) < 0.01
            near_fl245 = 22500 < pilot.alt < 26500
            has_handover = pilot.handover_blink
            pilot.boundary_alert = on_my_freq and near_fl245 and not has_handover
        elif self.radar.role in ("APP", "TWR"):
            # APP/TWR: Warnung wenn Pilot über FL245 steigt ohne Handover
            on_my_freq = abs(pilot.com1_freq - self.radar.atc_frequency) < 0.01
            above_fl245 = pilot.alt > 24500
            has_handover = pilot.handover_blink
            pilot.boundary_alert = on_my_freq and above_fl245 and not has_handover
        else:
            pilot.boundary_alert = False

        # Strip aktualisieren
        self._update_strip(pilot)

    # -- Flight Strips -------------------------------------------------------
    def _update_strip(self, pilot: PilotInfo):
        """Strip in der linken Leiste aktualisieren oder erstellen."""
        # Nur Piloten auf meiner Frequenz anzeigen (wenn Freq gesetzt)
        if self.radar.atc_frequency > 0:
            if abs(pilot.com1_freq - self.radar.atc_frequency) > 0.01:
                self._remove_strip(pilot.callsign)
                return

        item = self._strip_items.get(pilot.callsign)
        if item is None:
            item = StripItem(pilot.callsign)
            self._strip_items[pilot.callsign] = item
            self.strip_layout.insertWidget(self.strip_layout.count() - 1, item)
        item.update_data(pilot)

    def _remove_strip(self, callsign: str):
        item = self._strip_items.pop(callsign, None)
        if item:
            self.strip_layout.removeWidget(item)
            item.deleteLater()

    # -- INBOUND / OUTBOUND Auto-Strips ------------------------------------
    def _add_inbound_strip(self, callsign: str, dep: str = "", arr: str = "",
                           actype: str = "", freq: float = 0, alt: float = 0):
        """Erstellt einen rot blinkenden INBOUND-Strip."""
        if callsign in self._inbound_items:
            # Nur Info updaten
            self._inbound_items[callsign].set_info(dep, arr, freq, actype, alt)
            return
        strip = AutoStripItem(callsign, "INBOUND")
        strip.set_info(dep, arr, freq, actype, alt)
        strip.clicked.connect(self._on_inbound_clicked)
        self._inbound_items[callsign] = strip
        self.inbound_layout.insertWidget(self.inbound_layout.count() - 1, strip)

    def _remove_inbound_strip(self, callsign: str):
        item = self._inbound_items.pop(callsign, None)
        if item:
            self.inbound_layout.removeWidget(item)
            item.deleteLater()

    def _add_outbound_strip(self, callsign: str, dep: str = "", arr: str = "",
                            actype: str = "", freq: float = 0, alt: float = 0):
        """Erstellt einen OUTBOUND-Strip (Pilot wurde transferiert)."""
        if callsign in self._outbound_items:
            self._outbound_items[callsign].set_info(dep, arr, freq, actype, alt)
            return
        strip = AutoStripItem(callsign, "OUTBOUND")
        strip.set_info(dep, arr, freq, actype, alt)
        self._outbound_items[callsign] = strip
        self.outbound_layout.insertWidget(self.outbound_layout.count() - 1, strip)

    def _remove_outbound_strip(self, callsign: str):
        item = self._outbound_items.pop(callsign, None)
        if item:
            self.outbound_layout.removeWidget(item)
            item.deleteLater()

    def _on_inbound_clicked(self, callsign: str):
        """INBOUND-Strip wurde angeklickt → Pilot auswählen im Radar."""
        self.radar.selected_callsign = callsign
        pilot = self.radar.pilots.get(callsign)
        if pilot:
            self.strip.display_pilot(pilot)
            self.radar.center_lat = pilot.lat
            self.radar.center_lon = pilot.lon
            self.radar.invalidate_bg_cache()
            self.radar.update()
        self.status.showMessage(f"✈ INBOUND bestätigt: {callsign}")

    def _blink_inbound_strips(self):
        """Timer-Callback: Blinkt alle unbestätigten INBOUND-Strips."""
        for strip in self._inbound_items.values():
            strip.do_blink()
        self.radar.update()

    # -- Voice Dock Widget -----------------------------------------------------
    def _setup_voice_dock(self):
        """Erstellt das Voice Control Panel als abdockbares QDockWidget."""
        self.voice_dock = QDockWidget("🎙 Voice Control", self)
        self.voice_dock.setObjectName("VoiceControlDock")
        self.voice_dock.setStyleSheet("""
            QDockWidget { color: #8ba4b8; font-weight: bold; font-size: 11px; }
            QDockWidget::title {
                background: #161b22; border: 1px solid #30363d;
                padding: 4px; text-align: left;
            }
        """)

        voice_widget = QWidget()
        voice_vbox = QVBoxLayout(voice_widget)
        voice_vbox.setContentsMargins(6, 6, 6, 6)
        voice_vbox.setSpacing(4)

        self.lbl_voice_status = QLabel("⚫ Nicht verbunden")
        self.lbl_voice_status.setStyleSheet(
            "color: #888; font-size: 10px; font-family: Consolas;")
        voice_vbox.addWidget(self.lbl_voice_status)

        # PTT LED – groß, auffällig, sofort sichtbar
        self.ptt_led = QLabel("⚫  STANDBY")
        self.ptt_led.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.ptt_led.setFixedHeight(38)
        self.ptt_led.setStyleSheet("""
            QLabel {
                background: #111; color: #555;
                font-size: 16px; font-weight: bold; font-family: Consolas;
                border: 2px solid #30363d; border-radius: 8px;
                padding: 4px;
            }
        """)
        voice_vbox.addWidget(self.ptt_led)

        # ── PTT-Taste Auswahl ──────────────────────────────────────────
        ptt_config_row = QHBoxLayout()
        ptt_lbl = QLabel("⌨ PTT:")
        ptt_lbl.setStyleSheet("color: #8ba4b8; font-size: 9px; font-family: Consolas;")
        ptt_lbl.setFixedWidth(42)
        self.combo_ptt = QComboBox()
        self.combo_ptt.addItems([
            "V", "Strg (Links)", "Strg (Rechts)",
            "Shift (Links)", "Shift (Rechts)",
            "Leertaste", "Caps Lock",
            "F1", "F2", "F3", "F4", "F5",
        ])
        # Gespeicherte Taste setzen
        idx = self.combo_ptt.findText(self._ptt_key_name)
        if idx >= 0:
            self.combo_ptt.setCurrentIndex(idx)
        self.combo_ptt.setStyleSheet("""
            QComboBox { background: #161b22; color: #c8d6e5;
                        border: 1px solid #30363d; border-radius: 3px;
                        padding: 2px 4px; font-size: 9px; font-family: Consolas; }
            QComboBox::drop-down { border: none; }
            QComboBox QAbstractItemView { background: #0d1117; color: #c8d6e5;
                                          selection-background-color: #1a3a5c; }
        """)
        self.combo_ptt.currentTextChanged.connect(self._on_ptt_key_changed)
        ptt_config_row.addWidget(ptt_lbl)
        ptt_config_row.addWidget(self.combo_ptt)

        # Joystick-PTT Button
        self.btn_joy_ptt = QPushButton("🎮")
        self.btn_joy_ptt.setFixedWidth(30)
        self.btn_joy_ptt.setToolTip("Joystick-Button für PTT zuweisen")
        self.btn_joy_ptt.setStyleSheet("""
            QPushButton { background: #161b22; color: #c8d6e5;
                          border: 1px solid #30363d; border-radius: 3px;
                          font-size: 12px; }
            QPushButton:hover { background: #1a3a5c; border-color: #58a6ff; }
        """)
        self.btn_joy_ptt.clicked.connect(self._on_atc_joy_learn)
        # Falls Joystick schon konfiguriert
        if self._joy_id >= 0 and self._joy_button >= 0:
            self.btn_joy_ptt.setText("✅")
            self.btn_joy_ptt.setToolTip(
                f"Joystick {self._joy_id} · Button {self._joy_button}")
        ptt_config_row.addWidget(self.btn_joy_ptt)
        voice_vbox.addLayout(ptt_config_row)

        # Kanal-Baum (Frequenz → Piloten)
        self.tree_voice = QTreeWidget()
        self.tree_voice.setHeaderLabels(["Kanal / User", "Status"])
        self.tree_voice.setColumnWidth(0, 160)
        self.tree_voice.setMinimumHeight(150)
        self.tree_voice.setStyleSheet("""
            QTreeWidget { background: #0d1117; color: #c8d6e5;
                          border: 1px solid #30363d; font-family: Consolas;
                          font-size: 10px; border-radius: 3px; }
            QTreeWidget::item { padding: 2px 0; }
            QTreeWidget::item:selected { background: #1a3a5c; }
            QHeaderView::section { background: #161b22; color: #8ba4b8;
                                    border: 1px solid #30363d; padding: 3px;
                                    font-size: 10px; }
        """)
        self.tree_voice.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.tree_voice.customContextMenuRequested.connect(self._voice_context_menu)
        voice_vbox.addWidget(self.tree_voice)

        # ── Lautstärke-Regler ───────────────────────────────────
        vol_style = """
            QSlider::groove:horizontal {
                height: 4px; background: #30363d; border-radius: 2px;
            }
            QSlider::handle:horizontal {
                width: 12px; height: 12px; margin: -4px 0;
                background: #58a6ff; border-radius: 6px;
            }
            QSlider::sub-page:horizontal { background: #58a6ff; border-radius: 2px; }
        """
        lbl_style = "color: #8ba4b8; font-size: 9px; font-family: Consolas;"

        # RX (Empfang)
        rx_row = QHBoxLayout()
        rx_lbl = QLabel("🔊 RX")
        rx_lbl.setStyleSheet(lbl_style)
        rx_lbl.setFixedWidth(36)
        self.slider_rx = QSlider(Qt.Orientation.Horizontal)
        self.slider_rx.setRange(0, 100)
        self.slider_rx.setValue(self._voice_rx_volume)
        self.slider_rx.setStyleSheet(vol_style)
        self.lbl_rx_val = QLabel(f"{self._voice_rx_volume}%")
        self.lbl_rx_val.setStyleSheet(lbl_style)
        self.lbl_rx_val.setFixedWidth(28)
        self.slider_rx.valueChanged.connect(self._on_rx_volume)
        rx_row.addWidget(rx_lbl)
        rx_row.addWidget(self.slider_rx)
        rx_row.addWidget(self.lbl_rx_val)
        voice_vbox.addLayout(rx_row)

        # TX (Mikrofon)
        tx_row = QHBoxLayout()
        tx_lbl = QLabel("🎤 TX")
        tx_lbl.setStyleSheet(lbl_style)
        tx_lbl.setFixedWidth(36)
        self.slider_tx = QSlider(Qt.Orientation.Horizontal)
        self.slider_tx.setRange(0, 100)
        self.slider_tx.setValue(self._voice_tx_volume)
        self.slider_tx.setStyleSheet(vol_style)
        self.lbl_tx_val = QLabel(f"{self._voice_tx_volume}%")
        self.lbl_tx_val.setStyleSheet(lbl_style)
        self.lbl_tx_val.setFixedWidth(28)
        self.slider_tx.valueChanged.connect(self._on_tx_volume)
        tx_row.addWidget(tx_lbl)
        tx_row.addWidget(self.slider_tx)
        tx_row.addWidget(self.lbl_tx_val)
        voice_vbox.addLayout(tx_row)

        voice_widget.setStyleSheet("background: #0d1117;")
        self.voice_dock.setWidget(voice_widget)

        # Rechts andocken
        self.addDockWidget(Qt.DockWidgetArea.RightDockWidgetArea, self.voice_dock)

        # Menü → Ansicht hinzufügen
        menu_bar = self.menuBar()
        menu_bar.setStyleSheet("""
            QMenuBar { background: #111; color: #8ba4b8; border-bottom: 1px solid #30363d; }
            QMenuBar::item:selected { background: #1a3a5c; }
            QMenu { background: #16213e; color: #c8d6e5; border: 1px solid #0f3460; }
            QMenu::item:selected { background: #1a5276; }
        """)
        view_menu = menu_bar.addMenu("Ansicht")
        toggle_voice = self.voice_dock.toggleViewAction()
        toggle_voice.setText("🎙 Voice Panel")
        toggle_voice.setChecked(True)
        view_menu.addAction(toggle_voice)

        # Trend-Vektor Toggle im Menü
        self._act_trend = view_menu.addAction("📐 Trend-Vektoren (X)")
        self._act_trend.setCheckable(True)
        self._act_trend.setChecked(self.radar.show_trend_vectors)
        self._act_trend.triggered.connect(self._toggle_trend_vectors)

        # Trend-Vektor Minuten Sub-Menü
        tv_menu = view_menu.addMenu("📐 Trend Minuten")
        for mins_set, label in [([1, 2], "1 + 2 min"),
                                 ([1, 2, 5], "1 + 2 + 5 min"),
                                 ([1, 2, 5, 10], "1 + 2 + 5 + 10 min"),
                                 ([2, 5], "2 + 5 min"),
                                 ([5, 10], "5 + 10 min")]:
            act = tv_menu.addAction(label)
            act.triggered.connect(lambda _, m=mins_set: self._set_trend_minutes(m))

    # -- Voice WebSocket ------------------------------------------------------
    def _toggle_trend_vectors(self, checked: bool):
        self.radar.show_trend_vectors = checked
        self.radar.update()

    def _set_trend_minutes(self, minutes: list):
        self.radar.trend_vector_minutes = minutes
        self.radar.update()
        self.status.showMessage(f"Trend-Vektoren: {minutes} min", 3000)

    def _on_voice_connected(self):
        self.lbl_voice_status.setText("🟢 Voice verbunden")
        self.lbl_voice_status.setStyleSheet(
            "color: #58a6ff; font-size: 10px; font-family: Consolas;")

    def _on_voice_disconnected(self):
        self.lbl_voice_status.setText("⚫ Voice getrennt")
        self.lbl_voice_status.setStyleSheet(
            "color: #888; font-size: 10px; font-family: Consolas;")
        # Reconnect nach 3 Sekunden
        QTimer.singleShot(3000, self._voice_reconnect)

    def _voice_reconnect(self):
        try:
            self.voice_ws.abort()  # Alte Verbindung sauber abbrechen
        except Exception:
            pass
        voice_url = self.server_url.replace("/ws/atc", "/ws/voice")
        self.voice_ws.open(QUrl(voice_url))

    def _on_voice_message(self, text: str):
        """Handle VOICE_STATE broadcasts from server."""
        try:
            data = json.loads(text)
        except json.JSONDecodeError:
            return
        if data.get("type") != "VOICE_STATE":
            return
        self._voice_state = data
        # Talking-State an RadarWidget weitergeben
        self.radar._talking_users = data.get("talking", {})
        self._check_voice_joins(data)
        self._update_voice_tree(data)

    def _check_voice_joins(self, state: dict):
        """Erkennt neue Piloten auf einer Frequenz und spielt Ping-Sound."""
        channels = state.get("channels", {})
        for ch_name, users in channels.items():
            current_set = set(users) if isinstance(users, list) else set()
            old_set = self._voice_known_users.get(ch_name, set())
            new_users = current_set - old_set
            if new_users and old_set:  # Nur pingen wenn schon jemand da war
                for nick in new_users:
                    self.status.showMessage(
                        f"🔔 {nick} hat Frequenz {ch_name} betreten", 5000)
                self._play_join_ping()
            self._voice_known_users[ch_name] = current_set
        # Aufräumen: Kanäle die nicht mehr existieren
        for old_ch in list(self._voice_known_users.keys()):
            if old_ch not in channels:
                del self._voice_known_users[old_ch]

    def _update_voice_tree(self, state: dict):
        """Refresh the voice channel tree widget."""
        channels = state.get("channels", {})
        talking = state.get("talking", {})

        self.tree_voice.clear()
        for ch_name, users in sorted(channels.items()):
            ch_item = QTreeWidgetItem(self.tree_voice, [f"📻 {ch_name}", f"{len(users)}"])
            ch_item.setForeground(0, QColor("#58a6ff"))
            ch_item.setForeground(1, QColor("#8ba4b8"))
            ch_item.setExpanded(True)
            for user_name in users:
                is_talking = talking.get(user_name, False)
                icon = "🟢" if is_talking else "⚪"
                status = "SPRICHT" if is_talking else ""
                user_item = QTreeWidgetItem(ch_item, [f"{icon} {user_name}", status])
                if is_talking:
                    user_item.setForeground(0, QColor("#3fb950"))
                    user_item.setForeground(1, QColor("#3fb950"))
                else:
                    user_item.setForeground(0, QColor("#c8d6e5"))
                    user_item.setForeground(1, QColor("#555"))
                user_item.setData(0, Qt.ItemDataRole.UserRole, user_name)

    def _voice_context_menu(self, pos):
        """Right-click on a voice tree user: Mute / Kick / Move."""
        item = self.tree_voice.itemAt(pos)
        if not item:
            return
        nick = item.data(0, Qt.ItemDataRole.UserRole)
        if not nick:
            return

        menu = QMenu(self)
        menu.setStyleSheet("""
            QMenu { background: #16213e; color: #c8d6e5; border: 1px solid #0f3460; }
            QMenu::item:selected { background: #1a5276; }
        """)
        act_mute = menu.addAction("🔇 Mute")
        act_mute.triggered.connect(lambda: self._voice_admin("VOICE_MUTE", nick))
        act_kick = menu.addAction("❌ Kick")
        act_kick.triggered.connect(lambda: self._voice_admin("VOICE_KICK", nick))

        # Move to frequency sub-menu
        move_menu = menu.addMenu("📻 Move to Frequency")
        if self.radar.atc_frequency > 0:
            f = f"{self.radar.atc_frequency:.3f}"
            act_f = move_menu.addAction(f"Meine Freq: {f}")
            act_f.triggered.connect(lambda: self._voice_admin("VOICE_MOVE", nick, f))

        # Custom frequency
        act_custom = move_menu.addAction("Andere Frequenz…")
        act_custom.triggered.connect(lambda: self._voice_move_custom(nick))

        menu.exec(QCursor.pos())

    def _voice_admin(self, cmd: str, nick: str, freq: str = ""):
        """Send a voice admin command via the voice WebSocket."""
        msg: dict = {"type": cmd, "nick": nick}
        if freq:
            msg["frequency"] = freq
        self.voice_ws.sendTextMessage(json.dumps(msg))
        self.status.showMessage(f"🎙 {cmd}: {nick} {freq}")

    def _voice_move_custom(self, nick: str):
        """Move user to a custom frequency (dialog)."""
        dlg = QInputDialog(self)
        dlg.setWindowTitle("Move to Frequency")
        dlg.setLabelText(f"Frequenz für {nick}:")
        dlg.setTextValue("122.800")
        dlg.setStyleSheet("""
            QInputDialog { background: #16213e; }
            QLabel { color: #c8d6e5; font-size: 13px; font-weight: bold; }
            QLineEdit { background: #0d1117; color: #ffffff; border: 1px solid #30363d;
                        padding: 6px; font-size: 14px; font-family: Consolas;
                        border-radius: 4px; }
            QPushButton { background: #21262d; color: #58a6ff; padding: 6px 16px;
                          border: 1px solid #30363d; border-radius: 4px; font-weight: bold; }
            QPushButton:hover { background: #30363d; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            freq = dlg.textValue().strip()
            if freq:
                self._voice_admin("VOICE_MOVE", nick, freq)

    # -- PTT-Taste Mapping ---------------------------------------------------
    _PTT_KEY_MAP: dict[str, int] = {
        "V": Qt.Key.Key_V,
        "Strg (Links)": Qt.Key.Key_Control,
        "Strg (Rechts)": Qt.Key.Key_Control,
        "Shift (Links)": Qt.Key.Key_Shift,
        "Shift (Rechts)": Qt.Key.Key_Shift,
        "Leertaste": Qt.Key.Key_Space,
        "Caps Lock": Qt.Key.Key_CapsLock,
        "F1": Qt.Key.Key_F1,
        "F2": Qt.Key.Key_F2,
        "F3": Qt.Key.Key_F3,
        "F4": Qt.Key.Key_F4,
        "F5": Qt.Key.Key_F5,
    }

    @classmethod
    def _resolve_ptt_key(cls, name: str) -> int:
        return cls._PTT_KEY_MAP.get(name, Qt.Key.Key_V)

    def _on_ptt_key_changed(self, text: str):
        """PTT-Taste wurde geändert."""
        self._ptt_key_name = text
        self._ptt_qt_key = self._resolve_ptt_key(text)
        self._save_voice_config()

    # -- Joystick-PTT für ATC ------------------------------------------------
    def _start_atc_joy_thread(self):
        """Startet den Joystick-PTT-Thread (falls konfiguriert)."""
        if self._joy_thread:
            return
        if self._joy_id < 0 or self._joy_button < 0:
            return
        try:
            import pygame
            pygame.init()
            pygame.joystick.init()
        except ImportError:
            return

        import threading as _thr

        class _JoyPoll(_thr.Thread):
            def __init__(s, joy_id, btn, win):
                super().__init__(daemon=True)
                s.joy_id = joy_id
                s.btn = btn
                s.win = win
                s._running = True
                s._was = False

            def run(s):
                try:
                    j = pygame.joystick.Joystick(s.joy_id)
                    j.init()
                except Exception:
                    return
                import time
                while s._running:
                    try:
                        pygame.event.pump()
                        p = j.get_button(s.btn)
                        if p and not s._was:
                            s._was = True
                            QTimer.singleShot(0, s.win._joy_ptt_on)
                        elif not p and s._was:
                            s._was = False
                            QTimer.singleShot(0, s.win._joy_ptt_off)
                    except Exception:
                        pass
                    time.sleep(0.015)

            def stop(s):
                s._running = False

        self._joy_thread = _JoyPoll(self._joy_id, self._joy_button, self)
        self._joy_thread.start()

    def _joy_ptt_on(self):
        if not self._ptt_active:
            self._ptt_active = True
            self.ptt_led.setText("🟢  SENDEN")
            self.ptt_led.setStyleSheet("""
                QLabel {
                    background: qlineargradient(x1:0,y1:0,x2:0,y2:1,
                        stop:0 #1a4a1a, stop:0.5 #2d8a2d, stop:1 #1a4a1a);
                    color: #4eff4e;
                    font-size: 16px; font-weight: bold; font-family: Consolas;
                    border: 2px solid #3fb950; border-radius: 8px;
                    padding: 4px;
                }
            """)

    def _joy_ptt_off(self):
        if self._ptt_active:
            self._ptt_active = False
            self.ptt_led.setText("⚫  STANDBY")
            self.ptt_led.setStyleSheet("""
                QLabel {
                    background: #111; color: #555;
                    font-size: 16px; font-weight: bold; font-family: Consolas;
                    border: 2px solid #30363d; border-radius: 8px;
                    padding: 4px;
                }
            """)

    def _on_atc_joy_learn(self):
        """Joystick-Lern-Modus für ATC PTT."""
        try:
            import pygame
            pygame.init()
            pygame.joystick.init()
        except ImportError:
            QMessageBox.warning(self, "Fehler",
                                "pygame nicht installiert.\npip install pygame")
            return
        n = pygame.joystick.get_count()
        if n == 0:
            QMessageBox.warning(self, "Fehler",
                                "Kein Joystick/Gamepad erkannt.")
            return

        self.btn_joy_ptt.setText("⏳")
        self.btn_joy_ptt.setEnabled(False)
        QApplication.processEvents()

        import threading as _thr

        def _worker():
            joys = []
            for i in range(pygame.joystick.get_count()):
                j = pygame.joystick.Joystick(i)
                j.init()
                joys.append(j)
            import time
            deadline = time.monotonic() + 15.0
            result = None
            while time.monotonic() < deadline:
                pygame.event.pump()
                for joy in joys:
                    for b in range(joy.get_numbuttons()):
                        if joy.get_button(b):
                            result = (joy.get_instance_id(), b, joy.get_name())
                            break
                    if result:
                        break
                if result:
                    break
                time.sleep(0.02)
            QTimer.singleShot(0, lambda: self._atc_joy_learned(result))

        _thr.Thread(target=_worker, daemon=True).start()

    def _atc_joy_learned(self, result):
        self.btn_joy_ptt.setEnabled(True)
        if result is None:
            self.btn_joy_ptt.setText("🎮")
            self.btn_joy_ptt.setToolTip("Timeout – kein Button erkannt")
            return
        joy_id, btn, name = result
        self._joy_id = joy_id
        self._joy_button = btn
        self.btn_joy_ptt.setText("✅")
        self.btn_joy_ptt.setToolTip(f"{name} · Button {btn}")
        self._save_voice_config()
        # Thread starten
        self._start_atc_joy_thread()

    # -- Lautstärke-Regler ---------------------------------------------------
    def _on_rx_volume(self, val: int):
        """RX-Lautstärke geändert → an Voice-System weitergeben + Config speichern."""
        self._voice_rx_volume = val
        self.lbl_rx_val.setText(f"{val}%")
        msg = {"type": "VOICE_RX_VOLUME", "volume": val / 100.0}
        self.voice_ws.sendTextMessage(json.dumps(msg))
        self._save_voice_config()

    def _on_tx_volume(self, val: int):
        """TX-Lautstärke geändert → an Voice-System weitergeben + Config speichern."""
        self._voice_tx_volume = val
        self.lbl_tx_val.setText(f"{val}%")
        msg = {"type": "VOICE_TX_VOLUME", "volume": val / 100.0}
        self.voice_ws.sendTextMessage(json.dumps(msg))
        self._save_voice_config()

    # -- Config Persistence --------------------------------------------------
    _CONFIG_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)),
                                "voice_config.json")

    def _load_voice_config(self) -> dict:
        """Lädt Voice-Einstellungen aus voice_config.json."""
        try:
            with open(self._CONFIG_FILE, "r", encoding="utf-8") as f:
                return json.load(f)
        except (FileNotFoundError, json.JSONDecodeError):
            return {"rx_volume": 80, "tx_volume": 80}

    def _save_voice_config(self):
        """Speichert Voice-Einstellungen in voice_config.json."""
        cfg = {
            "rx_volume": self._voice_rx_volume,
            "tx_volume": self._voice_tx_volume,
            "atc_ptt_key": self._ptt_key_name,
            "atc_joystick_id": self._joy_id,
            "atc_joystick_button": self._joy_button,
        }
        try:
            # Bestehende Config einlesen und mergen
            existing = self._load_voice_config()
            existing.update(cfg)
            with open(self._CONFIG_FILE, "w", encoding="utf-8") as f:
                json.dump(existing, f, indent=2, ensure_ascii=False)
        except Exception:
            pass

    # -- Ping-Sound (Pilot betritt Frequenz) ---------------------------------
    @staticmethod
    def _generate_ping_wav() -> str:
        """Erzeugt einmalig eine Ping-WAV-Datei und gibt den Pfad zurück."""
        import struct
        import tempfile
        import wave
        import math as _m

        ping_path = os.path.join(tempfile.gettempdir(), "nexus_atc_ping.wav")
        if os.path.isfile(ping_path):
            return ping_path

        sample_rate = 44100
        # Doppel-Ping: 100ms Ton, 60ms Pause, 100ms Ton
        segments: list[tuple[float, float, float]] = [
            # (dauer_s, freq_hz, amplitude)
            (0.10, 1175.0, 0.45),  # D6 – erster Ping
            (0.06, 0, 0),          # Pause
            (0.12, 1568.0, 0.35),  # G6 – zweiter Ping (höher)
        ]

        samples = bytearray()
        for dur, freq, amp in segments:
            n = int(sample_rate * dur)
            for i in range(n):
                t = i / sample_rate
                if freq > 0:
                    val = amp * _m.sin(2 * _m.pi * freq * t)
                    # Soft-Envelope: 5ms attack, exponential decay
                    attack = min(1.0, i / (sample_rate * 0.005))
                    decay = _m.exp(-3.0 * i / n)
                    val *= attack * decay
                else:
                    val = 0.0
                s = int(val * 32767)
                s = max(-32768, min(32767, s))
                samples.extend(struct.pack('<h', s))

        try:
            with wave.open(ping_path, 'wb') as wf:
                wf.setnchannels(1)
                wf.setsampwidth(2)
                wf.setframerate(sample_rate)
                wf.writeframes(bytes(samples))
        except Exception:
            pass
        return ping_path

    def _play_join_ping(self):
        """Spielt den vorgenerierten Ping-Sound ab."""
        if not self._ping_wav_path or not os.path.isfile(self._ping_wav_path):
            return
        try:
            import platform
            if platform.system() == "Windows":
                import winsound
                winsound.PlaySound(
                    self._ping_wav_path,
                    winsound.SND_FILENAME | winsound.SND_ASYNC | winsound.SND_NODEFAULT)
            else:
                import subprocess
                subprocess.Popen(
                    ["aplay", "-q", self._ping_wav_path],
                    stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        except Exception:
            pass

    # -- PTT (Push-to-Talk) Tastatur – globaler Event-Filter ----------------
    def eventFilter(self, obj, event):
        """PTT: Fängt die konfigurierte Taste applikationsweit ab."""
        etype = event.type()
        if etype == QEvent.Type.KeyPress:
            if event.key() == self._ptt_qt_key and not event.isAutoRepeat():
                self._joy_ptt_on()
        elif etype == QEvent.Type.KeyRelease:
            if event.key() == self._ptt_qt_key and not event.isAutoRepeat():
                self._joy_ptt_off()
        return super().eventFilter(obj, event)

    # -- Timer ---------------------------------------------------------------
    def _tick(self):
        self.radar.update()
        sel = self.radar.selected_callsign
        pilot = self.radar.pilots.get(sel) if sel else None
        self.strip.update_info(pilot)
        # Wetter-UI aktualisieren wenn neue Daten vorhanden
        self._update_weather_panel()

    def _blink_tick(self):
        self.radar._blink_on = not self.radar._blink_on

    # -- Controller-Registrierung -------------------------------------------
    def _on_station_text_changed(self, text: str):
        """Auto-Fill: Callsign parsen → Frequenz + Airport-Info anzeigen."""
        callsign = text.strip().upper()

        # Mindestens 4 Zeichen ICAO + Unterstrich + Rolle
        parts = callsign.split("_")
        icao = parts[0] if parts else ""

        # Airport-Info anzeigen ab 4 Buchstaben
        if len(icao) >= 4 and _GLOBAL_NAV_DB.exists():
            try:
                conn = sqlite3.connect(str(_GLOBAL_NAV_DB))
                row = conn.execute(
                    "SELECT name, kind, country, municipality FROM airports WHERE icao = ?",
                    (icao,)
                ).fetchone()
                conn.close()
                if row:
                    name = row[0] or ""
                    kind = row[1] or ""
                    country = row[2] or ""
                    city = row[3] or ""
                    kind_de = {"large_airport": "Großflughafen",
                               "medium_airport": "Regionalflughafen",
                               "small_airport": "Kleinflughafen",
                               "heliport": "Heliport",
                               "seaplane_base": "Wasserflughafen"}.get(kind, kind)
                    info_text = f"✈ {name}"
                    if city:
                        info_text += f"\n{city}, {country}"
                    info_text += f"\n{kind_de}"
                    self.lbl_airport_info.setText(info_text)
                    self.lbl_airport_info.setVisible(True)
                else:
                    self.lbl_airport_info.setText(f"⚠ '{icao}' nicht gefunden")
                    self.lbl_airport_info.setVisible(True)
            except Exception:
                self.lbl_airport_info.setVisible(False)
        else:
            self.lbl_airport_info.setVisible(False)

        # Auto-Fill: Rolle + Frequenz
        if len(parts) >= 2 and len(icao) >= 4:
            role = parts[1]
            freq, _, _ = fetch_frequency_for_callsign(callsign)

            # Rolle im Dropdown setzen
            role_idx = self.cmb_role.findText(role)
            if role_idx >= 0:
                self.cmb_role.setCurrentIndex(role_idx)

            # Frequenz automatisch einfüllen
            if freq:
                self.txt_ctrl_freq.setText(f"{freq:.3f}")
                self.txt_ctrl_freq.setStyleSheet(
                    "background: #0a1a30; border: 1px solid #c8d6e5;"
                    " color: #c8d6e5; padding: 3px; font-family: Consolas;"
                    " font-weight: bold;")
                # Nach 2 Sekunden Farbe zurücksetzen
                QTimer.singleShot(2000, lambda: self.txt_ctrl_freq.setStyleSheet(
                    "background: #16213e; border: 1px solid #0f3460;"
                    " color: #c8d6e5; padding: 3px; font-family: Consolas;"))
            else:
                self.txt_ctrl_freq.setStyleSheet(
                    "background: #16213e; border: 1px solid #0f3460;"
                    " color: #c8d6e5; padding: 3px; font-family: Consolas;")

    def _on_register_controller(self):
        # Bereits registriert? → Erst Logout!
        if self.radar.my_station:
            self.lbl_ctrl_status.setText("Erst ausloggen bevor du dich neu registrierst!")
            self.lbl_ctrl_status.setStyleSheet("color: #ff4444; font-size: 10px;")
            return

        ctrl_name = self.txt_ctrl_name.text().strip()
        station = self.txt_station.text().strip().upper()
        freq_text = self.txt_ctrl_freq.text().strip()
        role = self.cmb_role.currentText()

        if not ctrl_name:
            self.lbl_ctrl_status.setText("Name darf nicht leer sein!")
            self.lbl_ctrl_status.setStyleSheet("color: #ff4444; font-size: 10px;")
            return

        if not station:
            self.lbl_ctrl_status.setText("Station darf nicht leer sein!")
            self.lbl_ctrl_status.setStyleSheet("color: #ff4444; font-size: 10px;")
            return

        # ICAO parsen für Smart-Teleport
        parts = station.split("_")
        icao = parts[0] if parts else ""

        try:
            freq = float(freq_text) if freq_text else 0.0
        except ValueError:
            self.lbl_ctrl_status.setText("Ungültige Frequenz!")
            self.lbl_ctrl_status.setStyleSheet("color: #ff4444; font-size: 10px;")
            return

        self.radar.my_station = station
        self.radar.atc_frequency = freq
        self.radar.role = role

        # An Server senden (inkl. Position für Broadcasting)
        coords = None
        if _AIRPORT_DB:
            ap = _AIRPORT_DB.get(icao)
            if ap:
                coords = (ap.lat, ap.lon)

        msg = {
            "type": "ATC_REGISTER",
            "station": station,
            "frequency": freq,
            "role": role,
            "controller_name": ctrl_name,
        }
        if coords:
            msg["pos"] = list(coords)
        self.ws.sendTextMessage(json.dumps(msg))

        self.lbl_ctrl_status.setText(f"✓ {station} ({freq:.3f} MHz)")
        self.lbl_ctrl_status.setStyleSheet("color: #c8d6e5; font-size: 10px;")
        self.status.showMessage(f"Controller: {station} | {freq:.3f} MHz | {role}")

        # UI sperren: Logout aktivieren, Registrieren sperren
        self.btn_register.setEnabled(False)
        self.btn_logout.setEnabled(True)
        self.txt_ctrl_name.setEnabled(False)
        self.txt_station.setEnabled(False)
        self.txt_ctrl_freq.setEnabled(False)
        self.cmb_role.setEnabled(False)

        # Rolle anwenden (setzt Zoom + Layer)
        self._apply_role(role)

        # ---- Smart-Teleport: Karte zum Flughafen zentrieren ----
        if coords and len(icao) >= 4:
            # Zoom-Level je nach Rolle
            zoom_map = {
                "DEL": 2000.0, "GND": 2500.0, "TWR": 600.0,
                "APP": 200.0, "DEP": 200.0, "CTR": 60.0,
            }
            target_scale = zoom_map.get(role, 200.0)
            self.radar.smooth_goto(coords[0], coords[1], target_scale)

            # ICAO-Suche auslösen (Sektor laden, OSM, Wetter)
            self.txt_icao.setText(icao)
            self._on_search_icao()

    def _on_logout_controller(self):
        """Logout: WS trennen → Server entfernt ATC, dann reconnect als frischer Client."""
        old_station = self.radar.my_station
        self.radar.my_station = ""
        self.radar.atc_frequency = 0.0
        self.radar.role = "CTR"

        # WS kurz schließen, damit der Server ATC_OFFLINE sendet
        self.ws.close()
        # Server entfernt die Station automatisch on disconnect

        # UI zurücksetzen
        self.btn_register.setEnabled(True)
        self.btn_logout.setEnabled(False)
        self.txt_ctrl_name.setEnabled(True)
        self.txt_station.setEnabled(True)
        self.txt_ctrl_freq.setEnabled(True)
        self.cmb_role.setEnabled(True)
        self.lbl_ctrl_status.setText(f"Abgemeldet von {old_station}")
        self.lbl_ctrl_status.setStyleSheet("color: #ffaa00; font-size: 10px;")
        self.status.showMessage("Controller abgemeldet – bereit für neue Registrierung")

        # Reconnect nach 1s (frische WS-Verbindung)
        QTimer.singleShot(1000, self._try_reconnect)

    # -- ICAO-Suche ---------------------------------------------------------
    def _on_search_icao(self):
        icao = self.txt_icao.text().strip().upper()
        if not icao:
            return

        # 1. Globale Airport-DB versuchen
        coords = None
        if _AIRPORT_DB:
            ap = _AIRPORT_DB.get(icao)
            if ap:
                coords = (ap.lat, ap.lon)

        if coords:
            # Sanftes Zoomen zum Flughafen
            self.radar.smooth_goto(coords[0], coords[1], 800.0)
            self.status.showMessage(f"Zentriert auf {icao}")

            # 2. Passende Sektordatei suchen
            sector_loaded = False
            sector_dir = Path(__file__).parent / "sectors"
            if sector_dir.exists():
                for pattern in [f"{icao.lower()}*.sct", f"{icao.lower()}*.SCT",
                                f"{icao.upper()}*.sct", f"{icao.upper()}*.SCT",
                                f"*{icao.lower()}*.sct", f"*{icao.upper()}*.sct",
                                f"*{icao.lower()}*.SCT", f"*{icao.upper()}*.SCT"]:
                    matches = list(sector_dir.glob(pattern))
                    if matches:
                        self._load_sector(str(matches[0]))
                        sector_loaded = True
                        break

            # 3. Auto-Sektor generieren wenn keine .sct vorhanden
            if not sector_loaded and _AIRPORT_DB:
                auto = generate_simple_airport_layout(icao, _AIRPORT_DB)
                if auto:
                    self.radar.sector = auto
                    info = auto.info
                    if info and (info.center_lat != 0 or info.center_lon != 0):
                        self.radar.center_lat = info.center_lat
                        self.radar.center_lon = info.center_lon
                    # Taxiways, Labels, Regions automatisch einschalten
                    self.radar.show_taxiways = True
                    self.radar.show_labels = True
                    self.radar.show_regions = True
                    self.radar.show_geo = True
                    self.radar.invalidate_bg_cache()
                    self.status.showMessage(
                        f"Auto-Sektor: {icao} ({len(auto.runways)} RWY, "
                        f"{len(auto.regions)} Regions, "
                        f"{len(auto.labels)} Labels)")

            # 4. OSM Overpass – echte Taxiways/Aprons/Gates laden
            if _OSM_OVERPASS:
                self.radar._osm_icao = icao
                osm_cached = _OSM_OVERPASS.get_cached(icao)
                if osm_cached:
                    self.radar._osm_data[icao] = osm_cached
                    self.radar.invalidate_bg_cache()
                else:
                    _OSM_OVERPASS.request(icao, coords[0], coords[1])

            # 5. Wetter holen (Hintergrund)
            self._fetch_weather_async(icao)

            # 6. Search-Marker setzen (visueller Indikator + Frequenz)
            self.radar._search_marker_lat = coords[0]
            self.radar._search_marker_lon = coords[1]
            self.radar._search_marker_icao = icao
            self.radar._search_marker_timer = 90   # ~3 Sekunden bei 30fps

            # 7. Suggested Frequency + FIR aus GlobalNavMesh
            self._fetch_nav_info_async(icao, coords[0], coords[1])

        else:
            # Navaid / Waypoint suchen
            if _NAV_DB:
                results = _NAV_DB.search(icao)
                if results:
                    nav = results[0]
                    self.radar.smooth_goto(nav.lat, nav.lon, 300.0)
                    # Marker auch für Navaids
                    self.radar._search_marker_lat = nav.lat
                    self.radar._search_marker_lon = nav.lon
                    self.radar._search_marker_icao = f"{nav.ident} ({nav.kind})"
                    self.radar._search_marker_timer = 90
                    self.radar._search_marker_freq = nav.freq if nav.freq else 0
                    self.radar._search_marker_fir = ""
                    self.status.showMessage(
                        f"Navaid: {nav.ident} ({nav.kind}) – {nav.name}"
                        + (f" {nav.freq}" if nav.freq else ""))
                    return
            self.status.showMessage(f"{icao} nicht in Datenbank")

    def _fetch_nav_info_async(self, icao: str, lat: float, lon: float):
        """Frequenz-Vorschlag + FIR im Hintergrund laden (aus GlobalNavMesh)."""
        def _fetch():
            try:
                # Importiere GlobalNavMesh
                from import_global_data import get_global_nav
                nav = get_global_nav()
                if not nav or not nav.exists:
                    return

                # Suggested Frequency (TWR bevorzugt, dann APP, dann CTAF)
                freq = None
                for role in ("TWR", "APP", "CTAF", "ATIS"):
                    freq = nav.get_suggested_frequency(icao, role)
                    if freq:
                        break
                self.radar._search_marker_freq = freq or 0.0

                # FIR ermitteln
                firs = nav.get_fir_for_position(lat, lon)
                if firs:
                    self.radar._search_marker_fir = (
                        f"{firs[0].icao} – {firs[0].name}"
                        + (f" ({firs[0].frequency:.3f})" if firs[0].frequency else ""))
                else:
                    self.radar._search_marker_fir = ""

                # Status aktualisieren
                freq_str = f" | {freq:.3f} MHz" if freq else ""
                fir_str = f" | FIR: {firs[0].icao}" if firs else ""
                self.status.showMessage(
                    f"✈ {icao}{freq_str}{fir_str}")

            except Exception as e:
                log.debug("Nav-Info Fehler für %s: %s", icao, e)

        t = threading.Thread(target=_fetch, daemon=True)
        t.start()

    def _fetch_weather_async(self, icao: str):
        """Wetter im Hintergrund laden und UI aktualisieren."""
        def _fetch():
            try:
                m = _WEATHER.get_metar(icao)
                # Signal an Main-Thread
                self._current_metar = m
                self._current_weather_icao = icao
            except Exception as e:
                print(f"Wetter-Fehler: {e}")
        t = threading.Thread(target=_fetch, daemon=True)
        t.start()

    def _update_weather_panel(self):
        """Wetter-Panel im Main-Thread aktualisieren."""
        m = self._current_metar
        icao = self._current_weather_icao
        if m is None or icao is None:
            return
        # Nur einmal anzeigen pro Abruf
        if getattr(self, '_last_shown_weather', None) == (icao, m.raw):
            return
        self._last_shown_weather = (icao, m.raw)

        self.lbl_wx_icao.setText(f" {icao}")
        # Wind
        wd = m.wind_dir if m.wind_dir is not None else 0
        ws = m.wind_speed_kt if m.wind_speed_kt is not None else 0
        gust = f" G{m.wind_gust_kt}" if m.wind_gust_kt else ""
        self.lbl_wx_wind.setText(f"Wind: {wd:03d}°/{ws}kt{gust}")
        # QNH
        qnh_str = f"{m.qnh:.0f} hPa" if m.qnh else "---"
        inhg = f" / {m.qnh_inhg:.2f} inHg" if m.qnh_inhg else ""
        self.lbl_wx_qnh.setText(f"QNH: {qnh_str}{inhg}")
        # Temp
        temp = f"{m.temperature_c}°C" if m.temperature_c is not None else "---"
        dew = f" / DP {m.dewpoint_c}°C" if m.dewpoint_c is not None else ""
        self.lbl_wx_temp.setText(f"Temp: {temp}{dew}")
        # Sicht
        vis = f"{m.visibility_m}m" if m.visibility_m is not None else ("CAVOK" if m.cavok else "---")
        self.lbl_wx_vis.setText(f"Sicht: {vis}")
        # Empfohlene Runway
        rwy_shown = False
        if _AIRPORT_DB:
            ap = _AIRPORT_DB.get(icao)
            if ap and ap.runways:
                rec = recommend_runway(m, ap.runways)
                self._recommended_rwy = rec
                # Alle verfügbaren Runways sammeln
                rwy_idents = []
                for rwy in ap.runways:
                    le = getattr(rwy, "le_ident", None)
                    he = getattr(rwy, "he_ident", None)
                    if le:
                        rwy_idents.append(le)
                    if he:
                        rwy_idents.append(he)
                rwy_list = " / ".join(rwy_idents[:8]) if rwy_idents else ""
                if rec:
                    self.lbl_wx_rwy.setText(f"Empf. RWY: {rec}")
                    rwy_shown = True
                elif rwy_list:
                    self.lbl_wx_rwy.setText(f"RWY: {rwy_list}")
                    rwy_shown = True
                # Store for wind arrow
                self.radar._wind_dir = wd
                self.radar._wind_speed = ws
            elif ap:
                # Airport gefunden aber keine Runways in DB
                self.lbl_wx_rwy.setText(f"RWY: keine Daten für {icao}")
                rwy_shown = True
        if not rwy_shown:
            # Fallback: Heading aus Wind berechnen
            if wd > 0:
                rwy_num = round(wd / 10) % 37
                if rwy_num == 0:
                    rwy_num = 36
                self.lbl_wx_rwy.setText(f"Empf. RWY: ~{rwy_num:02d} (Wind {wd:03d}°)")
            else:
                self.lbl_wx_rwy.setText("Empf. RWY: ---")
        # ATIS
        try:
            atis = generate_atis(icao, m)
            self.lbl_wx_atis.setText(atis[:200])
        except Exception:
            self.lbl_wx_atis.setText("")

    # -- Rechtsklick-Kontextmenü --------------------------------------------
    def _show_context_menu(self, callsign: str, gx: int, gy: int):
        menu = QMenu(self)
        menu.setStyleSheet("""
            QMenu { background: #16213e; color: #c8d6e5; border: 1px solid #0f3460; }
            QMenu::item:selected { background: #1a5276; }
        """)
        pilot = self.radar.pilots.get(callsign)
        if not pilot:
            return

        # -- Assigned Squawk --
        act_squawk = menu.addAction(f"Assigned Squawk [{pilot.assigned_squawk:04d}]")
        act_squawk.triggered.connect(lambda: self._ctx_assign_squawk(callsign))

        # -- Assigned Altitude --
        act_alt = menu.addAction(f"Assigned Altitude [{pilot.assigned_alt or '---'}]")
        act_alt.triggered.connect(lambda: self._ctx_assign_altitude(callsign))

        # -- CLEARED --
        clr_text = "CLEARED ✓" if pilot.cleared else "CLEARED setzen"
        act_clr = menu.addAction(clr_text)
        act_clr.triggered.connect(lambda: self._send_cleared(callsign))

        menu.addSeparator()

        # -- Contact Me (Frequenzwechsel-Anweisung) --
        if self.radar.atc_frequency > 0:
            freq_str = f"{self.radar.atc_frequency:.3f}"
            act_contact = menu.addAction(f"📻 Contact me {freq_str}")
            act_contact.triggered.connect(lambda: self._ctx_contact_me(callsign))
        else:
            act_contact = menu.addAction("📻 Contact me (Freq einstellen!)")
            act_contact.setEnabled(False)

        # -- Textnachricht senden --
        act_msg = menu.addAction("✉ Nachricht senden …")
        act_msg.triggered.connect(lambda: self._ctx_send_message(callsign))

        # -- CPDLC Sub-Menü --
        cpdlc_menu = menu.addMenu("📡 CPDLC")
        if self.radar.atc_frequency > 0:
            freq_str = f"{self.radar.atc_frequency:.3f}"
            act_cpdlc_transfer = cpdlc_menu.addAction(f"CONTACT {freq_str}")
            act_cpdlc_transfer.triggered.connect(
                lambda: self._ctx_cpdlc_send(callsign, "TRANSFER",
                    f"CONTACT {self.radar.my_station or 'ATC'} {self.radar.atc_frequency:.3f}"))
        act_cpdlc_climb = cpdlc_menu.addAction("CLIMB TO FL…")
        act_cpdlc_climb.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "CLIMB", "CLIMB TO FL"))
        act_cpdlc_descend = cpdlc_menu.addAction("DESCEND TO FL…")
        act_cpdlc_descend.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "DESCEND", "DESCEND TO FL"))
        act_cpdlc_direct = cpdlc_menu.addAction("DIRECT TO …")
        act_cpdlc_direct.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "DIRECT", "PROCEED DIRECT TO"))
        act_cpdlc_squawk = cpdlc_menu.addAction("SQUAWK …")
        act_cpdlc_squawk.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "SQUAWK", "SQUAWK"))
        act_cpdlc_speed = cpdlc_menu.addAction("SPEED …")
        act_cpdlc_speed.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "SPEED", "MAINTAIN SPEED"))

        # ── CTR-spezifische CPDLC-Befehle ──
        cpdlc_menu.addSeparator()
        ctr_label = cpdlc_menu.addAction("── CENTER / CTR ──")
        ctr_label.setEnabled(False)

        # CLIMB UNRESTRICTED
        act_climb_unr = cpdlc_menu.addAction("CLIMB UNRESTRICTED TO FL…")
        act_climb_unr.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "CLIMB",
                                             "CLIMB UNRESTRICTED TO FL"))

        # DESCEND UNRESTRICTED
        act_desc_unr = cpdlc_menu.addAction("DESCEND UNRESTRICTED TO FL…")
        act_desc_unr.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "DESCEND",
                                             "DESCEND UNRESTRICTED TO FL"))

        # CONTACT NEXT SECTOR (auto-filled from online centers)
        act_next_sector = cpdlc_menu.addAction("CONTACT NEXT SECTOR …")
        act_next_sector.triggered.connect(
            lambda: self._ctx_cpdlc_next_sector(callsign))

        # CHECK STUCK MIC
        act_stuck_mic = cpdlc_menu.addAction("⚠ CHECK STUCK MIC")
        act_stuck_mic.triggered.connect(
            lambda: self._ctx_cpdlc_send(callsign, "FREE",
                                         "CHECK STUCK MICROPHONE ON FREQUENCY"))

        # MAINTAIN FL
        act_maintain = cpdlc_menu.addAction("MAINTAIN FL…")
        act_maintain.triggered.connect(
            lambda: self._ctx_cpdlc_template(callsign, "MAINTAIN",
                                             "MAINTAIN FL"))

        # CROSS [WAYPOINT] AT FL
        act_cross = cpdlc_menu.addAction("CROSS … AT FL…")
        act_cross.triggered.connect(
            lambda: self._ctx_cpdlc_cross(callsign))

        cpdlc_menu.addSeparator()
        act_cpdlc_free = cpdlc_menu.addAction("✏ Freitext …")
        act_cpdlc_free.triggered.connect(
            lambda: self._ctx_cpdlc_freetext(callsign))

        # -- UNICOM 122.800 --
        act_unicom = menu.addAction("📡 UNICOM 122.800 zuweisen")
        act_unicom.triggered.connect(lambda: self._ctx_assign_unicom(callsign))

        menu.addSeparator()

        # -- Handover-Sub-Menü (zeigt echte Online-Stationen) --
        ho_menu = menu.addMenu("Handover to …")
        if hasattr(self, '_online_atc_stations') and self._online_atc_stations:
            for atc_st in self._online_atc_stations:
                st_name = atc_st.get('station', '')
                st_role = atc_st.get('role', '')
                st_freq = atc_st.get('frequency', 0)
                st_full = atc_st.get('full_name', st_name)
                # Eigene Station überspringen
                if st_name == self.radar.my_station:
                    continue
                label = f"{st_full}  ({st_role}  {st_freq:.3f})"
                act = ho_menu.addAction(label)
                act.triggered.connect(
                    lambda _, s=st_name: self._ctx_handover_direct(callsign, s))
            if ho_menu.isEmpty():
                act_none = ho_menu.addAction("(keine anderen Controller online)")
                act_none.setEnabled(False)
        else:
            # Fallback: Rollen-Auswahl (manuell)
            for target_role in ["GND", "TWR", "APP", "CTR"]:
                act = ho_menu.addAction(target_role)
                act.triggered.connect(lambda _, r=target_role: self._ctx_handover(callsign, r))

        # -- Handover Accept (wenn eingehend) --
        if callsign in self._handover_incoming:
            menu.addSeparator()
            act_accept = menu.addAction("✓ Handover ANNEHMEN")
            act_accept.triggered.connect(lambda: self._ctx_accept_handover(callsign))
            act_reject = menu.addAction("✗ Handover ABLEHNEN")
            act_reject.triggered.connect(lambda: self._ctx_reject_handover(callsign))

        menu.addSeparator()

        # -- Voice: WAKE_UP Signal an Piloten senden --
        if self.radar.atc_frequency > 0:
            wake_freq = f"{self.radar.atc_frequency:.3f}"
            act_wake = menu.addAction(f"📢 WAKE UP ({wake_freq})")
            act_wake.triggered.connect(
                lambda: self._ctx_wake_up(callsign))

        # -- Voice: Kick/Move aus Voice-Kanal --
        voice_menu = menu.addMenu("🎙 Voice")
        act_v_mute = voice_menu.addAction("🔇 Mute")
        act_v_mute.triggered.connect(
            lambda: self._voice_admin("VOICE_MUTE", callsign))
        act_v_kick = voice_menu.addAction("❌ Kick")
        act_v_kick.triggered.connect(
            lambda: self._voice_admin("VOICE_KICK", callsign))
        if self.radar.atc_frequency > 0:
            my_f = f"{self.radar.atc_frequency:.3f}"
            act_v_move = voice_menu.addAction(
                f"📻 Move → {my_f}")
            act_v_move.triggered.connect(
                lambda: self._voice_admin("VOICE_MOVE", callsign, my_f))

        menu.addSeparator()

        # -- Release --
        act_release = menu.addAction("Release (freigeben)")
        act_release.triggered.connect(lambda: self._ctx_release(callsign))

        menu.exec(QCursor.pos())

    def _ctx_assign_squawk(self, callsign: str):
        dlg = QInputDialog(self)
        dlg.setWindowTitle("Assigned Squawk")
        dlg.setLabelText(f"Squawk-Code für {callsign}:")
        dlg.setTextValue("2000")
        dlg.setStyleSheet("""
            QInputDialog { background: #16213e; }
            QLabel { color: #c8d6e5; font-size: 13px; font-weight: bold; }
            QLineEdit { background: #0d1117; color: #ffffff; border: 1px solid #30363d;
                        padding: 6px; font-size: 14px; font-family: Consolas;
                        border-radius: 4px; }
            QPushButton { background: #21262d; color: #58a6ff; padding: 6px 16px;
                          border: 1px solid #30363d; border-radius: 4px; font-weight: bold; }
            QPushButton:hover { background: #30363d; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            text = dlg.textValue()
            try:
                sq = int(text)
                self._send_atc_update(callsign, assigned_squawk=sq)
            except ValueError:
                pass

    def _ctx_assign_altitude(self, callsign: str):
        dlg = QInputDialog(self)
        dlg.setWindowTitle("Assigned Altitude")
        dlg.setLabelText(f"Höhe für {callsign} (z.B. FL100, 4000):")
        dlg.setTextValue("")
        dlg.setStyleSheet("""
            QInputDialog { background: #16213e; }
            QLabel { color: #c8d6e5; font-size: 13px; font-weight: bold; }
            QLineEdit { background: #0d1117; color: #ffffff; border: 1px solid #30363d;
                        padding: 6px; font-size: 14px; font-family: Consolas;
                        border-radius: 4px; }
            QPushButton { background: #21262d; color: #58a6ff; padding: 6px 16px;
                          border: 1px solid #30363d; border-radius: 4px; font-weight: bold; }
            QPushButton:hover { background: #30363d; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            text = dlg.textValue()
            if text:
                self._send_atc_update(callsign, assigned_alt=text.strip().upper())

    def _ctx_contact_me(self, callsign: str):
        """Sendet Frequenzwechsel-Anweisung an Piloten."""
        freq = self.radar.atc_frequency
        station = self.radar.my_station or "ATC"
        self._send_atc_update(
            callsign,
            contact_freq=freq,
            message=f"Contact {station} on {freq:.3f}",
        )
        self.status.showMessage(f"📻 Contact-Anweisung → {callsign}: {freq:.3f} MHz")

    def _ctx_assign_unicom(self, callsign: str):
        """Weist Piloten UNICOM 122.800 zu (kein Controller zuständig)."""
        self._send_atc_update(
            callsign,
            contact_freq=122.800,
            message="No controller available. Monitor UNICOM 122.800",
        )
        self.status.showMessage(f"📡 UNICOM 122.800 zugewiesen → {callsign}")

    def _ctx_send_message(self, callsign: str):
        """Freitext-Nachricht an Piloten senden."""
        dlg = QInputDialog(self)
        dlg.setWindowTitle("Nachricht")
        dlg.setLabelText(f"Nachricht an {callsign}:")
        dlg.setTextValue("")
        dlg.setStyleSheet("""
            QInputDialog { background: #16213e; }
            QLabel { color: #c8d6e5; font-size: 13px; font-weight: bold; }
            QLineEdit { background: #0d1117; color: #ffffff; border: 1px solid #30363d;
                        padding: 6px; font-size: 13px; border-radius: 4px; }
            QPushButton { background: #21262d; color: #58a6ff; padding: 6px 16px;
                          border: 1px solid #30363d; border-radius: 4px; font-weight: bold; }
            QPushButton:hover { background: #30363d; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            text = dlg.textValue()
            if text:
                station = self.radar.my_station or "ATC"
                self._send_atc_update(callsign, message=text.strip())
                self.status.showMessage(f"✉ Nachricht → {callsign}: {text.strip()[:40]}")

    # -- CPDLC-Sende-Funktionen ---------------------------------------------
    def _ctx_cpdlc_send(self, callsign: str, cpdlc_type: str, text: str):
        """Sendet eine fertige CPDLC-Nachricht an den Piloten via Server."""
        self.ws.sendTextMessage(json.dumps({
            "type": "CPDLC_SEND",
            "callsign": callsign,
            "cpdlc_type": cpdlc_type,
            "text": text,
        }))

    def _ctx_cpdlc_template(self, callsign: str, cpdlc_type: str, prefix: str):
        """CPDLC-Template mit Eingabefeld für den Wert."""
        dlg = QInputDialog(self)
        dlg.setWindowTitle(f"CPDLC – {cpdlc_type}")
        dlg.setLabelText(f"{prefix} … (Wert eingeben):")
        dlg.setTextValue("")
        dlg.setStyleSheet("""
            QInputDialog { background: #0a1a2a; }
            QLabel { color: #00e064; font-size: 13px; font-weight: bold; font-family: Consolas; }
            QLineEdit { background: #050810; color: #00ff80; border: 1px solid #00e064;
                        padding: 6px; font-size: 14px; font-family: Consolas; border-radius: 4px; }
            QPushButton { background: #0f3460; color: #00e064; padding: 6px 16px;
                          border: 1px solid #00e064; border-radius: 4px; font-weight: bold;
                          font-family: Consolas; }
            QPushButton:hover { background: #1a5276; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            val = dlg.textValue().strip().upper()
            if val:
                full_text = f"{prefix} {val}"
                self._ctx_cpdlc_send(callsign, cpdlc_type, full_text)

    def _ctx_cpdlc_freetext(self, callsign: str):
        """Freitext-CPDLC-Nachricht."""
        dlg = QInputDialog(self)
        dlg.setWindowTitle("CPDLC – Freitext")
        dlg.setLabelText(f"CPDLC an {callsign}:")
        dlg.setTextValue("")
        dlg.setStyleSheet("""
            QInputDialog { background: #0a1a2a; }
            QLabel { color: #00e064; font-size: 13px; font-weight: bold; font-family: Consolas; }
            QLineEdit { background: #050810; color: #00ff80; border: 1px solid #00e064;
                        padding: 6px; font-size: 14px; font-family: Consolas; border-radius: 4px; }
            QPushButton { background: #0f3460; color: #00e064; padding: 6px 16px;
                          border: 1px solid #00e064; border-radius: 4px; font-weight: bold;
                          font-family: Consolas; }
            QPushButton:hover { background: #1a5276; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            text = dlg.textValue().strip()
            if text:
                self._ctx_cpdlc_send(callsign, "FREE", text)

    def _ctx_cpdlc_next_sector(self, callsign: str):
        """CONTACT NEXT SECTOR – Auto-Fill mit Online-Center-Frequenz."""
        pilot = self.radar.pilots.get(callsign)
        # Versuche, den nächsten Sektor zu bestimmen
        default_text = ""
        if pilot and hasattr(self, '_online_atc_stations') and self._online_atc_stations:
            # Suche nächsten Center (nicht eigene Station)
            for atc_st in self._online_atc_stations:
                st_name = atc_st.get('station', '')
                st_role = atc_st.get('role', '')
                st_freq = atc_st.get('frequency', 0)
                if st_name != self.radar.my_station and st_role == "CTR" and st_freq > 0:
                    default_text = f"{st_name} {st_freq:.3f}"
                    break
        dlg = QInputDialog(self)
        dlg.setWindowTitle("CPDLC – Contact Next Sector")
        dlg.setLabelText("CONTACT … (Station Freq):")
        dlg.setTextValue(default_text)
        dlg.setStyleSheet("""
            QInputDialog { background: #0a1a2a; }
            QLabel { color: #00e064; font-size: 13px; font-weight: bold; font-family: Consolas; }
            QLineEdit { background: #050810; color: #00ff80; border: 1px solid #00e064;
                        padding: 6px; font-size: 14px; font-family: Consolas; border-radius: 4px; }
            QPushButton { background: #0f3460; color: #00e064; padding: 6px 16px;
                          border: 1px solid #00e064; border-radius: 4px; font-weight: bold;
                          font-family: Consolas; }
            QPushButton:hover { background: #1a5276; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            val = dlg.textValue().strip().upper()
            if val:
                self._ctx_cpdlc_send(callsign, "TRANSFER", f"CONTACT {val}")

    def _ctx_cpdlc_cross(self, callsign: str):
        """CROSS [WAYPOINT] AT FL… – Zwei-Felder-Eingabe."""
        dlg = QInputDialog(self)
        dlg.setWindowTitle("CPDLC – CROSS AT FL")
        dlg.setLabelText("Waypoint und FL (z.B. AMIKI FL240):")
        dlg.setTextValue("")
        dlg.setStyleSheet("""
            QInputDialog { background: #0a1a2a; }
            QLabel { color: #00e064; font-size: 13px; font-weight: bold; font-family: Consolas; }
            QLineEdit { background: #050810; color: #00ff80; border: 1px solid #00e064;
                        padding: 6px; font-size: 14px; font-family: Consolas; border-radius: 4px; }
            QPushButton { background: #0f3460; color: #00e064; padding: 6px 16px;
                          border: 1px solid #00e064; border-radius: 4px; font-weight: bold;
                          font-family: Consolas; }
            QPushButton:hover { background: #1a5276; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            val = dlg.textValue().strip().upper()
            if val:
                # Versuche Waypoint und FL zu trennen
                parts = val.split()
                if len(parts) >= 2:
                    waypoint = parts[0]
                    fl = parts[-1]
                    self._ctx_cpdlc_send(callsign, "CROSS",
                                         f"CROSS {waypoint} AT {fl}")
                else:
                    self._ctx_cpdlc_send(callsign, "CROSS",
                                         f"CROSS {val}")

    def _cpdlc_log_append(self, callsign: str, status: str, text: str, msg_id: str = ""):
        """Fügt eine CPDLC-Zeile im Log-Tab hinzu."""
        import time as _t
        ts = _t.strftime("%H:%M:%S")
        colors = {"SENT": "#00e064", "WILCO": "#58a6ff", "UNABLE": "#ff6060",
                  "STANDBY": "#ffcc44", "READ": "#8b949e"}
        c = colors.get(status, "#c8d6e5")
        mid_str = f" [{msg_id}]" if msg_id else ""
        self.txt_cpdlc_log.append(
            f'<span style="color:#555">[{ts}]</span> '
            f'<span style="color:{c};font-weight:bold">{status}</span> '
            f'<span style="color:#4a9edd">{callsign}</span>{mid_str} '
            f'<span style="color:#c8d6e5">{text}</span>'
        )

    def _ctx_handover_direct(self, callsign: str, to_station: str):
        """Direkter Handover an eine bekannte Online-Station."""
        self.ws.sendTextMessage(json.dumps({
            "type": "HANDOVER_OFFER",
            "callsign": callsign,
            "to_station": to_station,
        }))
        self.status.showMessage(f"Handover-Angebot: {callsign} → {to_station}")

    def _ctx_handover(self, callsign: str, target_role: str):
        """Handover-Dialog: Ziel-Station eingeben oder aus aktiven ATC wählen."""
        dlg = QInputDialog(self)
        dlg.setWindowTitle("Handover")
        dlg.setLabelText(f"Ziel-Station für {callsign} ({target_role}):")
        dlg.setTextValue("")
        dlg.setStyleSheet("""
            QInputDialog { background: #16213e; }
            QLabel { color: #c8d6e5; font-size: 13px; font-weight: bold; }
            QLineEdit { background: #0d1117; color: #ffffff; border: 1px solid #30363d;
                        padding: 6px; font-size: 13px; border-radius: 4px; }
            QPushButton { background: #21262d; color: #58a6ff; padding: 6px 16px;
                          border: 1px solid #30363d; border-radius: 4px; font-weight: bold; }
            QPushButton:hover { background: #30363d; }
        """)
        if dlg.exec() == QInputDialog.DialogCode.Accepted:
            text = dlg.textValue()
            if text:
                self.ws.sendTextMessage(json.dumps({
                    "type": "HANDOVER_OFFER",
                    "callsign": callsign,
                    "to_station": text.strip().upper(),
                    "role": target_role,
                }))
                self.status.showMessage(f"Handover-Angebot: {callsign} → {text.strip().upper()}")

    def _ctx_accept_handover(self, callsign: str):
        self._handover_incoming.pop(callsign, None)
        pilot = self.radar.pilots.get(callsign)
        if pilot:
            pilot.handover_blink = False
        self.ws.sendTextMessage(json.dumps({
            "type": "HANDOVER_ACCEPT",
            "callsign": callsign,
        }))

    def _ctx_reject_handover(self, callsign: str):
        self._handover_incoming.pop(callsign, None)
        pilot = self.radar.pilots.get(callsign)
        if pilot:
            pilot.handover_blink = False
        self.ws.sendTextMessage(json.dumps({
            "type": "HANDOVER_REJECT",
            "callsign": callsign,
        }))

    def _ctx_release(self, callsign: str):
        # REST-Call zum Freigeben
        try:
            req = urllib.request.Request(
                f"http://localhost:9000/handover/{callsign}",
                method="DELETE",
                headers={"Content-Type": "application/json"},
            )
            urllib.request.urlopen(req, timeout=2)
            pilot = self.radar.pilots.get(callsign)
            if pilot:
                pilot.controller = ""
                pilot.controller_role = ""
            self.status.showMessage(f"Released: {callsign}")
        except Exception as e:
            self.status.showMessage(f"Release-Fehler: {e}")

    def _ctx_wake_up(self, callsign: str):
        """WAKE_UP Signal an Piloten senden (über ATC-WS oder server_atc WS)."""
        if self.radar.atc_frequency <= 0:
            return
        msg = {
            "type": "WAKE_UP",
            "callsign": callsign,
            "frequency": self.radar.atc_frequency,
            "message": f"Contact {self.radar.my_station} on "
                       f"{self.radar.atc_frequency:.3f}",
        }
        self.ws.sendTextMessage(json.dumps(msg))
        self.status.showMessage(
            f"📢 WAKE_UP → {callsign}: {self.radar.atc_frequency:.3f}")

    # -- ATC-Anweisungen senden ---------------------------------------------
    def _send_atc_update(self, callsign: str, **kwargs):
        msg = {"type": "ATC_UPDATE", "callsign": callsign}
        msg.update(kwargs)
        self.ws.sendTextMessage(json.dumps(msg))

        # Lokal sofort aktualisieren
        pilot = self.radar.pilots.get(callsign)
        if pilot:
            if "assigned_squawk" in kwargs:
                pilot.assigned_squawk = kwargs["assigned_squawk"]
            if "assigned_alt" in kwargs:
                pilot.assigned_alt = kwargs["assigned_alt"]
            if "cleared" in kwargs:
                pilot.cleared = kwargs["cleared"]

    def _send_cleared(self, callsign: str):
        pilot = self.radar.pilots.get(callsign)
        if pilot:
            new_state = not pilot.cleared
            self._send_atc_update(callsign, cleared=new_state)

    def _send_assigned_alt(self, callsign: str, alt: str):
        self._send_atc_update(callsign, assigned_alt=alt)

    # -- Frequenz-Chat -------------------------------------------------------
    def _send_freq_chat(self):
        """Sendet eine Nachricht auf der ATC-Frequenz an alle Piloten."""
        text = self.txt_freq_input.text().strip()
        if not text:
            return
        station = self.radar.my_station or "ATC"
        freq = self.radar.atc_frequency
        msg = {
            "type": "FREQ_MSG",
            "from": station,
            "frequency": round(freq, 3),
            "message": text,
        }
        self.ws.sendTextMessage(json.dumps(msg))
        self.txt_freq_input.clear()
        self._append_freq_chat(station, text, freq, own=True)

    def _append_freq_chat(self, sender: str, text: str, freq: float = 0, own: bool = False):
        """Fügt eine Chat-Zeile im Frequenz-Chat hinzu."""
        import time as _t
        ts = _t.strftime("%H:%M:%S")
        color = "#58a6ff" if own else "#ffaa00"
        freq_str = f" [{freq:.3f}]" if freq > 0 else ""
        self.txt_freq_chat.append(
            f'<span style="color:#8b949e">[{ts}]{freq_str}</span> '
            f'<span style="color:{color};font-weight:bold">{sender}:</span> '
            f'<span style="color:#c8d6e5">{text}</span>'
        )
        sb = self.txt_freq_chat.verticalScrollBar()
        sb.setValue(sb.maximum())

    def _on_strip_transfer(self, callsign: str, target_station: str):
        """Transfer-Button im FlightStripWidget — Handover an Ziel-Station."""
        self.ws.sendTextMessage(json.dumps({
            "type": "HANDOVER_OFFER",
            "callsign": callsign,
            "to_station": target_station,
            "role": "CTR",
        }))
        self.status.showMessage(f"✈ Transfer: {callsign} → {target_station}")

    # -- Rolle / Ansichtsmodus -----------------------------------------------
    def _apply_role(self, role: str):
        self.radar.role = role
        self.cmb_role.setCurrentText(role)
        # Tuple: (scale, geo, taxiways, sids, stars, artcc, runways, navaids, labels, map_tiles, regions, low_airways, high_airways, firs, global_navaids, global_airways, sectors, gates)
        profiles = {
            "DEL": (2000, True,  True,  False, False, False, True,  False, True,  True,  True,  False, False, False, False, False, False, True),
            "GND": (2500, True,  True,  False, False, False, True,  False, True,  True,  True,  False, False, False, False, False, False, True),
            "TWR": (600,  False, True,  False, False, True,  True,  False, True,  True,  False, False, False, False, False, False, False, True),
            "APP": (200,  False, False, True,  True,  True,  True,  True,  False, False, False, True,  False, True,  True,  True,  True,  False),
            "DEP": (200,  False, False, True,  False, True,  True,  True,  False, False, False, True,  False, True,  True,  True,  True,  False),
            "CTR": (60,   True,  False, False, False, True,  True,  True,  False, False, False, False, True,  True,  True,  True,  True,  False),
        }
        p = profiles.get(role, profiles["CTR"])
        self.radar.scale = p[0]
        self.radar.show_geo = p[1]
        self.radar.show_taxiways = p[2]
        self.radar.show_sids = p[3]
        self.radar.show_stars = p[4]
        self.radar.show_artcc = p[5]
        self.radar.show_runways = p[6]
        self.radar.show_navaids = p[7]
        self.radar.show_labels = p[8]
        self.radar.show_map_tiles = p[9]
        self.radar.show_regions = p[10]
        self.radar.show_low_airways = p[11]
        self.radar.show_high_airways = p[12]
        self.radar.show_firs = p[13]
        self.radar.show_global_navaids = p[14]
        self.radar.show_global_airways = p[15]
        self.radar.show_sectors = p[16]
        self.radar.show_gates = p[17]
        self.chk_map.setChecked(self.radar.show_map_tiles)

        if role in ("GND", "TWR"):
            target = None
            if self.radar.selected_callsign:
                target = self.radar.pilots.get(self.radar.selected_callsign)
            if target is None and self.radar.pilots:
                target = next(iter(self.radar.pilots.values()))
            if target:
                self.radar.center_lat = target.lat
                self.radar.center_lon = target.lon

        # Bei GND/TWR/DEL automatisch ICAO-Suche auslösen, damit
        # Sektordaten + OSM-Taxiways/Gates immer geladen werden.
        if role in ("GND", "TWR", "DEL"):
            icao = self.txt_icao.text().strip().upper()
            if icao and not self.radar.sector:
                self._on_search_icao()

        # Bei CTR/APP: Sektor-Polygone sofort laden + Wetter holen
        if role in ("CTR", "APP", "DEP"):
            # Viewport-Cache zurücksetzen → erzwingt neuen API-Fetch
            self.radar._sector_last_viewport = None
            self.radar._sector_fetch_pending = False
            self.radar.invalidate_bg_cache()
            # ICAO-Suche auslösen falls ICAO gesetzt (Wetter + Sektor)
            icao = self.txt_icao.text().strip().upper()
            if icao:
                self._on_search_icao()

        self.radar.invalidate_bg_cache()
        self.status.showMessage(f"Rolle: {role}  |  Zoom: {self.radar.scale:.0f}")

    def _on_map_toggled(self, checked: bool):
        self.radar.show_map_tiles = checked
        self.radar.invalidate_bg_cache()
        self.radar.update()


# ---------------------------------------------------------------------------
# Main
# ---------------------------------------------------------------------------
# ═══════════════════════════════════════════════════════════════════════════
#  LOGIN DIALOG
# ═══════════════════════════════════════════════════════════════════════════
class LoginDialog(QDialog):
    """VID-based login dialog für NEXUS-ATC Radar."""

    def __init__(self, server_url: str, parent=None):
        super().__init__(parent)
        self.server_http = server_url.replace("ws://", "http://").replace("wss://", "https://")
        # strip /ws/atc path
        if "/ws/" in self.server_http:
            self.server_http = self.server_http[:self.server_http.index("/ws/")]
        self.user_data = None
        self._settings = QSettings("NEXUS-ATC", "ATCRadar")
        self.setWindowTitle("NEXUS-ATC — Lotsen-Login")
        self.setFixedSize(380, 400)
        self.setStyleSheet("""
            QDialog { background: #0d1117; }
            QLabel  { color: #c9d1d9; font-size: 13px; }
            QLineEdit { background: #161b22; border: 1px solid #30363d; color: #e6edf3;
                        border-radius: 6px; padding: 8px 12px; font-size: 13px; }
            QLineEdit:focus { border-color: #00ccff; }
            QCheckBox { color: #8b949e; font-size: 12px; }
            QCheckBox::indicator { width: 14px; height: 14px; }
            QPushButton#loginBtn { background: #00ccff; color: #0d1117; border: none;
                                   border-radius: 6px; padding: 10px; font-size: 14px;
                                   font-weight: bold; }
            QPushButton#loginBtn:hover { background: #33d6ff; }
            QPushButton#loginBtn:disabled { background: #21262d; color: #484f58; }
        """)

        layout = QVBoxLayout(self)
        layout.setSpacing(12)
        layout.setContentsMargins(32, 28, 32, 28)

        title = QLabel("NEXUS-ATC")
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title.setStyleSheet("font-size:22px; font-weight:700; color:#00ccff; letter-spacing:2px;")
        layout.addWidget(title)

        sub = QLabel("ATC-Radar Login mit deiner VID")
        sub.setAlignment(Qt.AlignmentFlag.AlignCenter)
        sub.setStyleSheet("color:#8b949e; font-size:12px; margin-bottom:8px;")
        layout.addWidget(sub)

        layout.addWidget(QLabel("VID (7-stellig)"))
        self.vid_input = QLineEdit()
        self.vid_input.setPlaceholderText("z.B. 1000000")
        self.vid_input.setMaxLength(7)
        layout.addWidget(self.vid_input)

        layout.addWidget(QLabel("Passwort"))
        self.pw_input = QLineEdit()
        self.pw_input.setEchoMode(QLineEdit.EchoMode.Password)
        self.pw_input.setPlaceholderText("Dein Passwort")
        layout.addWidget(self.pw_input)

        self.save_creds_cb = QCheckBox("VID & Passwort speichern")
        layout.addWidget(self.save_creds_cb)

        self.error_label = QLabel("")
        self.error_label.setStyleSheet("color:#f85149; font-size:12px;")
        self.error_label.setWordWrap(True)
        layout.addWidget(self.error_label)

        self.login_btn = QPushButton("ANMELDEN")
        self.login_btn.setObjectName("loginBtn")
        self.login_btn.clicked.connect(self._do_login)
        layout.addWidget(self.login_btn)

        self.pw_input.returnPressed.connect(self._do_login)
        self.vid_input.returnPressed.connect(lambda: self.pw_input.setFocus())

        # Restore saved credentials
        saved_vid = self._settings.value("saved_vid", "")
        saved_pw = self._settings.value("saved_pw", "")
        if saved_vid:
            self.vid_input.setText(saved_vid)
        if saved_pw:
            self.pw_input.setText(saved_pw)
        if saved_vid and saved_pw:
            self.save_creds_cb.setChecked(True)

    def _do_login(self):
        vid = self.vid_input.text().strip()
        pw  = self.pw_input.text()
        self.error_label.setText("")
        if not vid or not pw:
            self.error_label.setText("VID und Passwort eingeben!")
            return
        self.login_btn.setEnabled(False)
        self.login_btn.setText("Verbinde…")
        try:
            import urllib.request, json
            url = self.server_http + "/login"
            payload = json.dumps({"vid": vid, "password": pw}).encode()
            req = urllib.request.Request(url, data=payload,
                                        headers={"Content-Type": "application/json"})
            with urllib.request.urlopen(req, timeout=10) as resp:
                data = json.loads(resp.read())
            if data.get("error"):
                self.error_label.setText(data["error"])
                self.login_btn.setEnabled(True)
                self.login_btn.setText("ANMELDEN")
                return
            if data.get("is_banned"):
                self.error_label.setText("Dein Account wurde gesperrt.")
                self.login_btn.setEnabled(True)
                self.login_btn.setText("ANMELDEN")
                return
            # Save or clear credentials
            if self.save_creds_cb.isChecked():
                self._settings.setValue("saved_vid", vid)
                self._settings.setValue("saved_pw", pw)
            else:
                self._settings.remove("saved_vid")
                self._settings.remove("saved_pw")
            self.user_data = data
            self.accept()
        except Exception as exc:
            self.error_label.setText(f"Fehler: {exc}")
            self.login_btn.setEnabled(True)
            self.login_btn.setText("ANMELDEN")


def main():
    parser = argparse.ArgumentParser(description="NEXUS-ATC Radar")
    parser.add_argument("--server", "-s", default=DEFAULT_SERVER)
    parser.add_argument("--sector", default=None,
                        help="Pfad zu einer .sct Sektordatei")
    args = parser.parse_args()

    app = QApplication(sys.argv)
    app.setStyle("Fusion")

    # ── Login Gate ──
    login = LoginDialog(args.server)
    if login.exec() != QDialog.DialogCode.Accepted:
        sys.exit(0)

    user = login.user_data
    window = RadarWindow(args.server, sector_file=args.sector)
    window.setWindowTitle(f"NEXUS-ATC — {user.get('name','')} (VID {user.get('vid','')})") 

    # Auto-Config: Name aus Server-Daten in Controller-Feld, gesperrt
    window.txt_ctrl_name.setText(user.get('name', ''))
    window.txt_ctrl_name.setReadOnly(True)
    window.txt_ctrl_name.setStyleSheet(
        "background: #0d1b2a; color: #00ccff; border: 1px solid #00ccff;"
        " border-radius: 6px; padding: 5px 8px; font-size: 13px;")

    window.show()
    sys.exit(app.exec())


if __name__ == "__main__":
    main()
